/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.applet.*;
import java.io.*;
import java.net.*;
import java.util.*;

import java.awt.*;
import java.awt.image.*;
import javax.swing.*;

import org.opensourcephysics.controls.*;
import org.opensourcephysics.display.*;

/**
 * This defines static methods for loading resources.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class ResourceLoader {

  protected static ArrayList searchPaths = new ArrayList(); // search paths
  protected static int maxPaths = 20; // max number of paths in history
  protected static Hashtable resources = new Hashtable(); // cached resources
  protected static boolean cacheEnabled = false;

  /**
   * Private constructor to prevent instantiation.
   */
  private ResourceLoader() {
  }

  /**
  * Gets a resource specified by name. If no resource is found using the name
  * alone, the searchPaths are searched.
  *
  * @param name the file or URL name
  * @return the Resource, or null if none found
  */
  public static Resource getResource(String name) {
    return getResource(name, Resource.class);
  }

  /**
  * Gets a resource specified by name and Class. If no resource is found using
  * the name alone, the searchPaths are searched.
  *
  * @param name the file or URL name
  * @param type the Class providing ClassLoader resource loading
  * @return the Resource, or null if none found
  */
  public static Resource getResource(String name, Class type) {
    // look for resource with name only
    Resource res = findResource(name, type);
    if (res != null) return res;
    StringBuffer err = new StringBuffer("Not found: " + name);
    err.append(" [searched " + name);
    if (OSPFrame.appletMode) { // applet mode
      JApplet applet = OSPFrame.applet;
      String path = getPath(applet.getCodeBase().toExternalForm(), name);
      res = findResource(path, type);
      if (res != null) return res;
      err.append(";" + path);
      path = getPath(applet.getDocumentBase().toExternalForm(), name);
      res = findResource(path, type);
      if (res != null) return res;
      err.append(";" + path);
    }
    // look for resource with name and searchPaths
    for (Iterator it = searchPaths.iterator(); it.hasNext(); ) {
      String path = getPath((String)it.next(), name);
      res = findResource(path, type);
      if (res != null) return res;
      err.append(";" + path);
    }
    err.append("]");
    OSPLog.info(err.toString());
    return null;
  }

  /**
  * Gets a resource specified by base path and name. No additional searchpaths
  * are searched.
  *
  * @param basePath the base path
  * @param name the file or URL name
  * @return the Resource, or null if none found
  */
  public static Resource getResource(String basePath, String name) {
    return getResource(basePath, name, Resource.class);
  }

  /**
  * Gets a resource specified by base path, name and class. No additional
  * searchpaths are searched.
  *
  * @param basePath the base path
  * @param name the file or URL name
  * @param type the Class providing ClassLoader resource loading
  * @return the Resource, or null if none found
  */
  public static Resource getResource(String basePath, String name, Class type) {
    String path = getPath(basePath, name);
    Resource res = findResource(path, type);
    if (res == null) {
      OSPLog.info("Not found: " + path);
    }
    return res;
  }


   /**
   * Adds a path at the beginning of the searchPaths list.
   *
   * @param base the base path to add
   */
  public static void addSearchPath(String base) {
    if (base == null || base.equals("") || maxPaths < 1) return;
    synchronized(searchPaths) {
      if (searchPaths.contains(base)) searchPaths.remove(base);
      else OSPLog.fine("Added path: " + base);
      searchPaths.add(0, base);
      while (searchPaths.size() > Math.max(maxPaths, 0)) {
        base = (String)searchPaths.get(searchPaths.size() - 1);
        OSPLog.fine("Removed path: " + base);
        searchPaths.remove(base);
      }
    }
  }

  /**
   * Removes a path from the searchPaths list.
   *
   * @param base the base path to remove
   */
  public static void removeSearchPath(String base) {
    if (base == null || base.equals("")) return;
    synchronized(searchPaths) {
      if (searchPaths.contains(base)) {
        OSPLog.fine("Removed path: " + base);
        searchPaths.remove(base);
      }
    }
  }

  /**
   * Sets the cacheEnabled property.
   *
   * @param enabled true to enable the cache
   */
  public static void setCacheEnabled(boolean enabled) {
    cacheEnabled = enabled;
  }

  /**
   * Gets the cacheEnabled property.
   *
   * @return true if the cache is enabled
   */
  public static boolean isCacheEnabled() {
    return cacheEnabled;
  }

//___________________________ convenience methods ______________________________

  public static InputStream openInputStream(String path) {
    Resource res = getResource(path);
    return res == null? null: res.openInputStream();
  }

  public static Reader openReader(String path) {
    Resource res = getResource(path);
    return res == null? null: res.openReader();
  }

  public static String getString(String path) {
    Resource res = getResource(path);
    return res == null? null: res.getString();
  }

  public static ImageIcon getIcon(String path) {
    Resource res = getResource(path);
    return res == null? null: res.getIcon();
  }

  public static Image getImage(String path) {
    Resource res = getResource(path);
    return res == null? null: res.getImage();
  }

  public static BufferedImage getBufferedImage(String path) {
    Resource res = getResource(path);
    return res == null? null: res.getBufferedImage();
  }


  public static AudioClip getAudioClip(String path) {
    Resource res = getResource(path);
    return res == null? null: res.getAudioClip();
  }

//______________________________ private methods _______________________________

  /**
   * Creates a Resource from a file.
   *
   * @param path the file path
   * @return the resource, if any
   */
  static private Resource createFileResource(String path) {
    File file = new File(path);
    if (file.exists() && file.canRead()) {
      OSPLog.finest("Created from file: " + path);
      return new Resource(file);
    }
    return null;
  }

  /**
   * Creates a Resource from a class resource.
   *
   * @param name the resource name
   * @param type the class providing the classloader
   * @return the resource, if any
   */
   static private Resource createResource(String name, Class type) {
    // ignore any name that has a protocol
    if (name.indexOf(":/") != -1) return null;
    try { // check relative to jarfile root
      URL url = type.getResource("/" + name);
      Resource res = createResource(url);
      OSPLog.finest("Created from resource: " + name);
      return res;
    } catch (Exception ex) {}
    try { // check relative to specified class
      URL url = type.getResource(name);
      Resource res = createResource(url);
      OSPLog.finest("Created from resource: " + name);
      return res;
    }
    catch (Exception ex) {
    }
    return null; // resource not found
  }

  /**
   * Creates a Resource from a URL.
   *
   * @param path the url path
   * @return the resource, if any
   */
   static private Resource createURLResource(String path) {
    try {
      URL url = new URL(path);
      Resource res = createResource(url);
      OSPLog.finest("Created from URL: " + path);
      return res;
    }
    catch (Exception ex) {
      return null;
    }
  }

  /**
   * Creates a Resource.
   *
   * @param url the URL
   * @return the resource, if any
   * @throws IOException
   */
   static private Resource createResource(URL url) throws IOException {
     // check that url is accessible
    InputStream stream = url.openStream();
    stream.close();
    return new Resource(url);
  }

  private static Resource findResource(String path, Class type) {
    Resource res = null;
    // look for cached resource
    if (cacheEnabled) {
      res = (Resource) resources.get(path);
      if (res != null) {
        OSPLog.finest("Found in cache: " + path);
        return res;
      }
    }
    // try to load resource in file/web/jar order
    if ((res = createFileResource(path)) != null ||
               (res = createURLResource(path)) != null ||
                      (res = createResource(path, type)) != null) {
      if (cacheEnabled)
        resources.put(path, res);
      return res;
    }
    return null;
  }

  /**
   * Gets a path from the basePath and file name.
   *
   * @param basePath the base path
   * @param name the file name
   * @return the path
   */
  static String getPath(String basePath, String name) {
    if (!basePath.equals("") && !basePath.endsWith("/")) basePath += "/";
    // corrects the path so that it works with Mac
    if (basePath.startsWith("file:/") && !basePath.startsWith("file:///"))
      basePath = "file:///" + basePath.substring(6);
    return basePath + name;
  }

}
