/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.applet.*;
import java.io.*;
import java.net.*;

import java.awt.*;
import javax.swing.*;

import org.opensourcephysics.controls.*;
import java.awt.image.BufferedImage;

/**
 * This represents a resource obtained from a URL or File.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class Resource {

  private URL url;
  private File file;
  private boolean isAnImage = true;
  private ImageIcon icon;
  private String string;
  private AudioClip clip;
  private BufferedImage image;

  /**
   * Constructs a resource from a url.
   *
   * @param url the URL
   */
  public Resource(URL url) {
    this.url = url;
  }

  /**
   * Constructs a resource from a file.
   *
   * @param file the file
   */
  public Resource(File file) {
    this.file = file;
  }

  /**
   * Gets the absolute path.
   *
   * @return the absolute path
   */
  public String getAbsolutePath() {
    if (getFile() != null) return getFile().getAbsolutePath();
    if (getURL() != null) return getURL().toExternalForm();
    return null;
  }

  /**
   * Gets the url associated with this resource.
   *
   * @return the URL
   */
  public URL getURL() {
    if (url == null && file != null) {
      try {
        url = new URL("file:/" + file.getAbsolutePath());
      }
      catch (MalformedURLException ex) {
      }
    }
    return url;
  }

  /**
   * Gets the file associated with this resource.
   *
   * @return the File
   */
  public File getFile() {
    return file;
  }

  public Object getObject(Class type) {
    if (ImageIcon.class.equals(type))
      return getIcon();
    if (String.class.equals(type))
      return getString();
    return null;
  }

  /**
   * Opens an InputStream.
   *
   * @return the stream
   */
  public InputStream openInputStream() {
    if (getURL() != null) try {
      return getURL().openStream();
    } catch (IOException ex) {}
    return null;
  }

  /**
   * Opens a BufferedReader.
   *
   * @return the reader
   */
  public BufferedReader openReader() {
    if (getFile() != null) {
      try {
        return new BufferedReader(new FileReader(file));
      }
      catch (FileNotFoundException ex) {}
    }
    if (getURL() != null) {
      return new BufferedReader(new InputStreamReader(openInputStream()));
    }
    return null;
  }

  /**
   * Gets an ImageIcon.
   *
   * @return the icon
   */
  public ImageIcon getIcon() {
    if (icon == null && isAnImage) {
      icon = new ImageIcon(getURL());
      if (icon.getIconWidth() < 1) {
        icon = null;
        isAnImage = false;
      }
    }
    return icon;
  }

  /**
   * Gets an Image.
   *
   * @return the image
   */
  public Image getImage() {
    ImageIcon icon = getIcon();
    if (icon != null) return icon.getImage();
    return null;
  }

  /**
   * Gets a buffered image.
   *
   * @return the image
   */
  public BufferedImage getBufferedImage() {
    if (image == null && isAnImage) {
      Image im = getImage();
      if (im == null) {
        isAnImage = false;
      }
      else {
        image = new BufferedImage(im.getWidth(null),
                                  im.getHeight(null),
                                  BufferedImage.TYPE_INT_RGB);
        Graphics2D g2 = image.createGraphics();
        g2.drawImage(im, 0, 0, null);
      }
    }
    return image;
  }

  /**
   * Gets a String.
   *
   * @return the string
   */
  public String getString() {
    if (string == null) {
      StringBuffer buffer = new StringBuffer();
      try {
        BufferedReader in = new BufferedReader(openReader());
        String line = in.readLine();
        while (line != null) {
          buffer.append(line + XML.NEW_LINE);
          line = in.readLine();
        }
        in.close();
      } catch (IOException ex) {}
      string = buffer.toString();
    }
    return string;
  }

  /**
   * Gets an AudioClip.
   *
   * @return the audio clip
   */
  public AudioClip getAudioClip() {
    if (clip == null && getURL() != null) {
      clip = Applet.newAudioClip(getURL());
    }
    return clip;
  }
}
