
/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.io.*;
import java.lang.reflect.*;
import java.net.*;
import java.util.*;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.*;

import org.opensourcephysics.controls.*;
import org.opensourcephysics.display.*;

/**
 * This provides a GUI for launching osp applications and xml files.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class Launcher {

  // static fields
  protected static String defaultFileName = "launcher_default";
  protected static String defaultToolsFileName = "launcher_tools.xml";
  protected static String resourcesPath = "/org/opensourcephysics/resources/tools/";
  protected static String classPath; // list of jar names for classes, resources
  protected static String xmlBasePath = ""; // base path of open xml file
  protected static String tabSetBasePath = ""; // base path of open LaunchSet
  protected static JFileChooser chooser;
  protected static FileFilter xmlFileFilter;
  protected static int wInit = 480, hInit = 400;
  protected static Icon launchIcon, fileIcon, launchedIcon, singletonIcon;
  public static boolean singleAppMode = false;
  public static boolean singleVMMode = false;
  public static boolean singletonMode = false;
  private static boolean newVMAllowed = false;

  // instance fields
  protected JDialog inspector;
  protected int divider = 160;
  public JFrame frame;
  protected JPanel contentPane;
  protected JTabbedPane tabbedPane;
  protected JMenuItem singleAppItem;
  protected LaunchNode selectedNode;
  protected LaunchNode previousNode;
  protected String tabSetPath;
  protected boolean isTabSetWritable;
  protected JMenu fileMenu;
  protected JMenuItem openItem;
  protected JMenuItem closeTabItem;
  protected JMenuItem closeAllItem;
  protected JMenuItem editItem;
  protected JMenuItem exitItem;
  protected JMenu inspectMenu;
  protected JMenuItem tabSetItem;
  protected Action inspectAction;
  protected LaunchClassChooser classChooser;
  protected JPopupMenu popup = new JPopupMenu();
  protected Set openPaths = new HashSet();
  protected Launcher spawner;
  protected boolean editorEnabled = false; // enables editing current set
  protected Set changedFiles = new HashSet();
  protected MouseListener tabListener;
  protected boolean newNodeSelected = false;
  protected boolean isDefaultTabSet = false;

  /**
   * No-arg constructor.
   */
  public Launcher() {
    createGUI();
    XML.setLoader(LaunchSet.class, new LaunchSet());
    // determine whether launching in new vm is possible
    // create a test launch thread
    Runnable launchRunner = new Runnable() {
      public void run() {
        try {
          Process proc = Runtime.getRuntime().exec("java");
          // if it made it this far, new VM is allowed (?)
          newVMAllowed = true;
          proc.destroy();
        }
        catch (Exception ex) {
        }
      }
    };
    new Thread(launchRunner).start();
    // center frame on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - frame.getBounds().width) / 2;
    int y = (dim.height - frame.getBounds().height) / 2;
    frame.setLocation(x, y);
  }

  /**
   * Constructs a Launcher and opens the specified xml file.
   *
   * @param fileName the name of the xml file
   */
  public Launcher(String fileName) {
    createGUI();
    XML.setLoader(LaunchSet.class, new LaunchSet());
    if (fileName == null) fileName = defaultFileName + ".xset";
    String path = open(fileName);
    if (path == null && fileName.startsWith(defaultFileName)) {
      fileName = defaultFileName + ".xml";
      open(fileName);
    }
    refreshGUI();
    // center frame on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - frame.getBounds().width) / 2;
    int y = (dim.height - frame.getBounds().height) / 2;
    frame.setLocation(x, y);
  }

  /**
   * Gets the launch tree in the selected tab. May return null.
   *
   * @return the launch tree
   */
  public LaunchPanel getSelectedTab() {
    return (LaunchPanel)tabbedPane.getSelectedComponent();
  }

  /**
   * Gets the selected launch node. May return null.
   *
   * @return the selected launch node
   */
  public LaunchNode getSelectedNode() {
    if (getSelectedTab() == null) selectedNode = null;
    else selectedNode = getSelectedTab().getSelectedNode();
    return selectedNode;
  }

  /**
   * Gets the root node of the selected launch tree. May return null.
   *
   * @return the root node
   */
  public LaunchNode getRootNode() {
    if (getSelectedTab() == null) return null;
    return getSelectedTab().getRootNode();
  }

  /**
   * Gets the launch tree at the specified tab index. May return null.
   *
   * @param i the tab index
   * @return the launch tree
   */
  public LaunchPanel getTab(int i) {
    if (i >= tabbedPane.getTabCount()) return null;
    return (LaunchPanel)tabbedPane.getComponentAt(i);
  }

  /**
   * Gets the content pane.
   *
   * @return the content pane
   */
  public Container getContentPane() {
    return contentPane;
  }

  /**
   * Opens an xml document specified by name and displays it in a new tab.
   * If "name" is really an xml string, this method reads the string directly.
   *
   * @param name the name
   * @return the name, or void if failed
   */
  public String open(String name) {
    if (name == null) return null;
    // add search paths  to control element in last-to-first order
    ResourceLoader.addSearchPath(resourcesPath);
    ResourceLoader.addSearchPath(tabSetBasePath);
    ResourceLoader.addSearchPath(xmlBasePath); // will be first path searched
    String path = name;
    String absolutePath = "";
    XMLControlElement control = new XMLControlElement();
    // look for xml string
    if (name.startsWith("<?xml")) {
      control.readXML(name);
      if (control.failedToRead()) return null;
      path = control.getString("filename"); // could be null
      if (path == null) path = LaunchRes.getString("Menu.File.Preview");
      control.canWrite = control.getBoolean("writable");
    }
    if (control.getObjectClass() == Object.class) { // did not read xml string
      // read the named file
      absolutePath = control.read(path); // absolute path
    }
    if (control.failedToRead()) {
      if (!name.startsWith(defaultFileName)) {
        OSPLog.info(LaunchRes.getString("Log.Message.InvalidXML") + " " + name);
        JOptionPane.showMessageDialog(
            null,
            LaunchRes.getString("Dialog.InvalidXML.Message")
            + " \"" + name + "\"",
            LaunchRes.getString("Dialog.InvalidXML.Title"),
            JOptionPane.WARNING_MESSAGE);
      }
      return null;
    }
    OSPLog.fine(name);
    Class type = control.getObjectClass();
    if (type != null && LaunchSet.class.isAssignableFrom(type)) {
      if (path.equals(tabSetPath)) {
        return null;
      }
      isDefaultTabSet = path.startsWith(defaultFileName) &&
                        absolutePath.indexOf("jar:") > -1;
      // close all open tabs
      while (tabbedPane.getTabCount() > 0) {
        tabbedPane.setSelectedIndex(0);
        removeSelectedTab();
      }
      xmlBasePath = XML.getDirectoryPath(path);
      tabSetBasePath = xmlBasePath;
      tabSetPath = path;
      isTabSetWritable = control.canWrite;
      // load the xml file data
      OSPLog.finest(LaunchRes.getString("Log.Message.Loading") + ": " + path);
      control.loadObject(new LaunchSet(this, path));
      changedFiles.clear();
      return name;
    }
    else if (type != null && LaunchNode.class.isAssignableFrom(type)) {
      // load the xml file data
      OSPLog.finest(LaunchRes.getString("Log.Message.Loading") + ": " + path);
      LaunchNode node = new LaunchNode(LaunchRes.getString("NewNode.Name"));
      // assign file name BEFORE loading node
      node.fileName = XML.getRelativePath(absolutePath);
      control.loadObject(node);
      node.isFileWritable = control.canWrite;
      xmlBasePath = XML.getDirectoryPath(path);
      String tabName = getDisplayName(path);
      // if node is already open, select the tab
      for (int i = 0; i < tabbedPane.getComponentCount(); i++) {
        if (tabbedPane.getTitleAt(i).equals(tabName)) {
          LaunchNode root =
              ((LaunchPanel)tabbedPane.getComponent(i)).getRootNode();
          if (root.matches(node)) {
            tabbedPane.setSelectedIndex(i);
            return null;
          }
        }
      }
      addTab(node);
      Enumeration e = node.breadthFirstEnumeration();
      while (e.hasMoreElements()) {
        LaunchNode next = (LaunchNode) e.nextElement();
        next.setLaunchClass(next.launchClassName);
      }
      return name; // successfully opened and added to tab
    }
    else {
      OSPLog.info(LaunchRes.getString("Log.Message.NotLauncherFile"));
      JOptionPane.showMessageDialog(
          null,
          LaunchRes.getString("Dialog.NotLauncherFile.Message")
          + " \"" + name + "\"",
          LaunchRes.getString("Dialog.NotLauncherFile.Title"),
          JOptionPane.WARNING_MESSAGE);
    }
    return null;
  }

//______________________________ protected methods _____________________________

  /**
   * Creates a LaunchPanel with the specified root node and adds it to a new tab.
   *
   * @param root the root node
   */
  protected void addTab(LaunchNode root) {
    final LaunchPanel tab = new LaunchPanel(root);
    tab.tree.setCellRenderer(new LaunchRenderer());
    tab.tree.addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
        newNodeSelected = true;
        refreshGUI();
        newNodeSelected = false;
      }
    });
    tab.tree.addMouseListener(new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        popup.removeAll();
        handleMousePressed(e, tab);
      }
    });
    tabbedPane.addTab(getDisplayName(root.fileName), tab);
    tabbedPane.setSelectedComponent(tab);
    tab.setSelectedNode(root);
    tab.dataPanel.addComponentListener(new ComponentAdapter() {
      public void componentResized(ComponentEvent e) {
        divider = tab.splitPane.getDividerLocation();
      }
    });
  }

  /**
   * Opens an xml file selected with a chooser.
   *
   * @return the name of the opened file
   */
  protected String open() {
    int result = getXMLChooser().showOpenDialog(null);
    if (result == JFileChooser.APPROVE_OPTION) {
      String fileName = getXMLChooser().getSelectedFile().getAbsolutePath();
      fileName = XML.getRelativePath(fileName);
      xmlBasePath = "";
      return open(fileName);
    }
    return null;
  }

  /**
   * Removes the selected tab.
   *
   * @return true if the tab was removed
   */
  protected boolean removeSelectedTab() {
    int i = tabbedPane.getSelectedIndex();
    if (i < 0) return false;
    tabbedPane.removeTabAt(i);
    previousNode = selectedNode;
    newNodeSelected = true;
    if (tabbedPane.getTabCount() == 0) {
      tabSetPath = null;
      editorEnabled = false;
    }
    refreshGUI();
    return true;
  }

  /**
   * Refreshes the GUI.
   */
  protected void refreshGUI() {
    // set frame title
    String name = tabSetPath;
    if (name == null) frame.setTitle(LaunchRes.getString("Frame.Title"));
    else {
      frame.setTitle(LaunchRes.getString("Frame.Title")
                     + ": " + getDisplayName(name));
    }
    // set tab properties
    LaunchPanel tab = getSelectedTab();
    if (tab != null) {
      getSelectedTab().splitPane.setDividerLocation(divider);
      name = getDisplayName(getRootNode().fileName);
      closeTabItem.setText(LaunchRes.getString("Menu.File.CloseTab")
                           + " \"" + name + "\"");
    }
    // rebuild file menu
    fileMenu.removeAll();
    fileMenu.add(openItem);
    if (tab != null) {
      fileMenu.addSeparator();
      fileMenu.add(closeTabItem);
      fileMenu.add(closeAllItem);
      if (editorEnabled) {
        fileMenu.addSeparator();
        fileMenu.add(editItem);
      }
    }
    fileMenu.addSeparator();
    fileMenu.add(exitItem);
    // rebuild inspect menu
    inspectMenu.removeAll();
    inspectMenu.setEnabled(tabbedPane.getTabCount() > 0);
    if (inspectMenu.isEnabled()) {
      if (tabSetPath != null) {
        tabSetItem.setText(LaunchRes.getString("Menu.Help.Inspect.TabSet")
                           + " \"" + getDisplayName(tabSetPath) + "\"");
        inspectMenu.add(tabSetItem);
        inspectMenu.addSeparator();
      }
      for (int i = 0; i < tabbedPane.getTabCount(); i++) {
        JMenuItem item = new JMenuItem(LaunchRes.getString("Menu.Help.Inspect.Tab")
                           + " \"" + tabbedPane.getTitleAt(i) + "\"");
        item.setActionCommand(String.valueOf(i));
        item.addActionListener(inspectAction);
        inspectMenu.add(item);
      }
    }
  }

  /**
   * Creates the GUI.
   */
  protected void createGUI() {
    // create the frame
    frame = new LauncherFrame();
    inspector = new JDialog(frame, false);
    inspector.setSize(new java.awt.Dimension(600, 300));
    contentPane = new JPanel(new BorderLayout());
    contentPane.setPreferredSize(new Dimension(wInit, hInit));
    frame.setContentPane(contentPane);
    frame.setDefaultCloseOperation(JFrame.HIDE_ON_CLOSE);
    String imageFile="/org/opensourcephysics/resources/tools/images/launch.gif";
    launchIcon = new ImageIcon(Launcher.class.getResource(imageFile));
    imageFile="/org/opensourcephysics/resources/tools/images/launched.gif";
    launchedIcon = new ImageIcon(Launcher.class.getResource(imageFile));
    imageFile="/org/opensourcephysics/resources/tools/images/singleton.gif";
    singletonIcon = new ImageIcon(Launcher.class.getResource(imageFile));
    inspectAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        int i = Integer.parseInt(e.getActionCommand());
        LaunchNode node = getTab(i).getRootNode();
        inspectXML(node);
        inspector.setTitle(LaunchRes.getString("Inspector.Title.Tab") +
                           " \"" + tabbedPane.getTitleAt(i) + "\"");
      }
    };
    // create tabbed pane
    tabbedPane = new JTabbedPane(JTabbedPane.BOTTOM);
    contentPane.add(tabbedPane, BorderLayout.CENTER);
    tabbedPane.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        previousNode = selectedNode;
        newNodeSelected = true;
        refreshGUI();
      }
    });
    tabListener = new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        if (contentPane.getTopLevelAncestor() != frame) return;
        if (e.isPopupTrigger() || e.getButton() == MouseEvent.BUTTON3 ||
           (e.isControlDown() && System.getProperty("os.name", "").indexOf("Mac") > -1)) {
          // make popup and add items
          JPopupMenu popup = new JPopupMenu();
          JMenuItem item = new JMenuItem(LaunchRes.getString("MenuItem.Close"));
          item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              removeSelectedTab();
            }
          });
          popup.add(item);
          popup.show(tabbedPane, e.getX(), e.getY() + 8);
        }
      }
    };
    tabbedPane.addMouseListener(tabListener);
    // create the menu bar
    JMenuBar menubar = new JMenuBar();
    fileMenu = new JMenu(LaunchRes.getString("Menu.File"));
    menubar.add(fileMenu);
    openItem = new JMenuItem(LaunchRes.getString("Menu.File.Open"));
    int mask = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    openItem.setAccelerator(KeyStroke.getKeyStroke('O', mask));
    openItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        open();
      }
    });
    closeTabItem = new JMenuItem(LaunchRes.getString("Menu.File.CloseTab"));
    closeTabItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        removeSelectedTab();
      }
    });
    closeAllItem = new JMenuItem(LaunchRes.getString("Menu.File.CloseAll"));
    closeAllItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        while (tabbedPane.getTabCount() > 0) {
          tabbedPane.setSelectedIndex(0);
          removeSelectedTab();
        }
      }
    });
    editItem = new JMenuItem(LaunchRes.getString("Menu.File.Edit"));
    editItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (tabSetPath.startsWith(LaunchRes.getString("Menu.File.Preview"))) {
          exit(); // edit previews by exiting to builder
          return;
        }
        LaunchSet set = new LaunchSet(Launcher.this, tabSetPath);
        XMLControl control = new XMLControlElement(set);
        OSPLog.finest(control.toXML());
        LaunchBuilder builder = new LaunchBuilder(control.toXML());
        builder.spawner = Launcher.this;
        builder.tabSetPath = tabSetPath;
        builder.isTabSetWritable = isTabSetWritable;
        builder.isDefaultTabSet = isDefaultTabSet;
        builder.frame.setVisible(true);
        builder.frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
      }
    });
    JMenu helpMenu = new JMenu(LaunchRes.getString("Menu.Help"));
    menubar.add(helpMenu);
    JMenuItem logItem = new JMenuItem(LaunchRes.getString("Menu.Help.MessageLog"));
    logItem.setAccelerator(KeyStroke.getKeyStroke('L', mask));
    logItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (!OSPFrame.appletMode) {
          OSPLog log = OSPLog.getOSPLog();
          if (log.getLocation().x == 0 && log.getLocation().y == 0) {
            Point p = frame.getLocation();
            log.setLocation(p.x + 28, p.y + 28);
          }
        }
        OSPLog.showLog();
      }
    });
    helpMenu.add(logItem);
    inspectMenu = new JMenu(LaunchRes.getString("Menu.Help.Inspect"));
    helpMenu.add(inspectMenu);
    tabSetItem = new JMenuItem(LaunchRes.getString("Menu.Help.Inspect.TabSet"));
    tabSetItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        LaunchSet set = new LaunchSet(Launcher.this, tabSetPath);
        XMLControl xml = new XMLControlElement(set);
        xml.setValue("path", tabSetPath);
        XMLTreePanel treePanel = new XMLTreePanel(xml, false);
        inspector.setContentPane(treePanel);
        inspector.setTitle(LaunchRes.getString("Inspector.Title.TabSet") +
                           " \"" + getDisplayName(tabSetPath) + "\"");
        inspector.setVisible(true);
      }
    });
    helpMenu.addSeparator();
    JMenuItem aboutItem = new JMenuItem(LaunchRes.getString("Menu.Help.About"));
    aboutItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        showAboutDialog();
      }
    });
    helpMenu.add(aboutItem);
    if(!org.opensourcephysics.display.OSPFrame.appletMode){
      // add window listener to exit
      frame.addWindowListener(new WindowAdapter() {
        public void windowClosing(WindowEvent e) {
          exit();
        }
      });
      // add exit menu item
      fileMenu.addSeparator();
      exitItem = new JMenuItem(LaunchRes.getString("Menu.File.Exit"));
      exitItem.setAccelerator(KeyStroke.getKeyStroke('Q', mask));
      exitItem.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          exit();
        }
      });
    }
    frame.setJMenuBar(menubar);
    frame.pack();
  }

  /**
   * Adds launchable tools to the specified menu.
   *
   * @param menu the menu
   */
  protected void loadToolsMenu(JMenu menu) {
    String locator = resourcesPath + defaultToolsFileName;
    XMLControl control = new XMLControlElement(defaultToolsFileName);
    if (control.failedToRead()) {
      control = new XMLControlElement(locator);
    }
    if (!control.failedToRead()) {
      Class type = control.getObjectClass();
      if (type != null && LaunchNode.class.isAssignableFrom(type)) {
        // load the xml file data
        LaunchNode node = (LaunchNode) control.loadObject(null);
        node.addMenuItemsTo(menu);
      }
    }
  }

  /**
   * Gets the paths of currently open set and tabs.
   *
   * @return the open paths
   */
  protected Set getOpenPaths() {
    openPaths.clear();
    openPaths.add(tabSetPath);
    for (int i = 0; i < tabbedPane.getTabCount(); i++) {
      LaunchPanel panel = (LaunchPanel) tabbedPane.getComponentAt(i);
      LaunchNode[] nodes = panel.getRootNode().getOwnedNodes();
      for (int j = 0; j < nodes.length; j++) {
        openPaths.add(nodes[i].fileName);
      }
      openPaths.add(panel.getRootNode().fileName);
    }
    return openPaths;
  }

  /**
   * Shows the about dialog.
   */
  protected void showAboutDialog() {
    String aboutString = "Launcher 1.0  Aug 2004\n" +
                         "Open Source Physics Project\n" +
                         "www.opensourcephysics.org";
    JOptionPane.showMessageDialog(
        frame,
        aboutString,
        LaunchRes.getString("Help.About.Title") + " Launcher",
        JOptionPane.INFORMATION_MESSAGE);
  }

  /**
   * Determines whether the specified node is launchable.
   *
   * @param node the launch node to verify
   * @return <code>true</code> if the node is launchable
   */
  protected boolean isLaunchable(LaunchNode node) {
    if (node == null || !node.isLeaf()) return false;
    return isLaunchable(node.getLaunchClass());
  }

  /**
   * Determines whether the specified class is launchable.
   *
   * @param type the launch class to verify
   * @return <code>true</code> if the class is launchable
   */
  protected static boolean isLaunchable(Class type) {
    if (type == null) return false;
    try {
      Method m = type.getMethod("main", new Class[] {String[].class});
      return true;
    }
    catch (NoSuchMethodException ex) {
      return false;
    }
  }

  /**
   * Handles a mouse pressed event.
   *
   * @param e the mouse event
   * @param tab the launch panel receiving the event
   */
  protected void handleMousePressed(MouseEvent e, final LaunchPanel tab) {
    if (e.isPopupTrigger() || e.getButton() == MouseEvent.BUTTON3 ||
       (e.isControlDown() && System.getProperty("os.name", "").indexOf("Mac") > -1)) {
      // make sure node is selected
      TreePath path = tab.tree.getPathForLocation(e.getX(), e.getY());
      if (path == null) return;
      tab.tree.setSelectionPath(path);
      final LaunchNode node = getSelectedNode();
      if (node == null) return;
      // add items to popup menu
      JMenuItem inspectItem = new JMenuItem(
          LaunchRes.getString("MenuItem.Inspect"));
      inspectItem.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e) {
          inspectXML(node);
        }
      });
      if (node.launchCount == 0) {
        popup.add(inspectItem);
        popup.addSeparator();
        JMenuItem launchItem = new JMenuItem(LaunchRes.getString(
            "MenuItem.Launch"));
        popup.add(launchItem);
        launchItem.addActionListener(new ActionListener() {
          public void actionPerformed(ActionEvent e) {
            node.launch(tab);
          }
        });
      }
      else {
        popup.add(inspectItem);
        popup.addSeparator();
        JMenuItem terminateItem = new JMenuItem(LaunchRes.getString(
            "MenuItem.Terminate"));
        popup.add(terminateItem);
        terminateItem.addActionListener(new ActionListener() {
          public void actionPerformed(ActionEvent e) {
            for (Iterator it = node.processes.iterator(); it.hasNext(); ) {
              Process proc = (Process) it.next();
              proc.destroy();
            }
            for (Iterator it = node.frames.iterator(); it.hasNext(); ) {
              Frame frame = (Frame) it.next();
              WindowListener[] listeners = frame.getWindowListeners();
              for (int j = 0; j < listeners.length; j++) {
                frame.removeWindowListener(listeners[j]);
              }
              frame.dispose();
            }
            node.processes.clear();
            node.frames.clear();
            node.launchCount = 0;
          }
        });
        if (node.launchCount > 1) {
          terminateItem.setText(LaunchRes.getString("MenuItem.TerminateAll"));
        }
        if (!node.isSingleton()) {
          JMenuItem launchItem = new JMenuItem(LaunchRes.getString(
              "MenuItem.Relaunch"));
          popup.add(launchItem);
          launchItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              node.launch(tab);
            }
          });
        }
      }
      // show popup when running Launcher (LaunchBuilder)
      if (getClass().equals(Launcher.class))
        popup.show(tab, e.getX() + 4, e.getY() + 12);
    }
    else if (e.getClickCount() == 2 && isLaunchable(getSelectedNode())) {
      LaunchNode node = getSelectedNode();
      if (node.launchCount == 0) node.launch(tab);
      else if (node.isSingleton() ||
              (node.isSingleVM() && node.isSingleApp())) {
        JOptionPane.showMessageDialog(
            frame,
            LaunchRes.getString("Dialog.Singleton.Message")
            + " \"" + node.toString() + "\"",
            LaunchRes.getString("Dialog.Singleton.Title"),
            JOptionPane.INFORMATION_MESSAGE);
      }
      else {
        int selected = JOptionPane.showConfirmDialog(
            frame,
            LaunchRes.getString("Dialog.Relaunch.Message"),
            LaunchRes.getString("Dialog.Relaunch.Title"),
            JOptionPane.YES_NO_OPTION);
        if (selected == JOptionPane.YES_OPTION) {
          node.launch(tab);
        }
      }
      if (node.launchPanel != null) node.launchPanel.repaint();
    }
  }

  /**
   * Exits this application.
   */
  protected void exit() {
    if (!terminateApps()) return;
    if (frame.getDefaultCloseOperation() == JFrame.HIDE_ON_CLOSE) {
      System.exit(0);
    }
    else frame.dispose();
  }

  /**
   * Terminates running apps.
   *
   * @return false if process is cancelled by the user
   */
  protected boolean terminateApps() {
    if (frame.getDefaultCloseOperation() == JFrame.HIDE_ON_CLOSE) {
      // ask to terminate apps running in this process
      boolean approved = false;
      Frame[] frames = JFrame.getFrames();
      for (int i = 0, n = frames.length; i < n; i++) {
        if (!approved && frames[i].isVisible() &&
            ! (frames[i] instanceof LauncherFrame) &&
            ! (frames[i] instanceof OSPLog)) {
          int selected = JOptionPane.showConfirmDialog(
              frame,
              LaunchRes.getString("Dialog.Terminate.Message") +
              XML.NEW_LINE +
              LaunchRes.getString("Dialog.Terminate.Question"),
              LaunchRes.getString("Dialog.Terminate.Title"),
              JOptionPane.YES_NO_OPTION);
          if (selected == JOptionPane.YES_OPTION) approved = true;
          else return false;
        }
      }
      // ask to terminate apps running in separate processes
      approved = false;
      boolean declined = false;
      // look for nodes with running processes
      Component[] comps = tabbedPane.getComponents();
      for (int i = 0; i < comps.length; i++) {
        LaunchPanel tab = (LaunchPanel) comps[i];
        Enumeration e = tab.getRootNode().breadthFirstEnumeration();
        while (e.hasMoreElements()) {
          LaunchNode node = (LaunchNode) e.nextElement();
          if (!node.processes.isEmpty()) {
            if (!approved && !declined) { // ask for approval
              int selected = JOptionPane.showConfirmDialog(
                  frame,
                  LaunchRes.getString("Dialog.TerminateSeparateVM.Message") +
                  XML.NEW_LINE +
                  LaunchRes.getString("Dialog.TerminateSeparateVM.Question"),
                  LaunchRes.getString("Dialog.TerminateSeparateVM.Title"),
                  JOptionPane.YES_NO_OPTION);
              approved = (selected == JOptionPane.YES_OPTION);
              declined = !approved;
            }
            if (approved) { // terminate processes
              for (Iterator it = node.processes.iterator(); it.hasNext(); ) {
                Process proc = (Process) it.next();
                it.remove();
                proc.destroy();
              }
            }
            else return false;
          }
        }
      }
    }
    return true;
  }

  /**
   * A unique frame class for Launcher. This is used to differentiate Launcher
   * from other apps when disposing of frames in singleApp mode.
   */
  private class LauncherFrame extends OSPFrame {
    public LauncherFrame() {setName("LauncherTool");}
  }

  /**
   * A cell renderer to show launchable nodes.
   */
  private class LaunchRenderer extends DefaultTreeCellRenderer {
    public Component getTreeCellRendererComponent(
        JTree tree, Object value, boolean sel, boolean expanded,
        boolean leaf, int row, boolean hasFocus) {
      super.getTreeCellRendererComponent(
          tree, value, sel, expanded, leaf, row, hasFocus);
      LaunchNode node = (LaunchNode) value;
      if (node.fileName != null && Launcher.this instanceof LaunchBuilder) {
        setIcon(fileIcon);
      }
      else if (node.launchCount > 0) {
        if (node.isSingleton()) setIcon(singletonIcon);
        else if (node.isSingleVM() && node.isSingleApp()) setIcon(singletonIcon);
        else setIcon(launchedIcon);
      }
      else if (isLaunchable(node)) setIcon(launchIcon);
      return this;
    }
  }

  /**
   * A class to save and load a set of launch tabs and Launcher static fields.
   */
  protected class LaunchSet implements XML.ObjectLoader {

    private Launcher launcher;
    private String base;
    private String name;

    private LaunchSet() {}

    protected LaunchSet(Launcher launcher, String fileName) {
      this.launcher = launcher;
      name = fileName;
      base = XML.getDirectoryPath(fileName);
    }

    public void saveObject(XMLControl control, Object obj) {
      Launcher launcher = ((LaunchSet)obj).launcher;
      String base = ((LaunchSet)obj).base;
      // save static properties
      control.setValue("classpath", classPath);
      if (launcher.editorEnabled)
        control.setValue("editor_enabled", true);
      // save dimensions and split pane divider location
      Dimension dim = launcher.contentPane.getSize();
      control.setValue("width", dim.width);
      control.setValue("height", dim.height);
      control.setValue("divider", divider);
      // save tree root file names relative to base
      Collection nodes = new ArrayList();
      for (int i = 0; i < launcher.tabbedPane.getComponentCount(); i++) {
        LaunchNode root =
            ((LaunchPanel)launcher.tabbedPane.getComponent(i)).getRootNode();
        String name = XML.getPathRelativeTo(root.fileName, base);
        nodes.add(name);
      }
      control.setValue("launch_nodes", nodes);
    }

    public Object createObject(XMLControl control){
      return launcher;
    }

    public Object loadObject(XMLControl control, Object obj) {
      Launcher launcher = ((LaunchSet)obj).launcher;
      // load a different launch set
      if (control.getPropertyNames().contains("launchset")) {
        launcher.open(control.getString("launchset"));
        return obj;
      }
      // load static properties
      if (control.getPropertyNames().contains("classpath")) {
        classPath = control.getString("classpath");
      }
      // load launch nodes
      Collection nodes = (Collection)control.getObject("launch_nodes");
      if (nodes !=  null && !nodes.isEmpty()) {
        int i = launcher.tabbedPane.getSelectedIndex();
        Iterator it = nodes.iterator();
        boolean tabAdded = false;
        while (it.hasNext()) {
          String next = (String)it.next();
          // prevent circular references
          if (name != null && name.equals(next)) continue; // prevent circular references
          tabAdded = (launcher.open(next) != null) || tabAdded;
        }
        // select the first added tab
        if (tabAdded) launcher.tabbedPane.setSelectedIndex(i + 1);
      }
      launcher.editorEnabled = control.getBoolean("editor_enabled");
      // load dimensions
      if (control.getPropertyNames().contains("width") &&
          control.getPropertyNames().contains("height")) {
        int w = control.getInt("width");
        int h = control.getInt("height");
        launcher.contentPane.setPreferredSize(new Dimension(w, h));
        launcher.frame.pack();
      }
      // load divider position
      if (control.getPropertyNames().contains("divider")) {
        divider = control.getInt("divider");
        refreshGUI();
      }
      return obj;
    }
  }

//________________________________ static methods _____________________________

  /**
   * Launches an application with no arguments.
   *
   * @param type the class to be launched
   */
  public static void launch(Class type) {
    launch(type, null, null);
  }

  /**
   * Launches an application with an array of string arguments.
   *
   * @param type the class to be launched
   * @param args the String array of arguments
   */
  public static void launch(Class type, String[] args) {
    launch(type, args, null);
  }

  /**
   * Launches an application asociated with a launch node.
   *
   * @param type the class to be launched
   * @param args the argument array (may be null)
   * @param node the launch node (may be null)
   */
  public static void launch(final Class type, String[] args,
                            final LaunchNode node) {
    if (type == null) {
      OSPLog.info(LaunchRes.getString("Log.Message.NoClass"));
      JOptionPane.showMessageDialog(
          null,
          LaunchRes.getString("Dialog.NoLaunchClass.Message"),
          LaunchRes.getString("Dialog.NoLaunchClass.Title"),
          JOptionPane.WARNING_MESSAGE);
      return;
    }
    String desc = LaunchRes.getString("Log.Message.Launching")+" "+type+", args ";
    if (args == null) desc += args;
    else {
      desc += "{";
      for (int i = 0; i < args.length; i++) {
        desc += args[i];
        if (i < args.length-1) desc += ", ";
      }
      desc += "}";
    }
    OSPLog.fine(desc);
    // launch the app as an applet
    if (org.opensourcephysics.display.OSPFrame.appletMode){
      launchApplet(type);
      return;
    }
    // launch the app in single vm mode
    if (singleVMMode || !newVMAllowed) {
      OSPLog.finer(LaunchRes.getString("Log.Message.LaunchCurrentVM"));
      // get previous frames before launching
      Frame[] prevFrames = JFrame.getFrames();
      // dispose of previous frames if single app mode
      if (singleAppMode) {
        OSPLog.finer(LaunchRes.getString("Log.Message.LaunchSingleApp"));
        boolean vis = OSPLog.isLogVisible();
        for (int i = 0, n = prevFrames.length; i < n; i++) {
          if (!(prevFrames[i] instanceof LauncherFrame)) {
            WindowListener[] listeners = prevFrames[i].getWindowListeners();
            for (int j = 0; j < listeners.length; j++) {
              listeners[j].windowClosing(null);
            }
            prevFrames[i].dispose();
          }
        }
        if (vis) OSPLog.showLog();
      }
      // launch the app by invoking main method
      try {
        Method m = type.getMethod("main", new Class[] {String[].class});
        m.invoke(type, new Object[] {args});
      }
      catch (NoSuchMethodException ex) {}
      catch (InvocationTargetException ex) {}
      catch (IllegalAccessException ex) {}
      // get frames after launching
      Frame[] frames = JFrame.getFrames();
      // keep only frames not in prevFrames
      final Collection newFrames = new ArrayList();
      for (int i = 0; i < frames.length; i++) {
        newFrames.add(frames[i]);
      }
      for (int i = 0; i < prevFrames.length; i++) {
        newFrames.remove(prevFrames[i]);
      }
      frames = (Frame[])newFrames.toArray(new Frame[0]);
      newFrames.clear();
      boolean foundControl = false;
      for (int i = 0, n = frames.length; i < n; i++) {
        if (frames[i] instanceof JFrame &&
          !(frames[i] instanceof LauncherFrame)) {
          JFrame frame = (JFrame) frames[i];
          if (frame.getDefaultCloseOperation() == JFrame.EXIT_ON_CLOSE) {
            // change default close operation from EXIT to DISPOSE
            // (otherwise exiting a launched app also exits Launcher)
            frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
            foundControl = true;
            // add window listener so control frames can close associated windows
            frame.addWindowListener(new WindowAdapter() {
              public void windowClosing(WindowEvent e) {
                Iterator it = newFrames.iterator();
                while (it.hasNext()) {
                  Frame frame = (Frame)it.next();
                  frame.removeWindowListener(this);
                  frame.dispose();
                }
                if (node != null) {
                  node.frames.removeAll(newFrames);
                  node.launchCount--;
                }
                if (node.launchPanel != null) node.launchPanel.repaint();
              }
            });
          }
          newFrames.add(frame);
        }
      }
      if (node != null && foundControl) {
        node.frames.addAll(newFrames);
        node.launchCount++;
      }
      return;
    }
    // launch the app in a separate java process
    OSPLog.finer(LaunchRes.getString("Log.Message.LaunchSeparateVM"));
    // construct the command to execute
    final Vector cmd = new Vector();
    cmd.add("java");
    if (classPath != null && !classPath.equals("")) {
      String jar = getDefaultJar();
      if (jar != null) {
        classPath += ";" + jar;
      }
      cmd.add("-classpath");
      cmd.add(classPath);
    }
    cmd.add(type.getName());
    if (args != null) {
      for (int i = 0; i < args.length && args[i] != null; i++)
        cmd.add(args[i]);
    }
    // create a launch thread
    Runnable launchRunner = new Runnable() {
      public void run() {
        OSPLog.finer(LaunchRes.getString("Log.Message.Command") + " " + cmd.toString());
        String[] cmdarray = (String[]) cmd.toArray(new String[0]);
        try {
          Process proc = Runtime.getRuntime().exec(cmdarray);
          if (node != null) node.processes.add(proc);
          proc.waitFor();
          if (node != null) {
            node.threadRunning(false);
            node.processes.remove(proc);
          }
        }
        catch (Exception ex) {
          OSPLog.info(ex.toString());
          if (node != null) {
            node.threadRunning(false);
          }
        }
      }
    };
    if (node != null) node.threadRunning(true);
    new Thread(launchRunner).start();
  }

  /**
   * Launches an application in applet mode. Added by W. Christian.
   *
   * @param type the class to be launched
   */
  public static void launchApplet(Class type) {
    OSPLog.fine(LaunchRes.getString("Log.Message.LaunchApplet"));
    Method m = null;
    try {
      m = type.getMethod("main", new Class[] {String[].class});
    }
    catch (SecurityException ex) {}
    catch (NoSuchMethodException ex) {}
    try {
      m.invoke(type, new Object[] {new String[] {""}});
    }
    catch (InvocationTargetException ex) {}
    catch (IllegalArgumentException ex) {}
    catch (IllegalAccessException ex) {}
    Frame[] frames = JFrame.getFrames();
    for (int i = 0, n = frames.length; i < n; i++) {
      if (! (frames[i] instanceof JFrame))
        continue;
      JFrame frame = (JFrame) frames[i];
      if (frame.getDefaultCloseOperation() == JFrame.EXIT_ON_CLOSE) {
        frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      }
      if (frame instanceof OSPFrame) {
        ( (OSPFrame) frame).setKeepHidden(false);
      }
      if (frame instanceof ControlFrame) {
        ( (ControlFrame) frame).setVisible(true);
      }
    }
  }

  /**
   * Main entry point when used as application.
   *
   * @param args args[0] may be an xml file name
   */
  public static void main(String[] args) {
//    OSPLog.setLevel(ConsoleLevel.ALL);
    Launcher launcher = new Launcher();
    if (args != null && args.length > 0) {
      for (int i = 0; i < args.length; i++) {
        launcher.open(args[i]);
      }
    }
    else {
      String path = launcher.open(Launcher.defaultFileName + ".xset");
      if (path == null) launcher.open(Launcher.defaultFileName + ".xml");
    }
    launcher.refreshGUI();
    // center frame on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - launcher.frame.getBounds().width) / 2;
    int y = (dim.height - launcher.frame.getBounds().height) / 2;
    launcher.frame.setLocation(x, y);
    launcher.frame.setVisible(true);
  }

  /**
   * Gets a class chooser for selecting launchable classes from jar files.
   *
   * @return the jar class chooser
   */
  protected LaunchClassChooser getClassChooser() {
    if (classChooser == null) classChooser = new LaunchClassChooser(contentPane);
    return classChooser;
  }

  /**
   * Gets a file chooser for selecting xml files.
   *
   * @return the xml chooser
   */
  protected static JFileChooser getXMLChooser() {
    if (chooser != null) return chooser;
    chooser = new JFileChooser(new File("." + File.separator));
    // add xml file filter
    xmlFileFilter = new FileFilter() {
      // accept all directories and *.xml or *.xset files.
      public boolean accept(File f) {
        if (f == null)
          return false;
        if (f.isDirectory())
          return true;
        String extension = null;
        String name = f.getName();
        int i = name.lastIndexOf('.');
        if (i > 0 && i < name.length() - 1) {
          extension = name.substring(i + 1).toLowerCase();
        }
        if (extension != null &&
            (extension.equals("xml") ||
             extension.equals("xset") ||
             extension.equals("set")))
          return true;
        return false;
      }
      // the description of this filter
      public String getDescription() {
        return LaunchRes.getString("FileChooser.XMLFilter.Description");
      }
    };
    chooser.addChoosableFileFilter(xmlFileFilter);
    return chooser;
  }

  /**
   * Gets the extension of the specified file name.
   *
   * @param fileName the file name
   * @return the extension, or null if none
   */
  protected static String getExtension(String fileName) {
    if (fileName == null) return null;
    int i = fileName.lastIndexOf('.');
    if (i > 0 && i < fileName.length() - 1) {
      return fileName.substring(i + 1);
    }
    return null;
  }

  /**
   * Gets the display name of the specified file name.
   *
   * @param fileName the file name
   * @return the bare name without path or extension
   */
  protected static String getDisplayName(String fileName) {
    if (fileName == null) return "";
    // remove path
    int i = fileName.lastIndexOf("/");
    if (i == -1) i = fileName.lastIndexOf("\\");
    if (i != -1) fileName = fileName.substring(i + 1);
    i = fileName.lastIndexOf(".");
    if (i != -1) return fileName.substring(0, i);
    return fileName;
  }

  /**
   * Gets the name of the jar containing the default launcher xml file, if any.
   *
   * @return the jar name
   */
  protected static String getDefaultJar() {
    URL url = ClassLoader.getSystemResource(defaultFileName + ".xset");
    if (url == null)
      url = ClassLoader.getSystemResource(defaultFileName + ".xml");
    if (url == null) return null;
    String path = url.getPath();
    // trim trailing slash and file name
    int i = path.indexOf("/" + defaultFileName);
    if (i == -1) return null;
    path = path.substring(0, i);
    // jar name is followed by "!"
    i = path.lastIndexOf("!");
    if (i == -1) return null;
    // trim and return jar name
    return path.substring(path.lastIndexOf("/") + 1, i);
  }

  /**
   * Displays the properties of the specified node in an xml inspector.
   *
   * @param node the launch node
   */
  private void inspectXML(LaunchNode node){
    XMLControl xml = new XMLControlElement(node);
    if (node.fileName != null) xml.setValue("path", node.fileName);
    XMLTreePanel treePanel = new XMLTreePanel(xml, false);
    inspector.setContentPane(treePanel);
    inspector.setTitle(LaunchRes.getString("Inspector.Title.Node") +
                       " \"" + node.name + "\"");
    inspector.setVisible(true);
  }
}

/**
 * String resources for launcher classes.
 */
class LaunchRes {

  // static fields
  static ResourceBundle res = ResourceBundle.getBundle(
      "org.opensourcephysics.resources.tools.launcher");

  /**
   * Private constructor to prevent instantiation.
   */
  private LaunchRes() {}

  /**
   * Gets the localized value of a string. If no localized value is found, the
   * key is returned surrounded by exclamation points.
   *
   * @param key the string to localize
   * @return the localized string
   */
  static String getString(String key){
    return res.getString(key);
//    try {
//      return res.getString(key);
//    }
//    catch (MissingResourceException ex){
//      return "!" + key + "!";
//    }
  }
}
