/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.io.*;
import java.util.*;

import java.awt.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import org.opensourcephysics.controls.*;

/**
 * This is a panel that displays a tree with a LaunchNode root.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class LaunchPanel extends JPanel {

  // static fields
  protected static String defaultType = "text"; // text pane editor type

  // instance fields
  protected JTree tree;
  protected DefaultTreeModel treeModel;
  protected JSplitPane splitPane;
  protected JPanel dataPanel;              // on splitPane right
  protected JEditorPane textPane;          // on dataPanel center
  protected JScrollPane textScroller;
  protected boolean showText = true;

  /**
   * Constructor.
   *
   * @param rootNode the root node
   */
  public LaunchPanel(LaunchNode rootNode) {
    createGUI();
    createTree(rootNode);
    setSelectedNode(rootNode);
  }

  /**
   * Sets the selected node.
   *
   * @param node the node to select
   */
  public void setSelectedNode(LaunchNode node) {
    if (node == null) return;
    tree.setSelectionPath(new TreePath(node.getPath()));
  }

  /**
   * Gets the selected node.
   *
   * @return the selected node
   */
  public LaunchNode getSelectedNode() {
    TreePath path = tree.getSelectionPath();
    if (path == null) return null;
    else return (LaunchNode)path.getLastPathComponent();
  }

  /**
   * Gets the root node.
   *
   * @return the root node
   */
  public LaunchNode getRootNode() {
    return (LaunchNode)treeModel.getRoot();
  }

//______________________________ protected methods _____________________________

  /**
   * Returns the node with the same file name as the specified node.
   * May return null.
   *
   * @param node the node to match
   * @return the first node with the same file name
   */
  protected LaunchNode getClone(LaunchNode node) {
    if (node.fileName == null) return null;
    Enumeration e = getRootNode().breadthFirstEnumeration();
    while (e.hasMoreElements()) {
      LaunchNode next = (LaunchNode) e.nextElement();
      if (node.fileName.equals(next.fileName)) return next;
    }
    return null;
  }

  /**
   * Creates the GUI.
   */
  protected void createGUI() {
    setPreferredSize(new Dimension(400, 200));
    setLayout(new BorderLayout());
    splitPane = new JSplitPane();
    add(splitPane, BorderLayout.CENTER);
    dataPanel = new JPanel(new BorderLayout());
    splitPane.setRightComponent(dataPanel);
    textPane = new JTextPane();
    textPane.setEditable(false);
    textPane.addHyperlinkListener(new HyperlinkListener() {
      public void hyperlinkUpdate(HyperlinkEvent e) {
        if (e.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
          try {
            textPane.setPage(e.getURL());
          }
          catch (IOException ex) {
          }
        }
      }
    });
    textScroller = new JScrollPane(textPane);
    dataPanel.add(textScroller, BorderLayout.CENTER);
    splitPane.setDividerLocation(160);
  }

  /**
   * Creates the tree.
   *
   * @param rootNode the root node
   */
  protected void createTree(LaunchNode rootNode) {
    treeModel = new DefaultTreeModel(rootNode);
    tree = new JTree(treeModel);
    tree.setRootVisible(!rootNode.hiddenWhenRoot);
    tree.getSelectionModel().setSelectionMode
            (TreeSelectionModel.SINGLE_TREE_SELECTION);
    tree.addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
        LaunchNode node = getSelectedNode();
        OSPLog.finer(LaunchRes.getString("Log.Message.NodeSelected") + " " + node);
        if (node != null) {
          if (node.url != null) {
            try {
              if (node.url.getContent() != null)
                textPane.setPage(node.url);
            }
            catch (IOException ex) {
              OSPLog.finest(LaunchRes.getString("Log.Message.BadURL") + " " + node.url);
              if (showText) {
                textPane.setContentType(defaultType);
                textPane.setText(node.description);
              }
            }
          }
          else if (showText) {
            textPane.setContentType(defaultType);
            textPane.setText(node.description);
          }
        }
      }
    });
    // put tree in a scroller and add to the split pane
    JScrollPane treeScroller = new JScrollPane(tree);
    splitPane.setLeftComponent(treeScroller);
  }
}
