/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.logging.*;

import java.awt.event.*;
import javax.swing.*;
import javax.swing.tree.*;

import org.opensourcephysics.controls.*;

/**
 * This is a tree node that can describe and launch an application.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class LaunchNode extends DefaultMutableTreeNode {

  // instance fields
  Object launchObj; // object providing xml data when arg 0 is "this"
  String classPath; // path to jar file containing launch and support classes
  String launchClassName; // name of class to be launched
  Class launchClass; // class to be launched
  String[] args = new String[] {""}; // args passed to main method when launching
  boolean showLog = false; // shows OSPLog when in single vm
  Level logLevel = Level.FINE; // OSPLog set to this level before log is shown
  boolean singleVM = false; // launches in current vm
  boolean hiddenWhenRoot = false; // hides node when at root position
  boolean singleton = false; // allows single instance when in separate vm
  boolean singleApp = false; // allows single app when in single vm
  boolean webStart = false; // true when under web start control
  String name = ""; // display name of this node
  String description = ""; // description displayed if no url specified or found
  String author = ""; // author of the curricular activity defined by this node
  String codeAuthor = ""; // author of the program used by this node
  String comment = ""; // comments about this node
  String urlName; // name of url specifying html page to be displayed
  URL url; // url specifying html page to be displayed
  String fileName; // xml file name relative to user directory
  boolean isFileWritable = false; // true if file is writable
  ClassLoader classLoader; // loads classes from jar files
  Collection processes = new HashSet();
  Collection frames = new HashSet();
  int launchCount = 0;
  LaunchPanel launchPanel;


  /**
   * Contructs a node with the specified name.
   *
   * @param name the name
   */
  public LaunchNode(String name) {
    setUserObject(this);
    if (name != null) this.name = name;
//    try {
//      BasicService bs = (BasicService) ServiceManager.lookup(
//          "javax.jnlp.BasicService");
//      webStart = true;
//    }
//    catch (UnavailableServiceException ex) {}
  }

  /**
   * Signals that a launch thread for this node is about to start or end.
   *
   * @param starting true if the thread is starting
   */
  public void threadRunning(boolean starting) {
    launchCount += starting? +1: -1;
    launchCount = Math.max(0, launchCount);
    if (launchPanel != null) launchPanel.repaint();
  }

  /**
   * Launches this node.
   */
  public void launch() {
    launch(null);
  }

  /**
   * Launches this node from the specified launch panel.
   *
   * @param tab the launch panel
   */
  public void launch(LaunchPanel tab) {
    if (!isLeaf()) return;
    launchPanel = tab;
    Launcher.singleVMMode = this.isSingleVM();
    Launcher.singleAppMode = this.isSingleApp();
    String launcherPath = Launcher.classPath; // for later restoral
    Launcher.classPath = getClassPath(); // in node-to-root order
    if (isShowLog() && isSingleVM()) {
      OSPLog.setLevel(getLogLevel());
      OSPLog log = OSPLog.getOSPLog();
      log.setVisible(true);
    }
    setMinimumArgLength(1); // trim args if nec
    String arg0 = args[0];
    if (getLaunchClass() != null) {
      if (arg0.equals("this")) {
        Object launchObj = getLaunchObject();
        if (launchObj != null) {
          // replace with xml from launch object, if any
          XMLControl control = new XMLControlElement(launchObj);
          args[0] = control.toXML();
        }
        else args[0] = "";
      }
      if (args[0].equals("") && args.length == 1)
        Launcher.launch(getLaunchClass(), null, this);
      else
        Launcher.launch(getLaunchClass(), args, this);
    }
    // restore
    Launcher.singleVMMode = false;
    Launcher.singleAppMode = false;
    Launcher.classPath = launcherPath;
    args[0] = arg0;
  }

  /**
   * Returns the nearest ancestor with a non-null file name.
   *
   * @return the file node
   */
  public LaunchNode getOwner() {
    if (fileName != null) return this;
    if (getParent() != null) {
      return ((LaunchNode)getParent()).getOwner();
    }
    return null;
  }

  /**
   * Returns the descendents of this node with non-null file names.
   *
   * @return an array of launch nodes
   */
  public LaunchNode[] getOwnedNodes() {
    Collection nodes = new ArrayList();
    Enumeration e = breadthFirstEnumeration();
    while (e.hasMoreElements()) {
      LaunchNode next = (LaunchNode) e.nextElement();
      if (next.fileName != null) nodes.add(next);
    }
    return (LaunchNode[])nodes.toArray(new LaunchNode[0]);
  }

  /**
   * Returns a string used as a display name for this node.
   *
   * @return the string name of this node
   */
  public String toString() {
    // return name, if any
    if (name != null && !name.equals(""))
      return name;
    // else return launch class name, if any
    if (launchClassName != null) {
      int i = launchClassName.lastIndexOf(".");
      if (i > 0 && i < launchClassName.length() - 1) {
        return launchClassName.substring(i + 1, launchClassName.length());
      }
    }
    // else return args[0], if any
    if (!args[0].equals("")) {
      String name = args[0];
      // remove extension
      int i = name.lastIndexOf(".");
      if (i > 0 && i < name.length() - 1) {
        name = name.substring(0, i);
      }
      // remove path
      i = name.lastIndexOf("/");
      if (i > 0 && i < name.length() - 1) {
        name = name.substring(i + 1);
      }
      i = name.lastIndexOf("\\");
      if (i > 0 && i < name.length() - 1) {
        name = name.substring(i + 1);
      }
      return name;
    }
    return "";
  }

  /**
   * Gets the unique ID string for this node.
   *
   * @return the ID string
   */
  public String getID() {
    return String.valueOf(this.hashCode());
  }

  /**
   * Sets the name of this node.
   *
   * @param name the name
   */
  public void setName(String name) {
    this.name = name;
  }

  /**
   * Sets the description of this node.
   *
   * @param desc the description
   */
  public void setDescription(String desc) {
    this.description = desc;
  }

  /**
   * Sets the launch arguments of this node.
   *
   * @param args the arguments
   */
  public void setArgs(String[] args) {
    if (args != null && args.length > 0 && args[0] != null)
      this.args = args;
  }

  /**
   * Gets the complete class path in node-to-root order.
   *
   * @return the class path
   */
  public String getClassPath() {
    String path = "";
    if (classPath != null) path += classPath;
    LaunchNode node = this;
    while (!node.isRoot()) {
      node = (LaunchNode)node.getParent();
      if (node.classPath != null) {
        if (!path.equals("")) path += ";";
        path += node.classPath;
      }
    }
    return path;
  }

  /**
   * Sets the class path (jar file names separated by semicolons) for this node.
   *
   * @param jarNames the class path
   */
  public void setClassPath(String jarNames) {
    classPath = jarNames;
  }

  /**
   * Sets the launch class for this node.
   *
   * @param className the name of the class
   * @return true if the class was successfully loaded for the first time
   */
  public boolean setLaunchClass(String className) {
    if (className == null) return false;
    if (launchClassName == className && launchClass != null) return false;
    OSPLog.finest(LaunchRes.getString("Log.Message.SetLaunchClass") + " " + className);
    launchClassName = className;
    launchClass = LaunchClassChooser.getClass(getClassPath(), className);
    return launchClass != null;
  }

  /**
   * Gets the launch class for this node.
   *
   * @return the launch class
   */
  public Class getLaunchClass() {
    if (launchClass == null &&
        launchClassName != null &&
        !launchClassName.equals("")) {
      setLaunchClass(launchClassName);
    }
    return launchClass;
  }

  /**
   * Gets the launch object. May be null.
   *
   * @return the launch object
   */
  public Object getLaunchObject() {
    if (launchObj != null) return launchObj;
    else if (isRoot()) return null;
    LaunchNode node = (LaunchNode)getParent();
    return node.getLaunchObject();
  }

  /**
   * Sets the launch object.
   *
   * @param obj the launch object
   */
  public void setLaunchObject(Object obj) {
    launchObj = obj;
  }

  /**
   * Sets the url.
   *
   * @param name the name of the url
   * @return the url
   */
  public URL setURL(String name) {
    if (name == null || name.equals("")) {
      urlName = null;
      url = null;
      return null;
    }
    urlName = name;
    OSPLog.finest(LaunchRes.getString("Log.Message.SetURLPath") + " " + name);
    String err = "";
    // try getting URL with name alone
    String path = name;
    try {
      return setURL(new URL(path));
    }
    catch (Exception ex) {
      try {
        return setURL(Launcher.class.getResource(path));
      }
      catch (Exception ex1) {
        err += XML.NEW_LINE + LaunchRes.getString("Log.Message.NotFound") +
            " " + path;
      }
    }
    // try adding slash for resource at root
    if (!name.startsWith("/")) {
      path = "/" + name;
      try {
        return setURL(Launcher.class.getResource(path));
      }
      catch (Exception ex1) {
        err += XML.NEW_LINE + LaunchRes.getString("Log.Message.NotFound") +
            " " + path;
      }
    }
    // try adding http protocol
    if (name.indexOf("http:") == -1) {
      path = "http:" + name;
      try {
        return setURL(new URL(path));
      }
      catch (Exception ex1) {
        err += XML.NEW_LINE + LaunchRes.getString("Log.Message.NotFound") +
            " " + path;
      }
    }
    // try adding file protocol
    if (name.indexOf("file:") == -1) {
      path = "file:" + name;
      try {
        return setURL(new URL(path));
      }
      catch (Exception ex1) {
        err += XML.NEW_LINE + LaunchRes.getString("Log.Message.NotFound") +
            " " + path;
      }
    }
    OSPLog.info(err);
    url = null;
    return null;
  }

  /**
   * Gets the singleVM flag.
   *
   * @return true if singleVM is true for this or an ancestor
   */
  public boolean isSingleVM() {
    if (singleVM || webStart) return true;
    else if (isRoot()) return false;
    LaunchNode node = (LaunchNode)getParent();
    return node.isSingleVM();
  }

  /**
   * Sets the single VM flag.
   *
   * @param singleVM true if single vm
   */
  public void setSingleVM(boolean singleVM) {
    this.singleVM = singleVM;
  }

  /**
   * Gets the showLog value.
   *
   * @return true if showLog is true for this or an ancestor
   */
  public boolean isShowLog() {
    if (showLog) return true;
    else if (isRoot()) return false;
    LaunchNode parent = (LaunchNode)getParent();
    return parent.isShowLog();
  }

  /**
   * Sets the showLog flag.
   *
   * @param show true to show the OSPLog (single vm only)
   */
  public void setShowLog(boolean show) {
    showLog = show;
  }

  /**
   * Gets the log level.
   *
   * @return the level
   */
  public Level getLogLevel() {
    if (isRoot()) return logLevel;
    LaunchNode parent = (LaunchNode)getParent();
    if (parent.isShowLog()) {
      Level parentLevel = parent.getLogLevel();
      if (parentLevel.intValue() < logLevel.intValue()) return parentLevel;
    }
    return logLevel;
  }

  /**
   * Sets the log level.
   *
   * @param level the level
   */
  public void setLogLevel(Level level) {
     if (level != null) logLevel = level;
  }

  /**
   * Gets the singleApp value.
   *
   * @return true if singleApp is true for this or an ancestor
   */
  public boolean isSingleApp() {
    if (singleApp) return true;
    else if (isRoot()) return false;
    LaunchNode parent = (LaunchNode)getParent();
    return parent.isSingleApp();
  }

  /**
   * Sets the singleApp flag.
   *
   * @param singleApp true to close other apps when launching new app (single vm only)
   */
  public void setSingleApp(boolean singleApp) {
    this.singleApp = singleApp;
  }

  /**
   * Sets the hiddenWhenRoot flag.
   *
   * @param hide true to hide node when at root
   */
  public void setHiddenWhenRoot(boolean hide) {
    hiddenWhenRoot = hide;
  }

  /**
   * Gets the singleton value.
   *
   * @return true if singleApp is true for this or an ancestor
   */
  public boolean isSingleton() {
    if (singleton) return true;
    else if (isRoot()) return false;
    LaunchNode parent = (LaunchNode)getParent();
    return parent.isSingleton();
  }

  /**
   * Sets the singleton flag.
   *
   * @param singleton true to allow single instance when in separate vm
   */
  public void setSingleton(boolean singleton) {
    this.singleton = singleton;
  }

  /**
   * Determines if this node matches another node.
   *
   * @param node the node to match
   * @return true if the nodes match
   */
  public boolean matches(LaunchNode node) {
    if (node == null) return false;
    boolean match =
        showLog == node.showLog &&
        singleton == node.singleton &&
        singleVM == node.singleVM &&
        hiddenWhenRoot == node.hiddenWhenRoot &&
        name.equals(node.name) &&
        description.equals(node.description) &&
        args[0].equals(node.args[0]) &&
        ((fileName == null && node.fileName == null) ||
         (fileName != null && fileName.equals(node.fileName))) &&
        ((getLaunchClass() == null && node.getLaunchClass() == null) ||
         (getLaunchClass() != null && getLaunchClass().equals(node.getLaunchClass()))) &&
        ((classPath == null && node.classPath == null) ||
         (classPath != null && classPath.equals(node.classPath)));
    return match;
  }

  /**
   * Gets a child node specified by fileName.
   *
   * @param childFileName the file name of the child
   * @return the first child found, or null
   */
  public LaunchNode getChildNode(String childFileName) {
    Enumeration e = breadthFirstEnumeration();
    while (e.hasMoreElements()) {
      LaunchNode next = (LaunchNode) e.nextElement();
      if (childFileName.equals(next.fileName)) return next;
    }
    return null;
  }

  /**
   * Adds menu item to a JPopupMenu or JMenu.
   *
   * @param menu the menu
   */
  public void addMenuItemsTo(final JComponent menu) {
    Enumeration e = children();
    while (e.hasMoreElements()) {
      LaunchNode child = (LaunchNode) e.nextElement();
      if (child.isLeaf()) {
        JMenuItem item = new JMenuItem(child.toString());
        menu.add(item);
        item.setToolTipText(child.description);
        item.setActionCommand(child.getID());
        item.addActionListener(new ActionListener() {
          public void actionPerformed(ActionEvent e) {
            String id = e.getActionCommand();
            LaunchNode root = (LaunchNode) LaunchNode.this.getRoot();
            Enumeration e2 = root.postorderEnumeration();
            while (e2.hasMoreElements()) {
              LaunchNode node = (LaunchNode) e2.nextElement();
              if (node.getID().equals(id)) {
                node.launch();
                break;
              }
            }
          }
        });
      }
      else {
        JMenu item = new JMenu(child.toString());
        menu.add(item);
        child.addMenuItemsTo(item);
      }
    }
  }

  /**
   * Returns the XML.ObjectLoader for this class.
   *
   * @return the object loader
   */
  public static XML.ObjectLoader getLoader() {
    return new Loader();
  }

  /**
   * A class to save and load LaunchNode data in an XMLControl.
   */
  private static class Loader extends XMLLoader {

    public void saveObject(XMLControl control, Object obj) {
      LaunchNode node = (LaunchNode) obj;
      node.setMinimumArgLength(1); // trim args if nec
      if (!node.name.equals(""))
        control.setValue("name", node.name);
      if (!node.description.equals(""))
        control.setValue("description", node.description);
      if (node.urlName != null)
        control.setValue("url", node.urlName);
      if (node.getLaunchClass() != null)
        control.setValue("launch_class", node.getLaunchClass().getName());
      else if (node.launchClassName != null)
        control.setValue("launch_class", node.launchClassName);
      if (!node.args[0].equals("") || node.args.length > 1)
        control.setValue("launch_args", node.args);
      if (node.classPath != null && !node.classPath.equals(""))
        control.setValue("classpath", node.classPath);
      if (node.hiddenWhenRoot)
        control.setValue("root_hidden", true);
      if (node.singleton)
        control.setValue("singleton", true);
      if (node.singleVM)
        control.setValue("single_vm", true);
      if (node.showLog)
        control.setValue("show_log", true);
      if (node.singleApp)
        control.setValue("single_app", true);
      if (!node.author.equals(""))
        control.setValue("author", node.author);
      if (!node.codeAuthor.equals(""))
        control.setValue("code_author", node.codeAuthor);
      if (!node.comment.equals(""))
        control.setValue("comment", node.comment);
      if (node.children != null) {
        // list the children and save the list
        ArrayList children = new ArrayList();
        Enumeration e = node.children();
        while (e.hasMoreElements()) {
          LaunchNode child = (LaunchNode)e.nextElement();
          if (child.fileName != null) {
            // save file name relative to this node's directory
            String base = XML.getDirectoryPath(node.fileName);
            String path = XML.getPathRelativeTo(child.fileName, base);
            children.add(path);
          }
          else children.add(child);
        }
        control.setValue("child_nodes", children);
      }
    }

    public Object createObject(XMLControl control){
      String name = control.getString("name");
      return new LaunchNode(name);
    }

    public Object loadObject(XMLControl control, Object obj) {
      LaunchNode node = (LaunchNode) obj;
      String name = control.getString("name");
      if (name != null) node.name = name;
      String description = control.getString("description");
      if (description != null) node.description = description;
      node.setURL(control.getString("url"));
      node.classPath = control.getString("classpath");
      String className = control.getString("launch_class");
      if (className != null) node.launchClassName = className;
      String[] args = (String[])control.getObject("launch_args");
      if (args != null) node.setArgs(args);
      node.hiddenWhenRoot = control.getBoolean("root_hidden");
      node.singleton = control.getBoolean("singleton");
      node.singleVM = control.getBoolean("single_vm");
      node.showLog = control.getBoolean("show_log");
      node.singleApp = control.getBoolean("single_app");
      String author = control.getString("author");
      if (author != null) node.author = author;
      author = control.getString("code_author");
      if (author != null) node.codeAuthor = author;
      String comment = control.getString("comment");
      if (comment != null) node.comment = comment;
      Collection children = (ArrayList)control.getObject("child_nodes");
      if (children != null) {
        node.removeAllChildren();
        Iterator it = children.iterator();
        while (it.hasNext()) {
          Object next = it.next();
          if (next instanceof LaunchNode) { // child node
            // add the child node
            LaunchNode child =  (LaunchNode) next;
            node.add(child);
            child.setLaunchClass(child.launchClassName);
          }
          else if (next instanceof String) { // file name
            String fileName = (String)next;
            ResourceLoader.addSearchPath(Launcher.resourcesPath);
            ResourceLoader.addSearchPath(Launcher.tabSetBasePath);
            // last path added is first path searched
            ResourceLoader.addSearchPath(Launcher.xmlBasePath);
            // open the file in an xml control
            XMLControlElement childControl = new XMLControlElement();
            String absolutePath = childControl.read(fileName);
            if (childControl.failedToRead()) {
              JOptionPane.showMessageDialog(
                  null,
                  LaunchRes.getString("Dialog.InvalidXML.Message")
                  + " \"" + fileName + "\"",
                  LaunchRes.getString("Dialog.InvalidXML.Title"),
                  JOptionPane.WARNING_MESSAGE);
            }
            fileName = XML.getRelativePath(absolutePath);
            // check root to prevent circular references
            LaunchNode root = (LaunchNode)node.getRoot();
            if (root.getChildNode(fileName) != null ||
                fileName.equals(root.fileName))
              continue;
            Class type = childControl.getObjectClass();
            if (LaunchNode.class.isAssignableFrom(type)) {
              // first add the child with just the fileName
              // this allows descendents to get the root
              LaunchNode child = new LaunchNode(LaunchRes.getString("NewNode.Name"));
              child.fileName = fileName;
              OSPLog.finest(LaunchRes.getString("Log.Message.Loading") +
                            ": " + absolutePath);
              node.add(child);
              // then load the child with data
              childControl.loadObject(child);
              child.isFileWritable = childControl.canWrite;
            }
          }
        }
      }
      return node;
    }
  }

  private URL setURL(URL url) throws Exception {
    InputStream in = url.openStream();
    in.close();
    OSPLog.finer(LaunchRes.getString("Log.Message.URL") + " " + url);
    this.url = url;
    return url;
  }

  protected void setMinimumArgLength(int n) {
    n = Math.max(n, 1); // never shorter than 1 element
    if (n == args.length) return;
    if (n > args.length) {
      // increase the size of the args array
      String[] newArgs = new String[n];
      for (int i = 0; i < n; i++) {
        if (i < args.length)
          newArgs[i] = args[i];
        else newArgs[i] = "";
      }
      setArgs(newArgs);
    }
    else {
      while (args.length > n && args[args.length-1].equals("")) {
        String[] newArgs = new String[args.length-1];
        for (int i = 0; i < newArgs.length; i++) {
          newArgs[i] = args[i];
        }
        setArgs(newArgs);
      }
    }
  }

}
