/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.io.*;
import java.util.*;
import java.util.logging.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import java.awt.font.*;
import java.awt.geom.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.tree.*;

import org.opensourcephysics.controls.*;

/**
 * This provides a GUI for building LaunchNode xml files.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class LaunchBuilder extends Launcher {

  static JFileChooser fileChooser;
  static javax.swing.filechooser.FileFilter jarFileFilter;
  static javax.swing.filechooser.FileFilter htmlFileFilter;
  static javax.swing.filechooser.FileFilter allFileFilter;
  static final Color RED = new Color(255, 102, 102);
  static final Color YELLOW = Color.yellow;
  static int maxArgs = 8;

  // instance fields
  Action changeAction;      // loads input data into selected node
  Action newTreeAction;     // creates a new root node and tree
  Action addAction;         // adds new node to selected node
  Action removeAction;      // removes the selected node
  Action cutAction;         // cuts selected node to clipboard
  Action copyAction;        // copies selected node to clipboard
  Action pasteAction;       // adds clipboard node to selected node
  Action importAction;      // adds xml file node to selected node
  Action saveAsAction;      // saves selected node to a file
  Action moveUpAction;      // moves selected node up
  Action moveDownAction;    // moves selected node down
  Action openJarAction;     // opens a jar file and sets node path
  Action searchJarAction;   // searches the current jars for a launchable class
  Action openArgAction;     // opens an xml file and sets the current argument
  Action openURLAction;     // opens an html file and sets url
  Action openTabAction;     // opens a file node in a new tab
  FocusListener focusListener;
  KeyListener keyListener;
  JTabbedPane editorTabs;
  JTextField nameField;
  JTextField classField;
  JTextField argField;
  JSpinner argSpinner;
  JTextField jarField;
  JTextField urlField;
  JPanel urlPanel;
  JTextPane descriptionPane;
  JScrollPane descriptionScroller;
  JSplitPane displaySplitPane;
  JTextField authorField;
  JTextField codeAuthorField;
  JTextPane commentPane;
  JScrollPane commentScroller;
  JCheckBox editorEnabledCheckBox;
  JCheckBox hideRootCheckBox;
  JCheckBox singleVMCheckBox;
  JCheckBox showLogCheckBox;
  JCheckBox singletonCheckBox;
  JCheckBox singleAppCheckBox;
  JComboBox levelDropDown;
  Level[] allLevels;
  String previousPath;
  JMenuItem newItem;
  JMenuItem previewItem;
  JMenuItem saveItem;
  JMenuItem saveAsItem;
  JMenuItem saveAllItem;
  JMenuItem importItem;
  JMenuItem saveSetAsItem;
  JToolBar toolbar;

  /**
   * No-arg constructor.
   */
  public LaunchBuilder() {
  }

  /**
   * Constructs a builder and loads the specified file.
   *
   * @param fileName the file name
   */
  public LaunchBuilder(String fileName) {
    super(fileName);
  }

  /**
   * Main entry point when used as application.
   *
   * @param args args[0] may be an xml file name
   */
  public static void main(String[] args) {
//    java.util.Locale.setDefault(new java.util.Locale("es"));
//    OSPLog.setLevel(ConsoleLevel.ALL);
    String fileName = null;
    if (args != null && args.length != 0) fileName = args[0];
    LaunchBuilder builder = new LaunchBuilder(fileName);
    builder.frame.setVisible(true);
  }

  /**
   * Saves a node to the specified file.
   *
   * @param node the node
   * @param fileName the desired name of the file
   * @return the name of the saved file, or null if not saved
   */
  public String save(LaunchNode node, String fileName) {
    if (node == null) return null;
    if (fileName == null ||
        fileName.trim().equals("")) {
      return saveAs(node);
    }
    if (getExtension(fileName) == null) {
      while (fileName.endsWith(".")) {
        fileName = fileName.substring(0, fileName.length()-1);
      }
      fileName += ".xml";
    }
    if (!saveOwnedNodes(node)) return null;
    OSPLog.fine(fileName);
    XMLControlElement control = new XMLControlElement(node);
    control.write(fileName);
    if (!control.canWrite) {
      OSPLog.info(LaunchRes.getString("Dialog.SaveFailed.Message")
                     + " " + fileName);
      JOptionPane.showMessageDialog(
          null,
          LaunchRes.getString("Dialog.SaveFailed.Message") + " " + fileName,
          LaunchRes.getString("Dialog.SaveFailed.Title"),
          JOptionPane.WARNING_MESSAGE);
      return null;
    }
    node.fileName = fileName;
    node.isFileWritable = true;
    changedFiles.remove(fileName);
    return fileName;
  }

  /**
   * Saves a node to an xml file selected with a chooser.
   *
   * @param node the node
   * @return the name of the file
   */
  public String saveAs(LaunchNode node) {
    if (node.fileName != null)
      getXMLChooser().setSelectedFile(new File(node.fileName));
    else {
      String name = node.name + ".xml";
      getXMLChooser().setSelectedFile(new File(name));
    }
    int result = getXMLChooser().showDialog(null,
        LaunchRes.getString("Menu.File.SaveTab"));
    if (result == JFileChooser.APPROVE_OPTION) {
      File file = getXMLChooser().getSelectedFile();
      if (node.fileName != null &&
          node.fileName.endsWith(file.getName())) {
        // check node.fileName directory structure and reproduce if requested
        int n = node.fileName.lastIndexOf("/");
        if (n > 0) {
          String dirName = node.fileName.substring(0, n+1);
          String parent = file.getParent() + "/";
          // replace backslashes with forward slashes
          int i = parent.indexOf("\\");
          while (i != -1) {
            parent = parent.substring(0, i) + "/" + parent.substring(i + 1);
            i = parent.indexOf("\\");
          }
          if (!parent.endsWith(dirName)) {
            int selected = JOptionPane.showConfirmDialog(
                frame,
                LaunchRes.getString("Dialog.MakeDirectory.Message") +
                " " + dirName + XML.NEW_LINE +
                LaunchRes.getString("Dialog.MakeDirectory.Question"),
                LaunchRes.getString("Dialog.MakeDirectory.Title"),
                JOptionPane.YES_NO_OPTION);
            if (selected == JOptionPane.YES_OPTION) {
              File dir = new File(parent, dirName);
              if (dir.exists() || dir.mkdir()) {
                getXMLChooser().setCurrentDirectory(dir);
                file = new File(parent, node.fileName);
              }
            }
          }
        }
      }
      // check to see if file already exists
      if (file.exists()) {
        int selected = JOptionPane.showConfirmDialog(
            frame,
            LaunchRes.getString("Dialog.ReplaceFile.Message") +
            " " + file.getName() + XML.NEW_LINE +
            LaunchRes.getString("Dialog.ReplaceFile.Question"),
            LaunchRes.getString("Dialog.ReplaceFile.Title"),
            JOptionPane.YES_NO_OPTION);
        if (selected != JOptionPane.YES_OPTION) {
          return null;
        }
      }
      String fileName = XML.getRelativePath(file.getAbsolutePath());
      // get clones before saving
      Map clones = getClones(node);
      String path = save(node, fileName);
      if (path != null) {
        if (node.isRoot()) {
          // refresh title of root tab
          for (int i = 0; i < tabbedPane.getTabCount(); i++) {
            LaunchPanel tab = (LaunchPanel)tabbedPane.getComponentAt(i);
            if (tab.getRootNode() == node) {
              tabbedPane.setTitleAt(i, getDisplayName(path));
              break;
            }
          }
        }
        // refresh clones
        for (Iterator it = clones.keySet().iterator(); it.hasNext();) {
          LaunchPanel cloneTab = (LaunchPanel)it.next();
          LaunchNode clone = (LaunchNode)clones.get(cloneTab);
          clone.fileName = node.fileName;
          clone.isFileWritable = true;
          // refresh title of clone tab
          if (clone == cloneTab.getRootNode()) {
            int n = tabbedPane.indexOfComponent(cloneTab);
            tabbedPane.setTitleAt(n, getDisplayName(path));
          }
        }
        if (tabSetPath != null) changedFiles.add(tabSetPath);
      }
      return path;
    }
    return null;
  }

//______________________________ protected methods _____________________________

  /**
   * Saves the owned nodes of the specified node.
   *
   * @param node the node
   * @return true unless cancelled by user
   */
  public boolean saveOwnedNodes(LaunchNode node) {
    if (node == null) return false;
    LaunchNode[] nodes = node.getOwnedNodes();
    for (int i = 0; i < nodes.length; i++) {
      if (nodes[i] == node) continue;
      if (nodes[i].getOwnedNodes().length > 1) {
        if (!saveOwnedNodes(nodes[i])) return false;
      }
      if (nodes[i].isFileWritable) {
        if (save(nodes[i], nodes[i].fileName) == null)return false;
      }
      else if (saveAs(nodes[i]) == null) return false;
    }
    return true;
  }

  /**
   * Saves a set of launch nodes to an xml file selected with a chooser.
   *
   * @return the name of the file
   */
  protected String saveTabSetAs() {
    if (tabSetPath != null)
      getXMLChooser().setSelectedFile(new File(tabSetPath));
    else if (getRootNode() != null) {
      String name = getDisplayName(getRootNode().fileName) + ".xset";
      getXMLChooser().setSelectedFile(new File(name));
    }
    else getXMLChooser().setSelectedFile(null);
    int result = getXMLChooser().showDialog(null,
        LaunchRes.getString("Menu.File.SaveSet"));
    if (result == JFileChooser.APPROVE_OPTION) {
      File file = getXMLChooser().getSelectedFile();
      if (tabSetPath != null &&
          tabSetPath.endsWith(file.getName())) {
        // check tabSetPath directory structure and reproduce if requested
        int n = tabSetPath.lastIndexOf("/");
        if (n > 0) {
          String dirName = tabSetPath.substring(0, n+1);
          String parent = file.getParent() + "/";
          // replace backslashes with forward slashes
          int i = parent.indexOf("\\");
          while (i != -1) {
            parent = parent.substring(0, i) + "/" + parent.substring(i + 1);
            i = parent.indexOf("\\");
          }
          if (!parent.endsWith(dirName)) {
            int selected = JOptionPane.showConfirmDialog(
                frame,
                LaunchRes.getString("Dialog.MakeDirectory.Tabset.Message") +
                " " + dirName + XML.NEW_LINE +
                LaunchRes.getString("Dialog.MakeDirectory.Question"),
                LaunchRes.getString("Dialog.MakeDirectory.Title"),
                JOptionPane.YES_NO_OPTION);
            if (selected == JOptionPane.YES_OPTION) {
              File dir = new File(parent, dirName);
              if (dir.exists() || dir.mkdir()) {
                getXMLChooser().setCurrentDirectory(dir);
                file = new File(parent, tabSetPath);
              }
            }
          }
        }
      }
      // check to see if file already exists
      if (file.exists()) {
        int selected = JOptionPane.showConfirmDialog(
            frame,
            LaunchRes.getString("Dialog.ReplaceFile.Message") +
            " " + file.getName() + XML.NEW_LINE +
            LaunchRes.getString("Dialog.ReplaceFile.Question"),
            LaunchRes.getString("Dialog.ReplaceFile.Title"),
            JOptionPane.YES_NO_OPTION);
        if (selected != JOptionPane.YES_OPTION) {
          return null;
        }
      }
      String fileName = XML.getRelativePath(file.getAbsolutePath());
      tabSetPath = saveTabSet(fileName);
      return tabSetPath;
    }
    return null;
  }

  /**
   * Saves a set of launch nodes to the specified file.
   *
   * @param fileName the name of the file
   * @return the full path to the saved file
   */
  protected String saveTabSet(String fileName) {
    if (fileName == null ||
        fileName.trim().equals("")) {
      return saveTabSetAs();
    }
    if (getExtension(fileName) == null) {
      while (fileName.endsWith(".")) {
        fileName = fileName.substring(0, fileName.length()-1);
      }
      fileName += ".xset";
    }
    // save all tabs
    if (!saveAllTabs()) return null;
    // save launch set
    OSPLog.fine(fileName);
    LaunchSet set = new LaunchSet(this, fileName);
    XMLControlElement control = new XMLControlElement(set);
    control.write(fileName);
    isTabSetWritable = control.canWrite;
    changedFiles.clear();
    if (spawner != null) {
      spawner.tabSetPath = null; // allows it to open same filename
      spawner.open(fileName);
      spawner.refreshGUI();
    }
    return fileName;
  }

  /**
   * Saves all tabs.
   * @return true unless cancelled by user
   */
  protected boolean saveAllTabs() {
    Component[] tabs = tabbedPane.getComponents();
    for (int i = 0; i < tabs.length; i++) {
      LaunchPanel tab = (LaunchPanel) tabs[i];
      LaunchNode root = tab.getRootNode();
      if (root.isFileWritable) {
        if (save(root, root.fileName) == null) return false;
      }
      else if (saveAs(root) == null) return false;
    }
    return true;
  }

  /**
   * Refreshes the selected node.
   */
  protected void refreshSelectedNode() {
    refreshNode(getSelectedNode());
  }

  /**
   * Refreshes the specified node with data from the input fields.
   *
   * @param node the node to refresh
   */
  protected void refreshNode(LaunchNode node) {
    boolean changed = false;
    if (node != null) {
      if (node.isSingleVM() != singleVMCheckBox.isSelected()) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeVM"));
        node.singleVM = singleVMCheckBox.isSelected();
        if (node.isSingleVM()) {
          showLogCheckBox.setSelected(node.showLog);
          singleAppCheckBox.setSelected(node.singleApp);
        }
        else singletonCheckBox.setSelected(node.singleton);
        changed = true;
      }
      if (node.isSingleVM() && node.showLog != showLogCheckBox.isSelected()) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeShowLog"));
        node.showLog = showLogCheckBox.isSelected();
        changed = true;
      }
      if (node.isSingleVM() && node.singleApp != singleAppCheckBox.isSelected()) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeSingleApp"));
        node.singleApp = singleAppCheckBox.isSelected();
        changed = true;
      }
      if (node.singleton != singletonCheckBox.isSelected()) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeSingleton"));
        node.singleton = singletonCheckBox.isSelected();
        changed = true;
      }
      if (!node.name.equals(nameField.getText())) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeName"));
        node.name = nameField.getText();
        changed = true;
      }
      if (!node.description.equals(descriptionPane.getText())) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeDesc"));
        node.description = descriptionPane.getText();
        changed = true;
      }
      int n = ((Integer)argSpinner.getValue()).intValue();
      String arg = argField.getText();
      if (!arg.equals("")) node.setMinimumArgLength(n+1);
      if (node.args.length > n && !arg.equals(node.args[n])) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeArgs") + " " + n);
        node.args[n] = arg;
        if (arg.equals("")) node.setMinimumArgLength(1);
        changed = true;
      }
      String jarPath = jarField.getText();
      if ((jarPath.equals("") && node.classPath != null) ||
          (!jarPath.equals("") && !jarPath.equals(node.classPath))) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodePath"));
        node.classPath = jarPath.equals("")? null: jarPath;
        changed = true;
      }
      String input = urlField.getText();
      if (input.equals("")) input = null;
      if ((node.urlName != null && !node.urlName.equals(input)) ||
          (input != null && !input.equals(node.urlName))) {
        node.setURL(input);
        changed = true;
      }
      String className = classField.getText();
      if (className.equals("")) {
        if (node.launchClassName != null) {
          node.launchClassName = null;
          node.launchClass = null;
          OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeLaunchClass"));
          changed = true;
        }
      }
      else if (!className.equals(node.launchClassName) ||
               !className.equals("") && node.getLaunchClass() == null) {
        changed = node.setLaunchClass(className);
        if (changed)
          OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeLaunchClass"));
      }
      if (!node.author.equals(authorField.getText())) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeAuthor"));
        node.author = authorField.getText();
        changed = true;
      }
      if (!node.codeAuthor.equals(codeAuthorField.getText())) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeCodeAuthor"));
        node.codeAuthor = codeAuthorField.getText();
        changed = true;
      }
      if (!node.comment.equals(commentPane.getText())) {
        OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeComment"));
        node.comment = commentPane.getText();
        changed = true;
      }
      LaunchNode root = (LaunchNode)node.getRoot();
      if (root != null) {
        boolean hide = hideRootCheckBox.isSelected();
        if (hide != root.hiddenWhenRoot) {
          root.hiddenWhenRoot = hide;
          OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeRootHidden"));
          changed = true;
        }
        boolean edit = editorEnabledCheckBox.isSelected();
        if (edit != editorEnabled) {
          editorEnabled = edit;
          OSPLog.finest(LaunchRes.getString("Log.Message.ChangeNodeEditorEnabled"));
          if (tabSetPath != null) changedFiles.add(tabSetPath);
        }
      }
      if (changed) {
        OSPLog.fine(LaunchRes.getString("Log.Message.ChangeNode") +
                    " \"" + node.toString() + "\"");
        LaunchPanel tab = getSelectedTab();
        if (tab != null) tab.treeModel.nodeChanged(node);
        changedFiles.add(node.getOwner().fileName);
        refreshClones(node);
        refreshGUI();
      }
    }
  }

  /**
   * Creates a LaunchPanel with the specified root and adds it to a new tab.
   *
   * @param root the root node
   */
  protected void addTab(LaunchNode root) {
    if (root.fileName == null) {
      String fileName = saveAs(root);
      if (fileName == null) return;
      root.fileName = fileName;
      root.isFileWritable = true;
    }
    OSPLog.finest(root.toString());
    super.addTab(root);
    tabbedPane.setIconAt(tabbedPane.getSelectedIndex(), fileIcon);
    final LaunchPanel tab = getSelectedTab();
    tab.showText = false;
    tab.textPane.setText(null);
    if (tabSetPath != null) changedFiles.add(tabSetPath);
    tab.dataPanel.addComponentListener(new ComponentAdapter() {
      public void componentResized(ComponentEvent e) {
//        changedFiles.add(tabSetPath);
      }
    });
  }

  /**
   * Removes the selected tab.
   *
   * @return true if the tab was removed
   */
  protected boolean removeSelectedTab() {
    // check for unsaved changes in the selected tab
    LaunchPanel tab = (LaunchPanel) tabbedPane.getSelectedComponent();
    if (tab != null) {
      LaunchNode[] nodes = tab.getRootNode().getOwnedNodes();
      for (int j = 0; j < nodes.length; j++) {
        if (!saveChanges(nodes[j])) return false;
      }
    }
    boolean removed = super.removeSelectedTab();
    if (tabSetPath != null) changedFiles.add(tabSetPath);
    return removed;
  }

  /**
   * Offers to save changes, if any, to the root of the specified node.
   *
   * @param node the node to check
   * @return true unless cancelled by user
   */
  protected boolean saveChanges(LaunchNode node) {
    if (changedFiles.contains(node.fileName)) {
      LaunchNode root = (LaunchNode)node.getRoot();
      int selected = JOptionPane.showConfirmDialog(
              frame,
              LaunchRes.getString("Dialog.SaveChanges.Message") +
              XML.NEW_LINE +
              LaunchRes.getString("Dialog.SaveChanges.Question"),
              LaunchRes.getString("Dialog.SaveChanges.Title"),
              JOptionPane.YES_NO_CANCEL_OPTION);
      if (selected == JOptionPane.CANCEL_OPTION)
        return false;
      if (selected == JOptionPane.YES_OPTION) {
        if (node.isFileWritable) {
          if (save(node, node.fileName) == null) return false;
        }
        else {
          if (saveAs(node) == null) return false;
        }
      }
    }
    return true;
  }

  /**
   * Offers to save all changes, if any.
   *
   * @return true unless cancelled by user
   */
  protected boolean saveAllChanges() {
    // save changes to tab set
    if (!changedFiles.isEmpty()) {
      int selected = JOptionPane.showConfirmDialog(
          frame,
          LaunchRes.getString("Dialog.SaveChanges.Tabset.Message") +
          XML.NEW_LINE +
          LaunchRes.getString("Dialog.SaveChanges.Question"),
          LaunchRes.getString("Dialog.SaveChanges.Title"),
          JOptionPane.YES_NO_CANCEL_OPTION);
      if (selected == JOptionPane.CANCEL_OPTION) return false;
      if (selected == JOptionPane.YES_OPTION) {
        if (isTabSetWritable) {
          if (saveTabSet(tabSetPath) == null) return false;
        }
        else {
          if (saveTabSetAs() == null) return false;
        }
      }
    }
    return true;
  }

  /**
   * Refreshes the GUI.
   */
  protected void refreshGUI() {
    if (previousNode != null) { // new tab has been selected
      LaunchNode prev = previousNode;
      previousNode = null;
      refreshNode(prev);
    }
    if (newNodeSelected) argSpinner.setValue(new Integer(0));
    super.refreshGUI();
    final LaunchNode node = getSelectedNode();
    if (node != null) {
      // refresh display tab
      nameField.setText(node.toString());
      nameField.setBackground(Color.white);
      descriptionPane.setText(node.description);
      descriptionPane.setBackground(Color.white);
      urlField.setText(node.urlName);
      urlField.setBackground(
          node.url == null && node.urlName != null?
          RED : Color.white);
      JEditorPane html = getSelectedTab().textPane;
      if (node.url != null) {
        // set page
        try {
          if (node.url.getContent() != null)
            html.setPage(node.url);
        }
        catch (Exception ex) {
          html.setText(null);
        }
      }
      else {
        getSelectedTab().textPane.setContentType(LaunchPanel.defaultType);
        getSelectedTab().textPane.setText(null);
      }
      // refresh launch tab
      // check the path and update the class chooser
      String path = node.getClassPath(); // in node-to-root order
      if (!path.equals(previousPath)) {
        boolean success = getClassChooser().setPath(path);
        searchJarAction.setEnabled(success);
      }
      // store path for later comparison
      previousPath = node.getClassPath();
      jarField.setText(node.classPath);
      jarField.setBackground(
          node.classPath != null && !getClassChooser().isLoaded(node.classPath)?
          RED : Color.white);
      classField.setText(node.launchClassName);
      classField.setBackground(
          node.getLaunchClass() == null && node.launchClassName != null?
          RED : Color.white);
      int n = ((Integer)argSpinner.getValue()).intValue();
      if (node.args.length > n)
//      node.setMinimumArgLength(n+1);
        argField.setText(node.args[n]);
      else argField.setText("");
      argField.setBackground(Color.white);
      // set enabled and selected states of checkboxes
      LaunchNode parent = (LaunchNode)node.getParent(); // may be null
      singletonCheckBox.setEnabled(parent == null || !parent.isSingleton());
      singletonCheckBox.setSelected(node.isSingleton());
      singleVMCheckBox.setEnabled(parent == null || !parent.isSingleVM());
      singleVMCheckBox.setSelected(node.isSingleVM());
      if (node.isSingleVM()) {
        showLogCheckBox.setEnabled(parent == null || !parent.isShowLog());
        showLogCheckBox.setSelected(node.isShowLog());
        singleAppCheckBox.setEnabled(parent == null || !parent.isSingleApp());
        singleAppCheckBox.setSelected(node.isSingleApp());
      }
      else {
        showLogCheckBox.setEnabled(false);
        showLogCheckBox.setSelected(false);
        singleAppCheckBox.setEnabled(false);
        singleAppCheckBox.setSelected(false);
      }
      // refresh the level dropdown if visible
      levelDropDown.setVisible(node.isShowLog());
      if (levelDropDown.isVisible()) {
        boolean useAll = (parent == null || !parent.isShowLog());
        // disable during refresh to prevent triggering events
        levelDropDown.setEnabled(false);
        levelDropDown.removeAllItems();
        for (int i = 0; i < allLevels.length; i++) {
          if (useAll ||
              allLevels[i].intValue() <= parent.getLogLevel().intValue()) {
            levelDropDown.addItem(allLevels[i]);
          }
        }
        levelDropDown.setSelectedItem(node.getLogLevel());
        levelDropDown.setEnabled(true);
      }
      // refresh metadata tab
      authorField.setText(node.author);
      authorField.setBackground(Color.white);
      codeAuthorField.setText(node.codeAuthor);
      codeAuthorField.setBackground(Color.white);
      commentPane.setText(node.comment);
      commentPane.setBackground(Color.white);
    }
    // rebuild file menu
    fileMenu.removeAll();
    fileMenu.add(newItem);
    fileMenu.addSeparator();
    fileMenu.add(openItem);
    LaunchPanel tab = getSelectedTab();
    if (tab != null) {
      String name = getDisplayName(getRootNode().fileName);
      saveItem.setEnabled(getRootNode().isFileWritable);
      saveItem.setText(LaunchRes.getString("Menu.File.Save")
                       + " \"" + name + "\"");
      fileMenu.add(importItem);
      fileMenu.addSeparator();
      fileMenu.add(closeTabItem);
      fileMenu.add(closeAllItem);
      fileMenu.addSeparator();
      fileMenu.add(previewItem);
      fileMenu.addSeparator();
      String saveall = LaunchRes.getString("Menu.File.SaveAll");
      if (!isAllWritable()) saveall += "...";
      saveAllItem.setText(saveall);
      fileMenu.add(saveAllItem);
      fileMenu.addSeparator();
      fileMenu.add(saveItem);
      fileMenu.add(saveAsItem);
      fileMenu.add(saveSetAsItem);
      // set tab properties
      frame.getContentPane().add(toolbar, BorderLayout.NORTH);
      tab.dataPanel.add(editorTabs, BorderLayout.CENTER);
      tab.textScroller.setBorder(BorderFactory.createTitledBorder(
          LaunchRes.getString("Label.HTML")));
      displaySplitPane.setTopComponent(tab.textScroller);
      displaySplitPane.setDividerLocation(0.7);
      // hidden root
      if (getRootNode().getChildCount() == 0) {
        getRootNode().hiddenWhenRoot = false;
        hideRootCheckBox.setEnabled(false);
      }
      else hideRootCheckBox.setEnabled(true);
      boolean rootVisible = !getRootNode().hiddenWhenRoot;
      hideRootCheckBox.setSelected(!rootVisible);
      tab.tree.setRootVisible(rootVisible);
      if (getSelectedNode() == null && !rootVisible) {
        tab.setSelectedNode((LaunchNode)getRootNode().getChildAt(0));
      }
      // editor enabled
      editorEnabledCheckBox.setSelected(editorEnabled);
    }
    else { // no tab
      frame.getContentPane().remove(toolbar);
    }
    fileMenu.addSeparator();
    fileMenu.add(exitItem);
  }

  /**
   * Creates the GUI.
   */
  protected void createGUI() {
    wInit = 600; hInit = 540;
    ArrayList labels = new ArrayList();
    super.createGUI();
    // add window listener to refresh GUI
    tabbedPane.addComponentListener(new ComponentAdapter() {
      public void componentResized(ComponentEvent e) {
        refreshGUI();
      }
    });
    // create file icon
    String imageFile="/org/opensourcephysics/resources/tools/images/filenode.gif";
    fileIcon = new ImageIcon(Launcher.class.getResource(imageFile));
    // create actions
    createActions();
    // create fields
    nameField = new JTextField();
    nameField.addKeyListener(keyListener);
    nameField.addFocusListener(focusListener);
    classField = new JTextField();
    classField.addKeyListener(keyListener);
    classField.addFocusListener(focusListener);
    argField = new JTextField();
    argField.addKeyListener(keyListener);
    argField.addFocusListener(focusListener);
    jarField = new JTextField();
    jarField.addKeyListener(keyListener);
    jarField.addFocusListener(focusListener);
    urlField = new JTextField();
    urlField.addKeyListener(keyListener);
    urlField.addFocusListener(focusListener);
    codeAuthorField = new JTextField();
    codeAuthorField.addKeyListener(keyListener);
    codeAuthorField.addFocusListener(focusListener);
    authorField = new JTextField();
    authorField.addKeyListener(keyListener);
    authorField.addFocusListener(focusListener);
    commentPane = new JTextPane();
    commentPane.addKeyListener(keyListener);
    commentPane.addFocusListener(focusListener);
    commentScroller = new JScrollPane(commentPane);
    commentScroller.setBorder(
        BorderFactory.createTitledBorder(LaunchRes.getString("Label.Comments")));
    descriptionPane = new JTextPane();
    descriptionPane.addKeyListener(keyListener);
    descriptionPane.addFocusListener(focusListener);
    descriptionScroller = new JScrollPane(descriptionPane);
    descriptionScroller.setBorder(
        BorderFactory.createTitledBorder(LaunchRes.getString("Label.Description")));
    displaySplitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
    displaySplitPane.setEnabled(false);
    displaySplitPane.setBottomComponent(descriptionScroller);
    // assemble display panel
    JPanel displayPanel = new JPanel(new BorderLayout());
    JToolBar nameBar = new JToolBar();
    nameBar.setFloatable(false);
    JLabel label = new JLabel(LaunchRes.getString("Label.Name"));
    labels.add(label);
    nameBar.add(label);
    nameBar.add(nameField);
    displayPanel.add(nameBar, BorderLayout.NORTH);
    urlPanel = new JPanel(new BorderLayout());
    displayPanel.add(urlPanel, BorderLayout.CENTER);
    JToolBar urlBar = new JToolBar();
    urlBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.URL"));
    labels.add(label);
    urlBar.add(label);
    urlBar.add(urlField);
    urlBar.add(openURLAction);
    urlPanel.add(urlBar, BorderLayout.NORTH);
    urlPanel.add(displaySplitPane, BorderLayout.CENTER);
    // determine label size
    FontRenderContext frc
        = new FontRenderContext(null,   // no AffineTransform
                                false,  // no antialiasing
                                false); // no fractional metrics
    Font font = label.getFont();
    // determine display panel label size
    int w = 0;
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      Rectangle2D rect = font.getStringBounds(next.getText() + " ", frc);
      w = Math.max(w, (int)rect.getWidth() + 1);
    }
    Dimension labelSize = new Dimension(w, 20);
    // set label properties
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      next.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 2));
      next.setPreferredSize(labelSize);
      next.setHorizontalAlignment(SwingConstants.TRAILING);
    }
    labels.clear();
    // assemble launch panel
    JPanel launchPanel = new JPanel(new BorderLayout());
    JToolBar pathBar = new JToolBar();
    pathBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.Jar"));
    labels.add(label);
    pathBar.add(label);
    pathBar.add(jarField);
    pathBar.add(openJarAction);
    launchPanel.add(pathBar, BorderLayout.NORTH);
    JPanel classPanel = new JPanel(new BorderLayout());
    launchPanel.add(classPanel, BorderLayout.CENTER);
    JToolBar classBar = new JToolBar();
    classBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.Class"));
    labels.add(label);
    classBar.add(label);
    classBar.add(classField);
    classBar.add(searchJarAction);
    classPanel.add(classBar, BorderLayout.NORTH);
    JPanel argPanel = new JPanel(new BorderLayout());
    classPanel.add(argPanel, BorderLayout.CENTER);
    JToolBar argBar = new JToolBar();
    argBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.Args"));
    labels.add(label);
    argBar.add(label);
    SpinnerModel model = new SpinnerNumberModel(0, 0, maxArgs-1, 1);
    argSpinner = new JSpinner(model);
    JSpinner.NumberEditor editor = new JSpinner.NumberEditor(argSpinner);
    argSpinner.setEditor(editor);
    argSpinner.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
        if (newNodeSelected) return;
        if (argField.getBackground() == YELLOW)
          refreshSelectedNode();
        else refreshGUI();
       }
    });
    argBar.add(argSpinner);
    argBar.add(argField);
    argBar.add(openArgAction);
    argPanel.add(argBar, BorderLayout.NORTH);
    JPanel optionsPanel = new JPanel(new BorderLayout());
    argPanel.add(optionsPanel, BorderLayout.CENTER);
    JToolBar optionsBar = new JToolBar();
    optionsBar.setFloatable(false);
    // determine launch panel label size
    w = 0;
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      Rectangle2D rect = font.getStringBounds(next.getText() + " ", frc);
      w = Math.max(w, (int)rect.getWidth() + 1);
    }
    labelSize = new Dimension(w, 20);
    // set label properties
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      next.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 2));
      next.setPreferredSize(labelSize);
      next.setHorizontalAlignment(SwingConstants.TRAILING);
    }
    labels.clear();
    singleVMCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.SingleVM"));
    singleVMCheckBox.addActionListener(changeAction);
    singleVMCheckBox.setContentAreaFilled(false);
    singleVMCheckBox.setAlignmentX(Component.LEFT_ALIGNMENT);
    showLogCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.ShowLog"));
    showLogCheckBox.addActionListener(changeAction);
    showLogCheckBox.setContentAreaFilled(false);
    showLogCheckBox.setAlignmentX(Component.LEFT_ALIGNMENT);
    singletonCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.Singleton"));
    singletonCheckBox.addActionListener(changeAction);
    singletonCheckBox.setContentAreaFilled(false);
    singletonCheckBox.setAlignmentX(Component.LEFT_ALIGNMENT);
    singleAppCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.SingleApp"));
    singleAppCheckBox.addActionListener(changeAction);
    singleAppCheckBox.setContentAreaFilled(false);
    singleAppCheckBox.setAlignmentX(Component.LEFT_ALIGNMENT);
    allLevels = new Level[] {Level.OFF, Level.SEVERE, Level.WARNING,
        Level.INFO, ConsoleLevel.ERR_CONSOLE, ConsoleLevel.OUT_CONSOLE,
        Level.CONFIG, Level.FINE, Level.FINER, Level.FINEST, Level.ALL};
    levelDropDown = new JComboBox(allLevels);
    levelDropDown.setMaximumSize(levelDropDown.getMinimumSize());
    levelDropDown.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (levelDropDown.isEnabled()) {
          LaunchNode node = getSelectedNode();
          if (node != null) {
            node.setLogLevel( (Level) levelDropDown.getSelectedItem());
          }
        }
      }
    });
    Box checkBoxPanel = Box.createVerticalBox();
    JToolBar bar = new JToolBar();
    bar.setFloatable(false);
    bar.setAlignmentX(Component.LEFT_ALIGNMENT);
    bar.add(singletonCheckBox);
    checkBoxPanel.add(bar);
    bar = new JToolBar();
    bar.setFloatable(false);
    bar.setAlignmentX(Component.LEFT_ALIGNMENT);
    bar.add(singleVMCheckBox);
    checkBoxPanel.add(bar);
    bar = new JToolBar();
    bar.setFloatable(false);
    bar.setAlignmentX(Component.LEFT_ALIGNMENT);
    bar.add(singleAppCheckBox);
    checkBoxPanel.add(bar);
    bar = new JToolBar();
    bar.setFloatable(false);
    bar.add(showLogCheckBox);
    bar.add(Box.createHorizontalStrut(4));
    bar.add(levelDropDown);
    bar.add(Box.createHorizontalGlue());
    bar.setAlignmentX(Component.LEFT_ALIGNMENT);
    checkBoxPanel.add(bar);
    Border titled = BorderFactory.createTitledBorder(LaunchRes.getString("Label.Options"));
    Border recess = BorderFactory.createLoweredBevelBorder();
    optionsBar.setBorder(BorderFactory.createCompoundBorder(recess, titled));
    optionsBar.add(checkBoxPanel);
    optionsPanel.add(optionsBar, BorderLayout.NORTH);
    // create metadata panel
    JPanel metadataPanel = new JPanel(new BorderLayout());
    JToolBar authorBar = new JToolBar();
    authorBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.Author"));
    labels.add(label);
    authorBar.add(label);
    authorBar.add(authorField);
    metadataPanel.add(authorBar, BorderLayout.NORTH);
    JPanel codePanel = new JPanel(new BorderLayout());
    metadataPanel.add(codePanel, BorderLayout.CENTER);
    JToolBar codeBar = new JToolBar();
    codeBar.setFloatable(false);
    label = new JLabel(LaunchRes.getString("Label.CodeAuthor"));
    labels.add(label);
    codeBar.add(label);
    codeBar.add(codeAuthorField);
    codePanel.add(codeBar, BorderLayout.NORTH);
    codePanel.add(commentScroller, BorderLayout.CENTER);
    // determine metadata panel label size
    w = 0;
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      Rectangle2D rect = font.getStringBounds(next.getText() + " ", frc);
      w = Math.max(w, (int)rect.getWidth() + 1);
    }
    labelSize = new Dimension(w, 20);
    // set label properties
    for (Iterator it = labels.iterator(); it.hasNext();) {
      JLabel next = (JLabel)it.next();
      next.setBorder(BorderFactory.createEmptyBorder(0, 0, 0, 2));
      next.setPreferredSize(labelSize);
      next.setHorizontalAlignment(SwingConstants.TRAILING);
    }
    // create server panel
    JPanel serverPanel = new JPanel(new BorderLayout());
    // create editor tabbed pane
    editorTabs = new JTabbedPane(JTabbedPane.TOP);
    editorTabs.addTab(LaunchRes.getString("Tab.Display"), displayPanel);
    editorTabs.addTab(LaunchRes.getString("Tab.Launch"), launchPanel);
    editorTabs.addTab(LaunchRes.getString("Tab.Metadata"), metadataPanel);
//    editorTabs.addTab(LaunchRes.getString("Tab.Server"), serverPanel);
    // create toolbar
    toolbar = new JToolBar();
    toolbar.setFloatable(false);
    toolbar.setRollover(true);
    toolbar.setBorder(BorderFactory.createLineBorder(Color.gray));
    frame.getContentPane().add(toolbar, BorderLayout.NORTH);
    JButton button = new JButton(addAction);
    toolbar.add(button);
//    button = new JButton(removeAction);
//    toolbar.add(button);
    button = new JButton(cutAction);
    toolbar.add(button);
    button = new JButton(copyAction);
    toolbar.add(button);
    button = new JButton(pasteAction);
    toolbar.add(button);
    button = new JButton(moveUpAction);
    toolbar.add(button);
    button = new JButton(moveDownAction);
    toolbar.add(button);
    hideRootCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.HideRoot"));
    hideRootCheckBox.addActionListener(changeAction);
    hideRootCheckBox.setContentAreaFilled(false);
    toolbar.add(hideRootCheckBox);
    editorEnabledCheckBox = new JCheckBox(LaunchRes.getString("Checkbox.EditorEnabled"));
    editorEnabledCheckBox.addActionListener(changeAction);
    editorEnabledCheckBox.setContentAreaFilled(false);
    toolbar.add(editorEnabledCheckBox);
    // create menu items
    int mask = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    // create new item
    newItem = new JMenuItem(newTreeAction);
    newItem.setAccelerator(KeyStroke.getKeyStroke('N', mask));
    // create preview item
    previewItem = new JMenuItem(LaunchRes.getString("Menu.File.Preview"));
    previewItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedTab().getRootNode();
        if (node != null) {
          LaunchSet set = new LaunchSet(LaunchBuilder.this, tabSetPath);
          XMLControl control = new XMLControlElement(set);
          Launcher launcher = new Launcher(control.toXML());
          launcher.frame.setVisible(true);
          launcher.frame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
        }
      }
    });
    // create other menu items
    importItem = new JMenuItem(importAction);
    saveItem = new JMenuItem(LaunchRes.getString("Menu.File.Save"));
    saveItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedTab().getRootNode();
        if (save(node, node.fileName) != null)
          tabbedPane.setTitleAt(tabbedPane.getSelectedIndex(),
                                getDisplayName(node.fileName));
        refreshGUI();
     }
    });
    saveAsItem = new JMenuItem(LaunchRes.getString("Menu.File.SaveAs"));
    saveAsItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedTab().getRootNode();
        if (saveAs(node) != null)
          tabbedPane.setTitleAt(tabbedPane.getSelectedIndex(),
                                getDisplayName(node.fileName));
        refreshGUI();
     }
    });
    saveAllItem = new JMenuItem(LaunchRes.getString("Menu.File.SaveAll"));
    saveAllItem.setAccelerator(KeyStroke.getKeyStroke('S', mask));
    saveAllItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        saveAllTabs();
        if (isTabSetWritable) saveTabSet(tabSetPath);
        else saveTabSetAs();
        refreshGUI();
      }
    });
    saveSetAsItem = new JMenuItem(LaunchRes.getString("Menu.File.SaveSetAs"));
    saveSetAsItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        saveTabSetAs();
        refreshGUI();
      }
    });
    // replace tab listener
    tabbedPane.removeMouseListener(tabListener);
    tabListener = new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        if (contentPane.getTopLevelAncestor() != frame) return;
        if (e.isPopupTrigger() || e.getButton() == MouseEvent.BUTTON3 ||
           (e.isControlDown() && System.getProperty("os.name", "").indexOf("Mac") > -1)) {
          // make popup and add items
          JPopupMenu popup = new JPopupMenu();
          JMenuItem item = new JMenuItem(LaunchRes.getString("MenuItem.Close"));
          item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              removeSelectedTab();
            }
          });
          popup.add(item);
          popup.addSeparator();
          item = new JMenuItem(LaunchRes.getString("Menu.File.SaveAs"));
          item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              LaunchNode node = getSelectedTab().getRootNode();
              if (saveAs(node) != null)
                  tabbedPane.setTitleAt(tabbedPane.getSelectedIndex(),
                                        getDisplayName(node.fileName));
              refreshGUI();
            }
          });
          popup.add(item);
          final int i = tabbedPane.getSelectedIndex();
          if (i > 0 || i < tabbedPane.getTabCount()-1)
            popup.addSeparator();
          if (i < tabbedPane.getTabCount()-1) {
            item = new JMenuItem(LaunchRes.getString("Popup.MenuItem.MoveUp"));
            item.addActionListener(new ActionListener() {
              public void actionPerformed(ActionEvent e) {
                LaunchPanel tab = getSelectedTab();
                LaunchNode root = tab.getRootNode();
                removeSelectedTab();
                tabbedPane.insertTab(getDisplayName(root.fileName),
                                     null, tab, null, i+1);
                tabbedPane.setSelectedComponent(tab);

              }
            });
            popup.add(item);
          }
          if (i > 0) {
            item = new JMenuItem(LaunchRes.getString("Popup.MenuItem.MoveDown"));
            item.addActionListener(new ActionListener() {
              public void actionPerformed(ActionEvent e) {
                LaunchPanel tab = getSelectedTab();
                LaunchNode root = tab.getRootNode();
                removeSelectedTab();
                tabbedPane.insertTab(getDisplayName(root.fileName),
                                     null, tab, null, i-1);
                tabbedPane.setSelectedComponent(tab);

              }
            });
            popup.add(item);
          }
          popup.show(tabbedPane, e.getX(), e.getY() + 8);
        }
      }
    };
    tabbedPane.addMouseListener(tabListener);
    frame.pack();
  }

  /**
   * Creates the actions.
   */
  protected void createActions() {
    String imageFile="/org/opensourcephysics/resources/tools/images/open.gif";
    Icon openIcon = new ImageIcon(Launcher.class.getResource(imageFile));
    openJarAction = new AbstractAction(null, openIcon) {
      public void actionPerformed(ActionEvent e) {
        JFileChooser chooser = getJARChooser();
        int result = chooser.showOpenDialog(null);
        if (result == JFileChooser.APPROVE_OPTION) {
          File file = chooser.getSelectedFile();
          jarField.setText(XML.getRelativePath(file.getPath()));
          searchJarAction.setEnabled(true);
          refreshSelectedNode();
        }
      }
    };
    searchJarAction = new AbstractAction(null, openIcon) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedNode();
        if (node != null && getClassChooser().chooseClassFor(node)) {
          changedFiles.add(node.getOwner().fileName);
          refreshClones(node);
          refreshGUI();
        }
      }
    };
    searchJarAction.setEnabled(false);
    openArgAction = new AbstractAction(null, openIcon) {
      public void actionPerformed(ActionEvent e) {
        JFileChooser chooser = getFileChooser();
        int result = chooser.showOpenDialog(null);
        if (result == JFileChooser.APPROVE_OPTION) {
          File file = chooser.getSelectedFile();
          argField.setText(XML.getRelativePath(file.getPath()));
          refreshSelectedNode();
        }
      }
    };
    openURLAction = new AbstractAction(null, openIcon) {
      public void actionPerformed(ActionEvent e) {
        JFileChooser chooser = getHTMLChooser();
        int result = chooser.showOpenDialog(null);
        if (result == JFileChooser.APPROVE_OPTION) {
          File file = chooser.getSelectedFile();
          urlField.setText(XML.getRelativePath(file.getPath()));
          refreshSelectedNode();
        }
      }
    };
    openTabAction = new AbstractAction(null, openIcon) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedNode();
        String tabName = getDisplayName(node.fileName);
        for (int i = 0; i < tabbedPane.getComponentCount(); i++) {
          if (tabbedPane.getTitleAt(i).equals(tabName)) {
            tabbedPane.setSelectedIndex(i);
            return;
          }
        }
        XMLControl control = new XMLControlElement(node);
        XMLControl cloneControl = new XMLControlElement(control);
        LaunchNode clone = (LaunchNode) cloneControl.loadObject(null);
        clone.fileName = node.fileName;
        clone.isFileWritable = node.isFileWritable;
        addTab(clone);
      }
    };
    changeAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        refreshSelectedNode();
      }
    };
    addAction = new AbstractAction(LaunchRes.getString("Action.Add")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode newNode = new LaunchNode(LaunchRes.getString("NewNode.Name"));
        addChildToSelectedNode(newNode);
      }
    };
    removeAction = new AbstractAction(LaunchRes.getString("Action.Remove")) {
      public void actionPerformed(ActionEvent e) {
        removeSelectedNode();
      }
    };
    newTreeAction = new AbstractAction(LaunchRes.getString("Action.New")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode root = new LaunchNode(LaunchRes.getString("NewTree.Name"));
        addTab(root);
      }
    };
    cutAction = new AbstractAction(LaunchRes.getString("Action.Cut")) {
      public void actionPerformed(ActionEvent e) {
        copyAction.actionPerformed(null);
        removeSelectedNode();
      }
    };
    copyAction = new AbstractAction(LaunchRes.getString("Action.Copy")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedNode();
        if (node != null) {
          XMLControl control = new XMLControlElement(node);
          control.setValue("filename", node.fileName);
          control.setValue("writable", node.isFileWritable);
          StringSelection data = new StringSelection(control.toXML());
          Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
          clipboard.setContents(data, data);
        }
      }
    };
    pasteAction = new AbstractAction(LaunchRes.getString("Action.Paste")) {
      public void actionPerformed(ActionEvent e) {
        try {
          Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
          Transferable data = clipboard.getContents(null);
          String dataString = (String)data.getTransferData(DataFlavor.stringFlavor);
          if (dataString != null) {
            XMLControlElement control = new XMLControlElement();
            control.readXML(dataString);
            if (control.getObjectClass() == LaunchNode.class) {
              String fileName = control.getString("filename");
              LaunchNode newNode = (LaunchNode) control.loadObject(null);
              newNode.fileName = fileName;
              newNode.isFileWritable = control.getBoolean("writable");
              addChildToSelectedNode(newNode);
            }
          }
        }
        catch (UnsupportedFlavorException ex) {}
        catch (IOException ex) {}
        catch (HeadlessException ex) {}
      }
    };
    importAction = new AbstractAction(LaunchRes.getString("Action.Import")) {
      public void actionPerformed(ActionEvent e) {
        int result = getXMLChooser().showOpenDialog(null);
        if (result == JFileChooser.APPROVE_OPTION) {
          // open the file in an xml control
          String fileName = getXMLChooser().getSelectedFile().getAbsolutePath();
          XMLControlElement element = new XMLControlElement(fileName);
          if (element.getObjectClass() == LaunchNode.class) {
            // add the child node
            LaunchNode child = (LaunchNode)element.loadObject(null);
            child.fileName = XML.getRelativePath(fileName);
            child.isFileWritable = element.canWrite;
            addChildToSelectedNode(child);
          }
        }
      }
    };
    saveAsAction = new AbstractAction(LaunchRes.getString("Action.SaveNodeAs")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode parent = (LaunchNode)getSelectedNode().getParent();
        String fileName = saveAs(getSelectedNode());
        if (fileName != null) {
          saveAsAction.putValue("saved", "true");
          changedFiles.add(parent.getOwner().fileName);
          refreshClones(parent);
        }
        else saveAsAction.putValue("saved", "false");
      }
    };
    moveUpAction = new AbstractAction(LaunchRes.getString("Action.Up")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedNode();
        if (node == null) return;
        LaunchNode parent = (LaunchNode)node.getParent();
        if (parent == null) return;
        int i = parent.getIndex(node);
        if (i > 0) {
          getSelectedTab().treeModel.removeNodeFromParent(node);
          getSelectedTab().treeModel.insertNodeInto(node, parent, i - 1);
          getSelectedTab().setSelectedNode(node);
        }
      }
    };
    moveDownAction = new AbstractAction(LaunchRes.getString("Action.Down")) {
      public void actionPerformed(ActionEvent e) {
        LaunchNode node = getSelectedNode();
        if (node == null) return;
        LaunchNode parent = (LaunchNode)node.getParent();
        if (parent == null) return;
        int i = parent.getIndex(node);
        int end = parent.getChildCount();
        if (i < end - 1) {
          getSelectedTab().treeModel.removeNodeFromParent(node);
          getSelectedTab().treeModel.insertNodeInto(node, parent, i + 1);
          getSelectedTab().setSelectedNode(node);
        }
      }
    };
    // create focus listener
    focusListener = new FocusAdapter() {
      public void focusLost(FocusEvent e) {
        refreshSelectedNode();
        refreshGUI();
      }
    };
    // create key listener
    keyListener = new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
        JComponent comp = (JComponent)e.getSource();
        if (e.getKeyCode() == KeyEvent.VK_ENTER &&
           ((comp != descriptionPane && comp != commentPane) ||
            e.isControlDown() || e.isShiftDown())) {
         refreshSelectedNode();
         refreshGUI();
        }
        else {
          comp.setBackground(YELLOW);
        }
      }
    };
  }

  /**
   * Removes the selected node.
   */
  protected void removeSelectedNode() {
    LaunchNode node = getSelectedNode();
    if (node == null || node.getParent() == null) {
      Toolkit.getDefaultToolkit().beep();
      return;
    }
    LaunchNode parent = (LaunchNode)node.getParent();
    getSelectedTab().treeModel.removeNodeFromParent(node);
    getSelectedTab().setSelectedNode(parent);
    changedFiles.add(parent.getOwner().fileName);
    refreshClones(parent);
  }

  /**
   * Adds a child node to the selected node.
   *
   * @param child the child node to add
   */
  protected void addChildToSelectedNode(LaunchNode child) {
    LaunchNode parent = getSelectedNode();
    if (parent != null && child != null) {
      LaunchNode[] nodes = child.getOwnedNodes();
      for (int i = 0; i < nodes.length; i++) {
        LaunchNode node = getSelectedTab().getClone(nodes[i]);
        if (node != null) {
          getSelectedTab().setSelectedNode(node);
          JOptionPane.showMessageDialog(
              frame,
              LaunchRes.getString("Dialog.DuplicateNode.Message") +
              " \"" + node + "\"",
              LaunchRes.getString("Dialog.DuplicateNode.Title"),
              JOptionPane.WARNING_MESSAGE);
          return;
        }
      }
      getSelectedTab().treeModel.insertNodeInto(child, parent,
                                                 parent.getChildCount());
      getSelectedTab().tree.scrollPathToVisible(new TreePath(child.getPath()));
      child.setLaunchClass(child.launchClassName);
      changedFiles.add(parent.getOwner().fileName);
      refreshClones(parent);
    }
  }

  /**
   * Replaces clones of a specified node with new clones.
   *
   * @param node the current version of the node to clone
   */
  protected void refreshClones(LaunchNode node) {
    // find clones
    Map clones = getClones(node);
    if (clones.isEmpty()) return;
    // replace clones
    XMLControl control = new XMLControlElement(node.getOwner());
    Iterator it = clones.keySet().iterator();
    while (it.hasNext()) {
      LaunchPanel tab = (LaunchPanel)it.next();
      LaunchNode clone = (LaunchNode)clones.get(tab);
      LaunchNode parent = (LaunchNode)clone.getParent();
      boolean expanded = tab.tree.isExpanded(new TreePath(clone.getPath()));
      if (parent != null) {
        int index = parent.getIndex(clone);
        tab.treeModel.removeNodeFromParent(clone);
        clone = (LaunchNode)new XMLControlElement(control).loadObject(null);
        clone.fileName = node.fileName;
        clone.isFileWritable = node.isFileWritable;
        tab.treeModel.insertNodeInto(clone, parent, index);
      }
      else {
        clone = (LaunchNode)new XMLControlElement(control).loadObject(null);
        clone.fileName = node.fileName;
        clone.isFileWritable = node.isFileWritable;
        tab.treeModel.setRoot(clone);
      }
      if (expanded) {
        tab.tree.expandPath(new TreePath(clone.getPath()));
      }
    }
  }

  /**
   * Returns clones containing a specified node in a tab-to-node map.
   *
   * @param node the node
   * @return the tab-to-node map
   */
  protected Map getClones(LaunchNode node) {
    Map clones = new HashMap();
    // find clones
    node = node.getOwner();
    if (node == null) return clones;
    Component[] tabs = tabbedPane.getComponents();
    for (int i = 0; i < tabs.length; i++) {
      LaunchPanel tab = (LaunchPanel)tabs[i];
      LaunchNode clone = tab.getClone(node);
      if (clone != null && clone != node) {
        clones.put(tab, clone);
      }
    }
    return clones;
  }

  /**
   * Shows the about dialog.
   */
  protected void showAboutDialog() {
    String aboutString = "Launch Builder 1.0  Aug 2004\n" +
                         "Open Source Physics Project\n" +
                         "www.opensourcephysics.org";
    JOptionPane.showMessageDialog(
        frame,
        aboutString,
        LaunchRes.getString("Help.About.Title") + " LaunchBuilder",
        JOptionPane.INFORMATION_MESSAGE);
  }

  /**
   * Gets a file chooser for selecting jar files.
   *
   * @return the jar chooser
   */
  protected static JFileChooser getJARChooser() {
    getFileChooser().setFileFilter(jarFileFilter);
    return fileChooser;
  }

  /**
   * Gets a file chooser for selecting html files.
   *
   * @return the html chooser
   */
  protected static JFileChooser getHTMLChooser() {
    getFileChooser().setFileFilter(htmlFileFilter);
    return fileChooser;
  }

  /**
   * Gets a file chooser.
   *
   * @return the file chooser
   */
  protected static JFileChooser getFileChooser() {
    if (fileChooser == null) {
      fileChooser = new JFileChooser(new File("." + File.separator));
      allFileFilter = fileChooser.getFileFilter();
      // create jar file filter
      jarFileFilter = new javax.swing.filechooser.FileFilter() {
        // accept all directories and *.jar files.
        public boolean accept(File f) {
          if (f == null)
            return false;
          if (f.isDirectory())
            return true;
          String extension = null;
          String name = f.getName();
          int i = name.lastIndexOf('.');
          if (i > 0 && i < name.length() - 1) {
            extension = name.substring(i + 1).toLowerCase();
          }
          if (extension != null && extension.equals("jar")) {
            return true;
          }
          return false;
        }

        // the description of this filter
        public String getDescription() {
          return LaunchRes.getString("FileChooser.JarFilter.Description");
        }
      };
      // create html file filter
      htmlFileFilter = new javax.swing.filechooser.FileFilter() {
        // accept all directories, *.htm and *.html files.
        public boolean accept(File f) {
          if (f == null)
            return false;
          if (f.isDirectory())
            return true;
          String extension = null;
          String name = f.getName();
          int i = name.lastIndexOf('.');
          if (i > 0 && i < name.length() - 1) {
            extension = name.substring(i + 1).toLowerCase();
          }
          if (extension != null &&
              (extension.equals("htm") || extension.equals("html"))) {
            return true;
          }
          return false;
        }

        // the description of this filter
        public String getDescription() {
          return LaunchRes.getString("FileChooser.HTMLFilter.Description");
        }
      };
    }
    fileChooser.removeChoosableFileFilter(jarFileFilter);
    fileChooser.removeChoosableFileFilter(htmlFileFilter);
    fileChooser.setFileFilter(allFileFilter);
    return fileChooser;
  }

  /**
   * Handles a mouse pressed event.
   *
   * @param e the mouse event
   * @param tab the launch panel triggering the event
   */
  protected void handleMousePressed(MouseEvent e, final LaunchPanel tab) {
    super.handleMousePressed(e, tab);
    if (e.isPopupTrigger() || e.getButton() == MouseEvent.BUTTON3 ||
       (e.isControlDown() && System.getProperty("os.name", "").indexOf("Mac") > -1)) {
      TreePath path = tab.tree.getPathForLocation(e.getX(), e.getY());
      if (path == null) return;
      final LaunchNode node = getSelectedNode();
      if (node == null) return;
      // add items to popup
      if (node != getRootNode()) {
        if (popup.getComponentCount() != 0) popup.addSeparator();
        if (node.fileName == null) {
          JMenuItem item = new JMenuItem(LaunchRes.getString(
              "Popup.MenuItem.SaveAs"));
          popup.add(item);
          item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              saveAsAction.actionPerformed(e);
              if (saveAsAction.getValue("saved").equals("true"))
                openTabAction.actionPerformed(e);
            }
          });
        }
        else {
          JMenuItem item = new JMenuItem(LaunchRes.getString(
              "Popup.MenuItem.Open"));
          popup.add(item);
          item.addActionListener(openTabAction);
        }
      }
      popup.show(tab, e.getX() + 4, e.getY() + 12);
    }
  }

  /**
   * Overrides Launcher exit method.
   */
  protected void exit() {
    if (!saveAllChanges()) return;
    super.exit();
  }

  // for testing purposes
  class treeModelListener implements TreeModelListener {
    public void treeNodesChanged(TreeModelEvent e) {
      System.out.println("changed: " + e);
      LaunchNode node = (LaunchNode)e.getTreePath().getLastPathComponent();
      if (e.getChildren() != null) {
        node = (LaunchNode)e.getChildren()[0];
      }
    }

    public void treeNodesInserted(TreeModelEvent e) {
      System.out.println("inserted: " + e);
    }

    public void treeNodesRemoved(TreeModelEvent e) {
      System.out.println("removed: " + e);
    }

    public void treeStructureChanged(TreeModelEvent e) {
    }
  }

  protected boolean isAllWritable() {
    if (!isTabSetWritable) return false;
    Component[] tabs = tabbedPane.getComponents();
    for (int i = 0; i < tabs.length; i++) {
      LaunchPanel tab = (LaunchPanel) tabs[i];
      LaunchNode[] nodes = tab.getRootNode().getOwnedNodes();
      for (int j = 0; j < nodes.length; j++) {
        if (!nodes[j].isFileWritable) return false;
      }
    }
    return true;
  }
}
