/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.text.*;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;

import org.opensourcephysics.display.*;
import org.opensourcephysics.numerics.*;
import org.opensourcephysics.controls.*;

/**
 * This is a panel that displays and analyzes a dataset.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class DatasetToolPanel extends JPanel {

  // static fields
  protected static OSPLog log = OSPLog.getOSPLog();

  // instance fields
  protected Dataset dataset;
  protected JTabbedPane tabbedPane;
  protected PlottingPanel plot;
  protected DataTable table;
  protected JTable statsTable;
  protected StatsTableModel statsTableModel;
  protected String[] shapeNames;
  protected int[] shapeNumbers;
  protected String[] markerSizes;
  protected JButton shapeButton;
  protected JButton sizeButton;
  protected JButton connectedButton;
  protected JButton fillButton;
  protected JButton edgeButton;
  protected JButton lineButton;
  protected JButton colorsButton;
  protected JButton fitLineButton;
  protected JButton fitPolyButton;
  protected Action fitPolyAction;
  protected Map polyFits = new HashMap(); // polynomial to drawer
  protected Color[] polyColors = new Color[] {Color.cyan, Color.magenta,
                                              Color.red, Color.green};
  protected int buttonHeight = 24;
  protected Object[][] data;
  protected String[] headings;

  /**
   * Constructs a DataToolPanel for the specified dataset.
   *
   * @param dataset the dataset
   */
  public DatasetToolPanel(Dataset dataset) {
    this.dataset = dataset;
    createGUI();
  }

  /**
   * Gets the dataset.
   *
   * @return the dataset
   */
  protected Dataset getDataset() {
    return dataset;
  }

//_______________________ protected & private methods __________________________

  /**
   * Creates the GUI.
   */
  protected void createGUI() {
    setLayout(new BorderLayout());
    tabbedPane = new JTabbedPane(JTabbedPane.BOTTOM);
    add(tabbedPane, BorderLayout.CENTER);
    // create shape data and action
    shapeNames = new String[] {
        "Circle", "Square", "Pixel", "Bar", "Post", "Area", "None"};
    shapeNumbers = new int[] {
        Dataset.CIRCLE, Dataset.SQUARE, Dataset.PIXEL, Dataset.BAR,
        Dataset.POST, Dataset.AREA, Dataset.NO_MARKER};
    final Action shapeAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        String shape = e.getActionCommand();
        for (int i = 0; i < shapeNames.length; i++) {
          if (shapeNames[i].equals(shape)) {
            dataset.setMarkerShape(shapeNumbers[i]);
            shapeButton.setText(getShapeName());
            plot.repaint();
          }
        }
      }
    };
    // create shape button
    shapeButton = createButton(getShapeName(), buttonHeight);
    int w = shapeButton.getPreferredSize().width;
    Dimension labelSize = new Dimension(40, 24);
    shapeButton.setToolTipText("Marker Shape");
    shapeButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        // make popup with list of shapes
        JPopupMenu popup = new JPopupMenu();
        for (int i = 0; i < shapeNames.length; i++) {
          JMenuItem item = new JMenuItem(shapeNames[i]);
          item.setActionCommand(shapeNames[i]);
          item.addActionListener(shapeAction);
          popup.add(item);
        }
        popup.show(shapeButton, 0, shapeButton.getHeight());
      }
    });
    // create size data and action
    markerSizes = new String[] {"1", "2", "3", "4"};
    final Action sizeAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        int size = Integer.parseInt(e.getActionCommand());
        dataset.setMarkerSize(size);
        sizeButton.setText(e.getActionCommand());
        plot.repaint();
      }
    };
    // create size button
    sizeButton = createButton(
        String.valueOf(dataset.getMarkerSize()), buttonHeight);
    sizeButton.setToolTipText("Marker Size");
    sizeButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        // make popup with list of sizes
        JPopupMenu popup = new JPopupMenu();
        for (int i = 0; i < markerSizes.length; i++) {
          JMenuItem item = new JMenuItem(markerSizes[i]);
          item.setActionCommand(markerSizes[i]);
          item.addActionListener(sizeAction);
          popup.add(item);
        }
        popup.show(sizeButton, 0, sizeButton.getHeight());
      }
    });
    // create connected button
    connectedButton = createButton("Connected", buttonHeight);
    connectedButton.setSelected(dataset.isConnected());
    connectedButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        dataset.setConnected(!dataset.isConnected());
        refresh();
      }
    });
    // create color action
    final Action colorAction = new AbstractAction() {
      public void actionPerformed(ActionEvent e) {
        JButton button = (JButton)e.getSource();
        Color color = JColorChooser.showDialog(
            null, "Choose " + button.getText() + " Color", button.getBackground());
        if (color != null) {
          Color fill = dataset.getFillColor();
          Color edge = dataset.getEdgeColor();
          Color line = dataset.getLineColor();
          if (button == colorsButton) {
            // give fill some transparency
            fill = new Color(color.getRed(), color.getGreen(),
                             color.getBlue(), 127);
            edge = color;
            line = color.darker();
          }
          else if (button == fillButton) fill = color;
          else if (button == edgeButton) edge = color;
          else if (button == lineButton) line = color;
          dataset.setMarkerColor(fill, edge);
          dataset.setLineColor(line);
          refresh();
        }
      }
    };
    // create fill button
    fillButton = createButton("Fill", buttonHeight);
    fillButton.setToolTipText("Set Fill Color");
    fillButton.addActionListener(colorAction);
    // create edge button
    edgeButton = createButton("Edge", buttonHeight);
    edgeButton.setToolTipText("Set Edge Color");
    edgeButton.addActionListener(colorAction);
    // create line button
    lineButton = createButton("Line", buttonHeight);
    lineButton.setToolTipText("Set Line Color");
    lineButton.addActionListener(colorAction);
    // create colors button
    colorsButton = createButton("All", buttonHeight);
    colorsButton.setToolTipText("Set All Colors");
    colorsButton.addActionListener(colorAction);
    // create fitPoly action
    fitPolyAction = new AbstractAction("Poly", null) {
      public void actionPerformed(ActionEvent e) {
        // get desired degree from action command
        int i = Integer.parseInt(e.getActionCommand());
        // look for existing fit and drawer
        PolynomialLeastSquareFit fit = null;
        FunctionDrawer drawer = null;
        Iterator it = polyFits.keySet().iterator();
        while (it.hasNext()) {
          PolynomialLeastSquareFit next = (PolynomialLeastSquareFit)it.next();
          if (next.degree() == i) { // found it
            fit = next;
            drawer = (FunctionDrawer)polyFits.get(fit);
            break;
          }
        }
        // create fit and drawer if none found
        if (fit == null) {
          double[] x = dataset.getXPoints();
          double[] y = dataset.getYPoints();
          fit = new PolynomialLeastSquareFit(x, y, i);
          drawer = new FunctionDrawer(fit);
          if (i-1 < polyColors.length && i != 0) {
            drawer.setColor(polyColors[i-1]);
          }
          // map fit to drawer
          polyFits.put(fit, drawer);
        }
        String statName = "polyfit " + i + ", term ";
        double[] coeff = fit.getCoefficients();
        if (plot.getDrawables(FunctionDrawer.class).contains(drawer)) {
          plot.removeDrawable(drawer);
          for (int j = 0; j < coeff.length; j++) {
            statsTableModel.removeStat(statName + j);
          }
        }
        else {
          plot.addDrawable(drawer);
          for (int j = 0; j < coeff.length; j++) {
            statsTableModel.addStat(statName + j, coeff[j]);
          }
        }
        plot.repaint();
      }
    };
    // create fitLine button
    fitLineButton = createButton("Line", buttonHeight);
    fitLineButton.setToolTipText("Linear Fit");
    fitLineButton.setActionCommand("1");
    fitLineButton.addActionListener(fitPolyAction);
    // create fitPoly button
    fitPolyButton = createButton("Parabola", buttonHeight);
    fitPolyButton.setToolTipText("Quadratic Fit");
    fitPolyButton.setActionCommand("2");
    fitPolyButton.addActionListener(fitPolyAction);
    // create plotting panel with toolbar
    plot = new PlottingPanel(dataset.getColumnName(0),
                             dataset.getColumnName(1), "Plot");
    plot.addDrawable(dataset);
    JPanel plotPanel = new JPanel(new BorderLayout());
    plotPanel.add(plot, BorderLayout.CENTER);
    JToolBar toolbar = new JToolBar();
    toolbar.setFloatable(false);
    toolbar.setBorder(BorderFactory.createEtchedBorder());
    plotPanel.add(toolbar, BorderLayout.NORTH);
    toolbar.addSeparator(new Dimension(4, 4));
    toolbar.add(new JLabel("Style: "));
    toolbar.add(shapeButton);
    toolbar.add(sizeButton);
    toolbar.add(connectedButton);
    toolbar.addSeparator();
    toolbar.add(new JLabel("Color: "));
    toolbar.add(fillButton);
    toolbar.add(edgeButton);
    toolbar.add(lineButton);
    toolbar.add(colorsButton);
    toolbar.addSeparator();
    toolbar.add(new JLabel("Fit: "));
    toolbar.add(fitLineButton);
    toolbar.add(fitPolyButton);
    tabbedPane.addTab("Plot", plotPanel);
    // create data table
    table = new DataTable();
    table.add(dataset);
    JScrollPane scroller = new JScrollPane(table);
    tabbedPane.addTab("Data", scroller);
    table.refreshTable();
    // assemble data for statistics table
    Double xmax = new Double(dataset.getXMax());
    Double xmin = new Double(dataset.getXMin());
    Double ymax = new Double(dataset.getYMax());
    Double ymin = new Double(dataset.getYMin());
    String sxmax = "x max";
    String sxmin = "x min";
    String symax = "y max";
    String symin = "y min";
    Object[] row1 = new Object[] {sxmax, xmax};
    Object[] row2 = new Object[] {sxmin, xmin};
    Object[] row3 = new Object[] {symax, ymax};
    Object[] row4 = new Object[] {symin, ymin};
    // create statistics table
    headings = new String[] {"Statistic", "Value"};
    data = new Object[][] {row1, row2, row3, row4};
    statsTableModel = new StatsTableModel();
    statsTable = new JTable(new StatsTableModel());
    statsTable.setDefaultRenderer(Double.class, new ScientificRenderer(3));
    scroller = new JScrollPane(statsTable);
    tabbedPane.addTab("Stats", scroller);
  }

  /**
   * Creates a button with a specified maximum height.
   *
   * @param text the button text
   * @param h the button height
   * @return the button
   */
  protected JButton createButton(String text, final int h) {
    JButton button = new JButton(text) {
      public Dimension getMaximumSize() {
        Dimension dim = super.getMaximumSize();
        dim.height = h;
        return dim;
      }
    };
    return button;
  }

  /**
   * Refreshes the display.
   */
  protected void refresh() {
    fillButton.setForeground(dataset.getFillColor());
    edgeButton.setForeground(dataset.getEdgeColor());
    lineButton.setForeground(dataset.getLineColor());
    String text = dataset.isConnected()? "Connected": "Spaced";
    connectedButton.setText(text);
    plot.repaint();
  }

  /**
   * Gets the name of the current marker shape.
   *
   * @return the shape name
   */
  private String getShapeName() {
    int m = dataset.getMarkerShape();
    for (int i = 0; i < shapeNumbers.length; i++) {
      if (shapeNumbers[i] == m) {
        return shapeNames[i];
      }
    }
    return "Unknown";
  }

  /**
   * A class to render numbers in scientific format.
   */
  class ScientificRenderer extends JLabel implements TableCellRenderer {

    NumberFormat format = NumberFormat.getInstance();

    public ScientificRenderer(int sigfigs) {
      sigfigs = Math.min(sigfigs, 6);
      if (format instanceof DecimalFormat) {
        String pattern = "0.0";
        for (int i = 0; i < sigfigs-1; i++) {
          pattern += "0";
        }
        pattern += "E0";
        ((DecimalFormat)format).applyPattern(pattern);
      }
    }

    public Component getTableCellRendererComponent(
        JTable table, Object value, boolean isSelected, boolean hasFocus,
        int row, int column) {
      setFont(statsTable.getDefaultRenderer(String.class).
              getTableCellRendererComponent(statsTable, "a", false, false, 0, 0).
              getFont());
      setText(format.format(value));
      setHorizontalAlignment(SwingConstants.TRAILING);
      return this;
    }
  }

  /**
   * A class to provide model data for the statistics table.
   */
  class StatsTableModel extends AbstractTableModel {
    public String getColumnName(int col) { return headings[col].toString(); }

    public int getRowCount() {
      return data.length;
    }

    public int getColumnCount() {
      return headings.length;
    }

    public Object getValueAt(int row, int col) {
      return data[row][col];
    }

    public boolean isCellEditable(int row, int col) {
      return false;
    }

    public Class getColumnClass(int c) {
      return getValueAt(0, c).getClass();
    }

    protected void addStat(String name, double value) {
      int len = data.length;
      for (int i = 0; i < len; i++) {
        if (data[i][0].equals(name)) {
          data[i][1] = new Double(value);
          fireTableRowsUpdated(i, i);
          return;
        }
      }
      Object[][] newData = new Object[len + 1][2];
      for (int i = 0; i < len; i++) {
        newData[i][0] = data[i][0];
        newData[i][1] = data[i][1];
      }
      newData[len][0] = name;
      newData[len][1] = new Double(value);
      data = newData;
      fireTableStructureChanged();
      statsTable.revalidate();
    }

    protected void removeStat(String name) {
      int len = data.length;
      for (int i = 0; i < len; i++) {
        if (data[i][0].equals(name)) {
          Object[][] newData = new Object[len - 1][2];
          for (int j = 0; j < i; j++) {
            newData[j][0] = data[j][0];
            newData[j][1] = data[j][1];
          }
          for (int j = i + 1; j < len; j++) {
            newData[j - 1][0] = data[j][0];
            newData[j - 1][1] = data[j][1];
          }
          data = newData;
          fireTableStructureChanged();
          statsTable.revalidate();
          return;
        }
      }
    }
  }
}
