/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.io.*;
import java.util.*;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.event.*;
import javax.swing.*;

import org.opensourcephysics.display.*;
import org.opensourcephysics.controls.*;
import java.beans.PropertyChangeSupport;
import java.rmi.RemoteException;

/**
 * This provides a GUI for analyzing datasets.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class DatasetTool extends OSPFrame implements Tool {

  // static fields
  protected static JFileChooser chooser;
  protected static OSPLog log = OSPLog.getOSPLog();
  protected static Dimension dim = new Dimension(500, 400);

  // instance fields
  protected JTabbedPane tabbedPane;
  protected boolean useChooser = true;
  protected JPanel contentPane = new JPanel(new BorderLayout());
  protected PropertyChangeSupport support;
  protected XMLControl control = new XMLControlElement();
  protected JobManager jobManager = new JobManager(this);

  /**
   * A shared data tool.
   */
  final static DatasetTool DATASET_TOOL = new DatasetTool();

  /**
   * Gets the shared DatasetTool.
   *
   * @return the shared DatasetTool
   */
  public static DatasetTool getTool(){
    return DATASET_TOOL;
  }

  /**
   * Constructs a blank DatasetTool.
   */
  public DatasetTool() {
    super("Dataset Tool");
    String name = "DatasetTool";
    setName(name);
    createGUI();
    Toolbox.addTool(name, this);
//    Toolbox.addRMITool(name, this);
  }

  /**
   * Constructs a DatasetTool and opens the specified xml file.
   *
   * @param fileName the name of the xml file
   */
  public DatasetTool(String fileName) {
    this();
    open(fileName);
  }

  /**
   * Constructs a DatasetTool and opens the specified dataset.
   *
   * @param dataset the dataset
   */
  public DatasetTool(Dataset dataset) {
    this();
    addTab(dataset);
  }

  /**
   * Constructs a DatasetTool and opens datasets in the specified xml control.
   *
   * @param control the xml control
   */
  public DatasetTool(XMLControl control) {
    this();
    loadDatasets(control);
  }

  /**
   * Opens an xml file specified by name.
   *
   * @param fileName the file name
   * @return the file name, if successfully opened (datasets loaded)
   */
  public String open(String fileName) {
    OSPLog.fine("opening " + fileName);
    // read the file into an XML control and load datasets
    XMLControlElement control = new XMLControlElement(fileName);
    if (!loadDatasets(control).isEmpty()) return fileName;
    OSPLog.finest("no datasets found");
    return null;
  }

  /**
   * Sends a job to this tool and specifies a tool to reply to.
   *
   * @param job the Job
   * @param replyTo the tool to notify when the job is complete (may be null)
   * @throws RemoteException
   */
  public void send(Job job, Tool replyTo) throws RemoteException {
    XMLControlElement control = new XMLControlElement(job.getXML());
    if (control.failedToRead() || control.getObjectClass() == Object.class) return;
    // log the job in
    jobManager.log(job, replyTo);
    // load and get a list of loaded datasets from the control
    Collection datasetsLoaded = loadDatasets(control);
    // associate datasets with this job for easy replies
    Iterator it = datasetsLoaded.iterator();
    while (it.hasNext()) {
      jobManager.associate(job, it.next());
    }
  }

  /**
   * Sets the useChooser flag.
   *
   * @param useChooser true to load datasets with a chooser
   */
  public void setUseChooser(boolean useChooser) {
    this.useChooser = useChooser;
  }

  /**
   * Gets the useChooser flag.
   *
   * @return true if loading datasets with a chooser
   */
  public boolean isUseChooser() {
    return useChooser;
  }

  /**
   * Loads the datasets found in the specified xml control.
   *
   * @param control the xml control
   *
   * @return true if any datasets are loaded
   */
  public Collection loadDatasets(XMLControl control) {
    return loadDatasets(control, useChooser);
  }

  /**
   * Loads the datasets found in the specified xml control.
   *
   * @param control the xml control
   * @param useChooser true to present choices to user
   *
   * @return a collection of newly loaded datasets
   */
  public Collection loadDatasets(XMLControl control, boolean useChooser) {
    java.util.List props;
    Collection datasets = new HashSet();
    if (useChooser) {
      // open selected datasets using an xml tree chooser
      XMLTreeChooser chooser = new XMLTreeChooser(
          "Select Datasets",
          "Select one or more Datasets",
          this);
      props = chooser.choose(control, Dataset.class);
    }
    else {
      // open all datasets in the control using an xml tree
      XMLTree tree = new XMLTree(control);
      tree.setHighlightedClass(Dataset.class);
      tree.selectHighlightedProperties();
      props = tree.getSelectedProperties();
      if (props.isEmpty()) {
        JOptionPane.showMessageDialog(null, "No Datasets found!");
      }
    }
    if (!props.isEmpty()) {
      Iterator it = props.iterator();
      while (it.hasNext()) {
        XMLControl prop = (XMLControl) it.next();
        Dataset dataset = (Dataset) prop.loadObject(null);
        if (addTab(dataset));
          datasets.add(dataset);
      }
    }
    return datasets;
  }

  /**
   * Returns all open datasets.
   *
   * @return a collection of all open datasets
   */
  public Collection getDatasets() {
    Collection datasets = new HashSet();
    for (int i = 0; i < tabbedPane.getTabCount(); i++) {
      DatasetToolPanel tab = (DatasetToolPanel)tabbedPane.getComponentAt(i);
      datasets.add(tab.getDataset());
    }
    return datasets;
  }

  /**
   * Gets the content pane.
   *
   * @return the content pane
   */
  public Container getContentPane() {
    return contentPane;
  }

//______________________________ protected methods _____________________________

  /**
   * Adds or selects a tab for the specified dataset.
   *
   * @param dataset the dataset
   * @return true if a new tab was added
   */
  protected boolean addTab(Dataset dataset) {
    int i = getTabIndex(dataset);
    if (i >= 0) {
      tabbedPane.setSelectedIndex(i);
      double[] x = dataset.getXPoints();
      double[] y = dataset.getYPoints();
      Dataset data = getSelectedDataset();
      data.clear();
      data.append(x, y);
      return false;
    }
    DatasetToolPanel tab = new DatasetToolPanel(dataset);
    String title = dataset.getColumnName(1) + " vs " + dataset.getColumnName(0);
    // add numbering to similar titles
    int j = 1;
    for (i = 0; i < tabbedPane.getTabCount(); i++) {
      String name = tabbedPane.getTitleAt(i);
      if (name.startsWith(title)) {
        j++;
        if (name.equals(title)) {
          name += " (1)";
          tabbedPane.setTitleAt(i, name);
        }
      }
    }
    if (j > 1) {
      title += " (" + j + ")";
    }
    OSPLog.finer("adding tab " + title);
    tabbedPane.addTab(title, tab);
    tabbedPane.setSelectedComponent(tab);
    validate();
    tab.refresh();
    return true;
  }

  /**
   * Gets the currently selected tab (DatasetToolPanel), if any.
   *
   * @return the selected tab
   */
  protected DatasetToolPanel getSelectedTab() {
    return (DatasetToolPanel)tabbedPane.getSelectedComponent();
  }

  /**
   * Gets the dataset in the currently selected tab, if any.
   *
   * @return the selected dataset
   */
  public Dataset getSelectedDataset() {
    DatasetToolPanel tab = getSelectedTab();
    if (tab != null) return tab.getDataset();
    return null;
  }

  /**
   * Opens an xml file selected with a chooser.
   *
   * @return the name of the opened file
   */
  protected String open() {
    int result = getChooser().showOpenDialog(null);
    if (result == JFileChooser.APPROVE_OPTION) {
      String fileName = getChooser().getSelectedFile().getAbsolutePath();
      fileName = XML.getRelativePath(fileName);
      return open(fileName);
    }
    return null;
  }

  /**
   * Returns the index of the tab containing the specified Dataset.
   *
   * @param data the Dataset
   * @return the name of the opened file
   */
  protected int getTabIndex(Dataset data) {
    for (int i = 0; i < tabbedPane.getTabCount(); i++) {
      DatasetToolPanel tab = (DatasetToolPanel)tabbedPane.getComponentAt(i);
      Dataset dataset = tab.getDataset();
      if (dataset == data)
        return i;
//      if (dataset.getID() == data.getID() &&
//          dataset.getColumnName(0).equals(data.getColumnName(0)) &&
//          dataset.getColumnName(1).equals(data.getColumnName(1)))
//        return i;
    }
    return -1;
  }

  /**
   * Returns the dataset at the specified index.
   *
   * @param i the tab index
   * @return the dataset, or null if none
   */
  protected Dataset getDataset(int i) {
    if (i < 0 || i >= tabbedPane.getTabCount())
      return null;
    DatasetToolPanel tab = (DatasetToolPanel)tabbedPane.getComponentAt(i);
    return tab.getDataset();
  }

  /**
   * Removes the selected tab.
   */
  protected void removeSelectedTab() {
    int i = tabbedPane.getSelectedIndex();
    String title = tabbedPane.getTitleAt(i);
    OSPLog.finer("removing tab " + title);
    if (i >= 0) tabbedPane.removeTabAt(i);
  }

  /**
   * Creates the GUI.
   */
  protected void createGUI() {
    // create the frame
    contentPane.setPreferredSize(dim);
    setContentPane(contentPane);
    if(!org.opensourcephysics.display.OSPFrame.appletMode){
      setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    }
    // create tabbed pane
    tabbedPane = new JTabbedPane(JTabbedPane.TOP);
    contentPane.add(tabbedPane, BorderLayout.CENTER);
    tabbedPane.addMouseListener(new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        if (e.isPopupTrigger() || e.getButton() == MouseEvent.BUTTON3 ||
           (e.isControlDown() && System.getProperty("os.name", "").indexOf("Mac") > -1)) {
          // make popup with close item
          JMenuItem item = new JMenuItem("Close");
          JPopupMenu popup = new JPopupMenu();
          popup.add(item);
          item.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
              removeSelectedTab();
            }
          });
          // add more items if other tabs have titles similar to this one
          int index = tabbedPane.getSelectedIndex();
          String title = tabbedPane.getTitleAt(index);
          int n = title.indexOf("(");
          if (n > 0) {
            title = title.substring(0, n-1);
          }
          ArrayList tabs = new ArrayList();
          for (int i = 0; i < tabbedPane.getTabCount(); i++) {
            if (i != index && tabbedPane.getTitleAt(i).startsWith(title)) {
              tabs.add(String.valueOf(i));
            }
          }
          if (!tabs.isEmpty()) {
            Action diffAction = new AbstractAction() {
              public void actionPerformed(ActionEvent e) {
                // subtract y-values
                Dataset data1 = getSelectedDataset();
                double[] y1 = data1.getYPoints();
                double[] x = data1.getXPoints();
                String tab = e.getActionCommand();
                Dataset data2 = getDataset(Integer.parseInt(tab));
                double[] y2 = data2.getYPoints();
                int n = Math.min(y1.length, y2.length);
                Dataset newdata = new Dataset();
                for (int i = 0; i < n; i++) {
                  newdata.append(x[i], y1[i] - y2[i]);
                }
                addTab(newdata);
              }
            };
            JMenu menu = new JMenu("Subtract");
            popup.add(menu);
            Iterator it = tabs.iterator();
            while (it.hasNext()) {
              String tab = (String)it.next();
              item = new JMenuItem(tabbedPane.getTitleAt(Integer.parseInt(tab)));
              item.setActionCommand(tab);
              item.addActionListener(diffAction);
              menu.add(item);
            }
          }
          popup.show(tabbedPane, e.getX(), e.getY() + 8);
        }
      }
    });
    // create the menu bar
    int keyMask = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    JMenuBar menubar = new JMenuBar();
    JMenu fileMenu = new JMenu("File");
    menubar.add(fileMenu);
    JMenuItem openItem = new JMenuItem("Open...");
    openItem.setAccelerator(KeyStroke.getKeyStroke('O', keyMask));
    openItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        open();
      }
    });
    fileMenu.add(openItem);
    JMenuItem closeItem = new JMenuItem("Close");
    closeItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        removeSelectedTab();
      }
    });
    fileMenu.add(closeItem);
    JMenuItem closeAllItem = new JMenuItem("Close All");
    closeAllItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        while (tabbedPane.getTabCount() > 0) {
          tabbedPane.setSelectedIndex(0);
          removeSelectedTab();
        }
      }
    });
    fileMenu.add(closeAllItem);
    fileMenu.addSeparator();
    JMenuItem exitItem = new JMenuItem("Exit");
    exitItem.setAccelerator(KeyStroke.getKeyStroke('Q', keyMask));
    exitItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        while (tabbedPane.getTabCount() > 0) {
          tabbedPane.setSelectedIndex(0);
          removeSelectedTab();
        }
        System.exit(0);
      }
    });
    fileMenu.add(exitItem);
    JMenu editMenu = new JMenu("Edit");
    menubar.add(editMenu);
    Action copyAction = new AbstractAction("Copy") {
      public void actionPerformed(ActionEvent e) {
        Dataset dataset = getSelectedDataset();
        if (dataset == null) return;
        int i = tabbedPane.getSelectedIndex();
        String title = tabbedPane.getTitleAt(i);
        OSPLog.finest("copying " + title);
        XMLControl control = new XMLControlElement(dataset);
        StringSelection data = new StringSelection(control.toXML());
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(data, data);
      }
    };
    JMenuItem copyItem = new JMenuItem(copyAction);
    copyItem.setAccelerator(KeyStroke.getKeyStroke('C', keyMask));
    editMenu.add(copyItem);
    Action pasteAction = new AbstractAction("Paste") {
      public void actionPerformed(ActionEvent e) {
        try {
          Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
          Transferable data = clipboard.getContents(null);
          String dataString = (String)data.getTransferData(DataFlavor.stringFlavor);
          if (dataString != null) {
            XMLControl control = new XMLControlElement();
            control.readXML(dataString);
            if (!control.failedToRead()) OSPLog.finest("pasting");
            if (loadDatasets(control).isEmpty()) OSPLog.finest("no datasets found");
          }
        }
        catch (UnsupportedFlavorException ex) {}
        catch (IOException ex) {}
        catch (HeadlessException ex) {}
      }
    };
    JMenuItem pasteItem = new JMenuItem(pasteAction);
    pasteItem.setAccelerator(KeyStroke.getKeyStroke('V', keyMask));
    editMenu.add(pasteItem);
    JMenu helpMenu = new JMenu("Help");
    menubar.add(helpMenu);
    JMenuItem logItem = new JMenuItem("Message Log...");
    logItem.setAccelerator(KeyStroke.getKeyStroke('L', keyMask));
    logItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        if (log.getLocation().x == 0 && log.getLocation().y == 0) {
          Point p = getLocation();
          log.setLocation(p.x + 28, p.y + 28);
        }
        log.setVisible(true);
      }
    });
    helpMenu.add(logItem);
    helpMenu.addSeparator();
    JMenuItem aboutItem = new JMenuItem("About...");
    aboutItem.setAccelerator(KeyStroke.getKeyStroke('A', keyMask));
    aboutItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        showAboutDialog();
      }
    });
    helpMenu.add(aboutItem);
    setJMenuBar(menubar);
    JMenu sendMenu = new JMenu("Apply Changes");
    menubar.add(sendMenu);
    JMenuItem applyItem = new JMenuItem("To Selected");
    applyItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        Dataset dataset = getSelectedDataset();
        jobManager.sendReplies(dataset);
      }
    });
    sendMenu.add(applyItem);
    JMenuItem applyAllItem = new JMenuItem("To All");
    applyAllItem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        for (int i = 0; i < tabbedPane.getTabCount(); i++) {
          DatasetToolPanel tab = (DatasetToolPanel)tabbedPane.getComponentAt(i);
          Dataset dataset = tab.getDataset();
          jobManager.sendReplies(dataset);
        }
      }
    });
    sendMenu.add(applyAllItem);

    pack();
    // center this on the screen
    Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();
    int x = (dim.width - getBounds().width) / 2;
    int y = (dim.height - getBounds().height) / 2;
    setLocation(x, y);
  }

  /**
   * Shows the about dialog.
   */
  protected void showAboutDialog() {
    String aboutString = getName() + " 1.0  June 2004\n" +
                         "Open Source Physics Project\n" +
                         "www.opensourcephysics.org";
    JOptionPane.showMessageDialog(this,
                                  aboutString,
                                  "About " + getName(),
                                  JOptionPane.INFORMATION_MESSAGE);
  }

 /**
  * Main entry point when used as application.
  *
  * @param args ignored
  */
  public static void main(String[] args) {
    DatasetTool tool = getTool();
    tool.setVisible(true);
  }

}
