/*
 * The org.opensourcephysics.tools package defines classes for managing OSP
 * applications and objects.
 */
package org.opensourcephysics.tools;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;

import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.event.TableModelEvent;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;

/**
 * A JTable to display int and double array values.
 *
 * @author Douglas Brown
 * @version 1.0
 */
public class ArrayTable extends JTable {

  // instance fields
  ArrayTableModel tableModel;
  ArrayIndexRenderer indexRenderer;

  /**
   * Constructor for 1D int array.
   *
   * @param array the array
   */
  public ArrayTable(int[] array) {
    tableModel = new ArrayTableModel(array);
    init();
  }

  /**
   * Constructor for 2D int array.
   *
   * @param array the array
   */
  public ArrayTable(int[][] array) {
    tableModel = new ArrayTableModel(array);
    init();
  }

  /**
   * Constructor for 1D double array.
   *
   * @param array the array
   */
  public ArrayTable(double[] array) {
    tableModel = new ArrayTableModel(array);
    init();
  }

  /**
   * Constructor for 2D double array.
   *
   * @param array the array
   */
  public ArrayTable(double[][] array) {
    tableModel = new ArrayTableModel(array);
    init();
  }

  /**
   * Initializes the table.
   */
  public void init() {
    setModel(tableModel);
    setColumnSelectionAllowed(true);
    indexRenderer = new ArrayIndexRenderer();
    getTableHeader().setReorderingAllowed(false);
    getTableHeader().setDefaultRenderer(indexRenderer);
    setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
    // set width of column 0 (row index)
    String name = getColumnName(0);
    TableColumn column = getColumn(name);
    int width = 24;
    column.setMinWidth(width);
    column.setMaxWidth(2*width);
    column.setWidth(width);
    // set width of other columns
    width = 60;
    for (int i = 1, n = getColumnCount(); i < n; i++) {
      name = getColumnName(i);
      column = getColumn(name);
      column.setMinWidth(width);
      column.setMaxWidth(3*width);
      column.setWidth(width);
    }
//    sizeColumnsToFit(0);
  }

  /**
   *  Refresh the data in the table.
   */
  public void refreshTable() {
    Runnable refresh = new Runnable() {
      public synchronized void run() {
        tableChanged(new TableModelEvent(tableModel,
                                         TableModelEvent.HEADER_ROW));
      }
    };
    if (SwingUtilities.isEventDispatchThread()) {
      refresh.run();
    }
    else {
      SwingUtilities.invokeLater(refresh);
    }
  }

  /**
   * Returns the renderer for a cell specified by row and column.
   *
   * @param row the row number
   * @param column the column number
   * @return the cell renderer
   */
  public TableCellRenderer getCellRenderer(int row, int column) {
    int i = convertColumnIndexToModel(column);
    if (i == 0) return indexRenderer;
    return getDefaultRenderer(getColumnClass(column));
  }

  /**
   * A cell renderer for array indices.
   */
  static class ArrayIndexRenderer
      extends JLabel
      implements TableCellRenderer {

    /**
     * Constructor
     */
    public ArrayIndexRenderer() {
      super();
      setBorder(BorderFactory.createRaisedBevelBorder());
      setOpaque(true); // make background visible
      setForeground(Color.BLACK);
      setBackground(Color.LIGHT_GRAY);
    }

    /**
     * Returns a label for the specified cell.
     *
     * @param table ignored
     * @param value the row number to be displayed
     * @param isSelected ignored
     * @param hasFocus ignored
     * @param row ignored
     * @param column the column number
     * @return a label with the row number
     */
    public Component getTableCellRendererComponent(JTable table, Object value,
        boolean isSelected, boolean hasFocus, int row, int column) {
      if (column == 0) setHorizontalAlignment(SwingConstants.RIGHT);
      else setHorizontalAlignment(SwingConstants.CENTER);
      setText(value.toString());
      setPreferredSize(new Dimension(20, 18));
      return this;
    }
  }

}
