/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.numerics;

/**
 * Title:        RK4
 * Description:  A fourth order Runge-Kutta ODE solver.
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class RK4 implements ODESolver {

  private double   stepSize = 0.1;
  private int      numEqn   = 0;
  private double[] rate1;
  private double[] rate2;
  private double[] rate3;
  private double[] rate4;
  private double[] k1;
  private double[] k2;
  private double[] k3;
  private ODE      ode;

  /**
   * Constructs the RK4 ODESolver for a system of ordinary  differential equations.
   *
   * @param _ode the system of differential equations.
   */
  public RK4(ODE _ode) {
    ode = _ode;
    initialize(stepSize);
  }

  /**
   * Initializes the ODE solver.
   *
   * Three rate arrays and four temporary state arrays are allocated.
   * The number of differential equations is determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
  public void initialize(double _stepSize) {
    stepSize = _stepSize;
    double state[] = ode.getState();
    if(state == null) {  // state vector not defined.
      return;
    }
    if(numEqn != state.length) {
      numEqn = state.length;
      rate1  = new double[numEqn];
      rate2  = new double[numEqn];
      rate3  = new double[numEqn];
      rate4  = new double[numEqn];
      k1     = new double[numEqn];
      k2     = new double[numEqn];
      k3     = new double[numEqn];
    }
  }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
  public double step() {
    double state[] = ode.getState();
    if(state.length != numEqn) {
      initialize(stepSize);
    }
    ode.getRate(state, rate1);
    for(int i = 0; i < numEqn; i++) {
      k1[i] = state[i] + stepSize * rate1[i] / 2;
    }
    ode.getRate(k1, rate2);
    for(int i = 0; i < numEqn; i++) {
      k2[i] = state[i] + stepSize * rate2[i] / 2;
    }
    ode.getRate(k2, rate3);
    for(int i = 0; i < numEqn; i++) {
      k3[i] = state[i] + stepSize * rate3[i];
    }
    ode.getRate(k3, rate4);
    for(int i = 0; i < numEqn; i++) {
      state[i] = state[i] + stepSize * (rate1[i] + 2 * rate2[i] + 2 * rate3[i] + rate4[i]) / 6.0;
    }
    return stepSize;
  }

  /**
   * Sets the step size.
   *
   * The step size remains fixed in this algorithm.
   *
   * @param _stepSize
   */
  public void setStepSize(double _stepSize) {
    stepSize = _stepSize;
  }

  /**
   * Gets the step size.
   *
   * The stepsize is constant in this algorithm.
   *
   * @return the step size
   */
  public double getStepSize() {
    return stepSize;
  }
}
