package org.opensourcephysics.numerics;

/**
 * Polynomial least square fit without any error estimation.
 *
 * See Object Oriented Implementation of Numerical Methods by Didier H. Besset for fitting with error estimation.
 *
 * @author Wolfgang Christian.
 */
public class PolynomialLeastSquareFit extends Polynomial{
  double[][] systemMatrix;
  double[] systemConstants;

  /**
   * Constructs a PolynomialLeastSquareFit with the given order.
   * @param xd double[]
   * @param yd double[]
   * @param degree int the degree of the polynomial
   */
  public PolynomialLeastSquareFit(double[] xd, double[] yd, int degree) {
    super(new double[degree + 1]);
    if (xd.length != yd.length) {
      throw new IllegalArgumentException("Arrays must be of equal length.");
    }
    int ncoef = degree + 1;
    systemMatrix = new double[ncoef][ncoef];
    systemConstants = new double[ncoef];
    for(int i=0, n=xd.length; i<n; i++){
      double xp1 = 1;
      for (int j = 0; j < systemConstants.length; j++) {
        systemConstants[j] += xp1 * yd[i];
        double xp2 = xp1;
        for (int k = 0; k <= j; k++) {
          systemMatrix[j][k] += xp2;
          xp2 *= xd[i];
        }
        xp1 *= xd[i];
      }
    }
    computeCoefficients();
  }


  /**
   * Computes the polynomial coefficients.
   */
  void computeCoefficients() {
    for (int i = 0; i < systemConstants.length; i++) {
      for (int j = i + 1; j < systemConstants.length; j++)
        systemMatrix[i][j] = systemMatrix[j][i];
    }
    LUPDecomposition lupSystem = new LUPDecomposition(systemMatrix);
    double[][] components = lupSystem.inverseMatrixComponents();
    LUPDecomposition.symmetrizeComponents(components);
    coefficients=lupSystem.solve(systemConstants);
  }

}
