package org.opensourcephysics.numerics;
import org.opensourcephysics.controls.*;

/**
 * Matrix3DTransformation implements 3D affine transformations using a matrix representation.
 */
public class Matrix3DTransformation implements Transformation {

   private double[][] matrix = new double[3][3]; // the transformation matrix
   private double[][] inverse = null;            // the inverse transformation matrix if it exsits

   /**
    * Constructs a 3D transformation using the given matrix.
    *
    * Affine transformations can be applied to 3D coordinates.
    * A 3 by 3 matrix sets the rotation and shear.
    * A null matrix sets the transformation to the identity transformation.
    *
    * @param mat double[][]
    */
   public Matrix3DTransformation(double[][] matrix) {
      if(matrix==null) { // identiy matrix
         this.matrix[0][0] = this.matrix[1][1] = this.matrix[2][2] = 1;
         return;
      }
      for(int i = 0; i<matrix.length; i++) { // loop over the rows
         System.arraycopy(matrix[i], 0, this.matrix[i], 0, matrix[i].length);
      }
   }

   /**
    * Creates a 3D transforamtion representing a rotation about the origin by the given angle around
    * the given axis.
    *
    * @param theta double
    * @param axis double[]
    * @return Affine3DTransformation
    */
   public static Matrix3DTransformation Rotation(double theta, double[] axis) {
      Matrix3DTransformation at = new Matrix3DTransformation(null);
      double[][] atMatrix = at.matrix;
      double norm = Math.sqrt(axis[0]*axis[0]+axis[1]*axis[1]+axis[2]*axis[2]);
      double x = axis[0]/norm, y = axis[1]/norm, z = axis[2]/norm;
      double c = Math.cos(theta), s = Math.sin(theta);
      double t = 1-c;
      // matrix elements not listed are zero
      atMatrix[0][0] = t*x*x+c;
      atMatrix[0][1] = t*x*y-s*z;
      atMatrix[0][2] = t*x*y+s*y;
      atMatrix[1][0] = t*x*y+s*z;
      atMatrix[1][1] = t*y*y+c;
      atMatrix[1][2] = t*y*z-s*x;
      atMatrix[2][0] = t*x*z-s*y;
      atMatrix[2][1] = t*y*z+s*x;
      atMatrix[2][2] = t*z*z+c;
      return at;
   }

   /**
 * Creates an transformation representing a rotation about the origin by the given quaternion.
 *
 * @param quaternion double[]
 * @return Affine3DTransformation
 */
public static Matrix3DTransformation Quaternion(double[] quaternion) {
   return Quaternion(quaternion[0],quaternion[1],quaternion[2],quaternion[3]);
}

/**
 * Creates an AffineMatrix representing a rotation about the origin by the given quaternion components.
 *
 * @param quaternion double[]
 * @return Affine3DTransformation
 */
public static Matrix3DTransformation Quaternion(double q0, double q1, double q2, double q3){
   Matrix3DTransformation at = new Matrix3DTransformation(null);
   double[][] atMatrix = at.matrix;
   double norm = Math.sqrt(q0*q0+q1*q1+q2*q2+q3*q3);
   q0 = q0/norm;
   q1 = q1/norm;
   q2 = q2/norm;
   q3 = q3/norm;
   double q11 = 2*q1*q1, q22 = 2*q2*q2, q33 = 2*q3*q3;
   double q12 = 2*q1*q2, q13 = 2*q1*q3, q23 = 2*q2*q3;
   double q01 = 2*q0*q1, q02 = 2*q0*q2, q03 = 2*q0*q3;

// matrix elements not listed are zero
   atMatrix[0][0] = 1-q22-q33;
   atMatrix[0][1] = q12-q03;
   atMatrix[0][2] = q13+q02;
   atMatrix[1][0] = q12+q03;
   atMatrix[1][1] = 1-q11-q33;
   atMatrix[1][2] = q23-q01;
   atMatrix[2][0] = q13-q02;
   atMatrix[2][1] = q23+q01;
   atMatrix[2][2] = 1-q11-q22;
   return at;
}

   /**
    * Provides a copy of this transformation.
    */
   public Object clone() {
      return new Matrix3DTransformation(matrix);
   }

   /**
    * Transforms the given point.
    *
    * Convertes the point to homogeneous coordinates if the size of the point array is less than
    *
    * @param point the coordinates to be transformed
    */
   public void direct(double[] point) {
      double[] tempPoint= (double[])point.clone();
      for(int i = 0,n = point.length; i<n; i++) {
         point[i] = 0;
         for(int j = 0; j<n; j++) {
            point[i] += matrix[i][j]*tempPoint[j];
         }
      }
   }

   /**
    * Transforms the given point using the inverse transformation (if it exists).
    *
    * If the transformation is not invertible, then a call to this
    * method must throw a UnsupportedOperationException exception.
    *
    * @param point the coordinates to be transformed
    */
   public void inverse(double[] point) throws UnsupportedOperationException {
      if (inverse==null){
         calcInverse(); // computes the inverse using LU decompostion
         if (inverse==null){ // inverse does not exist
            throw new UnsupportedOperationException("The inverse matrix does not exist.");
         }

      }
      double[] tempPoint= (double[])point.clone();
      for(int i = 0, n = point.length; i<n; i++) {
         point[i] = 0;
         for(int j = 0; j<n; j++) {
            point[i] += inverse[i][j]*tempPoint[j];
         }
      }
   }

   private void calcInverse() {
      LUPDecomposition lupd = new LUPDecomposition(matrix);
      inverse = lupd.inverseMatrixComponents();
   }

   public static XML.ObjectLoader getLoader() {
      return new Affine3DTransformationLoader();
   }

   protected static class Affine3DTransformationLoader extends XMLLoader {

      public void saveObject(XMLControl control, Object obj) {
         Matrix3DTransformation transf = (Matrix3DTransformation) obj;
         control.setValue("matrix", transf.matrix);
         if(transf.inverse!=null)control.setValue("inverse", transf.inverse);
      }

      public Object createObject(XMLControl control) {
         return new Matrix3DTransformation(null);
      }

      public Object loadObject(XMLControl control, Object obj) {
         Matrix3DTransformation transf = (Matrix3DTransformation) obj;
         transf.matrix = (double[][]) control.getObject("matrix");
         transf.inverse = (double[][]) control.getObject("inverse");
         return obj;
      }
   }
}
