/*
 * The org.opensourcephysics.numerics package contains numerical methods
 * for the book Simulations in Physics.
 * Copyright (c) 2005  H. Gould, J. Tobochnik, and W. Christian.
 */
package org.opensourcephysics.numerics;

/**
 * An Euler-Richardson (midpoint) method ODE solver.
 *
 * The Euler-Richardson method uses the state at the beginning of the interval
 * to estimate the state at the midpoint.
 *
 * x(midpoint) = x(n) + v(n)*dt/2
 * v(midpoint) = v(n) + a(n)*dt/2
 * t(midpoint) = t(n) + dt/2
 *
 * The midpoint state is then used to calculate the final state.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public class EulerRichardson extends AbstractODE {
  private double[] midstate;  // midpoint

  /**
   * Constructs the EulerRichardson ODESolver for a system of ordinary  differential equations.
   *
   * @param _ode the system of differential equations.
   */
  public EulerRichardson(ODE ode) {
    super(ode);
  }

  /**
   * Initializes the ODE solver.
   *
   * The rate and midstate arrays are allocated.
   * The number of differential equations is determined by invoking getState().length on the ODE.
   *
   * @param stepSize
   */
  public void initialize(double stepSize) {
    super.initialize(stepSize);
    midstate = new double[numEqn];
  }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
  public double step() {
    double[] state = ode.getState();
    ode.getRate(state, rate);  // get the rate at the start
    double dt2=stepSize/2;
    for(int i = 0; i < numEqn; i++) {
      // estimate the state at the midpoint
      midstate[i] = state[i] + rate[i]*dt2 ;
    }
    ode.getRate(midstate, rate);  // get the rate at the midpoint
    for(int i = 0; i < numEqn; i++) {
      state[i] = state[i] + stepSize * rate[i];
    }
    return stepSize;
  }

}
