package org.opensourcephysics.numerics;

/**
 * AbstractODE provides a common superclass for ODESolvers.
 *
 * @author       Wolfgang Christian
 * @version 1.0
 */
public abstract class AbstractODE extends Object implements ODESolver {

   protected double stepSize = 0.1; // parameter increment such as delta time
   protected int  numEqn   = 0;     // number of equations
   protected double[] rate;         // array that stores the rate
   protected ODE ode;               // object that computes rate

  /**
   * Constructs the ODESolver for a system of ordinary differential equations.
   *
   * @param _ode the system of differential equations.
   */
   public AbstractODE(ODE _ode) {
      ode = _ode;
      initialize(0.1);
   }

  /**
   * Steps (advances) the differential equations by the stepSize.
   *
   * The ODESolver invokes the ODE's getRate method to obtain the initial state of the system.
   * The ODESolver then advances the solution and copies the new state into the
   * state array at the end of the solution step.
   *
   * @return the step size
   */
   abstract public double step();

  /**
   * Sets the step size.
   *
   * The step size remains fixed in this algorithm
   *
   * @param _stepSize
   */
   public void setStepSize(double _stepSize) {
      stepSize = _stepSize;
   }

  /**
   * Initializes the ODE solver.
   *
   * The rate array is allocated.  The number of differential equations is
   * determined by invoking getState().length on the ODE.
   *
   * @param _stepSize
   */
   public void initialize(double _stepSize) {
      stepSize = _stepSize;
      double state[] = ode.getState();
      if (state == null) {  // state vector not defined
         numEqn = 0;
      } else {
         numEqn = state.length;
      }
      rate = new double[numEqn];
   }

  /**
   * Gets the step size.
   *
   * The stepsize is constant in this algorithm
   *
   * @return the step size
   */
   public double getStepSize() {
      return stepSize;
   }
}

