package org.opensourcephysics.frames;

import java.awt.Graphics;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Iterator;

import org.opensourcephysics.display.DatasetManager;
import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.display.DrawingFrame;
import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.display.Interactive;
import org.opensourcephysics.display.InteractiveCircle;
import org.opensourcephysics.display.Measurable;
import org.opensourcephysics.display.PlottingPanel;

/**
 * A DrawingFrame that displays particles.
 *
 * Particle locations are specified using a Point2D.
 * The default particle drawing shape is a circle.
 *
 * @author W. Christian
 * @version 1.0
 */
public class ParticleFrame extends DrawingFrame {

  ArrayList      partlist = new ArrayList();

  /**
   * Constructs the ParticleFrame with the given labels and frame title.
   *
   * @param xlabel String
   * @param ylabel String
   * @param frameTitle String
   */
  public ParticleFrame(String xlabel, String ylabel, String frameTitle) {
    super(new PlottingPanel(xlabel, ylabel, null));
    setTitle(frameTitle);
    setAnimated(true);
    setAutoclear(true);
  }

  /**
   * Ensures capacity
   *
   * @param  index
   */
  protected Particles checkIndex(int index) {
  while(index >= partlist.size()) {
    Particles p = new Particles();
    partlist.add(p);
    ((InteractiveCircle)p.shape).color=DatasetManager.getLineColor(partlist.indexOf(p));
    addDrawable(p);
  }
  return (Particles)partlist.get(index);
}


  /**
   * Adds a particle to the frame.
   *
   * @param  index
   * @param point
   */
  public void addParticle(int i, Point2D point){
    checkIndex(i).addParticle(point);
  }

  /**
   * Adds an array of particles to the frame.
   *
   * @param  index
   * @param point
   */
  public void addParicle(int i, Point2D[] points) {
      checkIndex(i).addParticles(points);
  }


  /**
   * Sets the drawing shape for the particles.
   *
   * @param  index
   * @param shape Interactive
   */
  public void setDrawingShape(int i, Interactive shape){
    checkIndex(i).shape=shape;
  }

  /**
   * Cleares drawable objects added by the user from this frame and clears the particles.
   */
  public void clearDrawables() {
    clearData();
    drawingPanel.clear(); // removes drawables added by user
  }


  /**
   * Clears all particles from this frame.
   */
  public void clearData(){
    Iterator it=partlist.iterator();
    while(it.hasNext()){ // clears the data from the arrays
      Particles p=(Particles)it.next();
      (p).clear();
      drawingPanel.removeDrawable(p);
    }
    partlist.clear();
    drawingPanel.invalidateImage();
  }

  class Particles implements Drawable, Measurable{
    Interactive shape= new InteractiveCircle();
    private ArrayList pointList = new ArrayList();
    double xmin=Double.MAX_VALUE,xmax=-Double.MAX_VALUE,ymin=Double.MAX_VALUE,ymax=-Double.MAX_VALUE;

    Particles(){
      shape.setEnabled(false);  // default cannot drag particles
    }

    void addParticle(Point2D point) {
      synchronized (pointList) {
        xmax=Math.max(xmax,point.getX());
        xmin=Math.min(xmin,point.getX());
        ymax=Math.max(ymax,point.getY());
        ymin=Math.min(ymin,point.getY());
        pointList.add(point);
      }
    }

    void addParticles(Point2D[] points) {
      for(int i=0, n=points.length; i<n; i++){
        xmax=Math.max(xmax,points[i].getX());
        xmin=Math.min(xmin,points[i].getX());
        ymax=Math.max(ymax,points[i].getY());
        ymin=Math.min(ymin,points[i].getY());
      }
      synchronized (pointList) {
        pointList.add(points);
      }
    }

    void clear() {
      synchronized (pointList) {
        pointList.clear();
        xmin = Double.MAX_VALUE;
        xmax = -Double.MAX_VALUE;
        ymin = Double.MAX_VALUE;
        ymax = -Double.MAX_VALUE;
      }
    }

    /**
     * Draws the particles.
     *
     * @param panel DrawingPanel
     * @param g Graphics
     */
    public void draw(DrawingPanel panel, Graphics g) {
      synchronized(pointList){
        Iterator it = pointList.iterator();
        while (it.hasNext()) {
          Point2D point=((Point2D)it.next());
          shape.setXY(point.getX(),point.getY());
          shape.draw(panel,g);
        }
      }
    }

    /**
     * getXMax
     *
     * @return double
     */
    public double getXMax() {
      return xmax;
    }

    /**
     * getXMin
     *
     * @return double
     */
    public double getXMin() {
      return xmin;
    }

    /**
     * getYMax
     *
     * @return double
     */
    public double getYMax() {
      return ymax;
    }

    /**
     * getYMin
     *
     * @return double
     */
    public double getYMin() {
      return ymin;
    }

    /**
     * There is a measure if there is at least one particle.
     *
     * @return boolean
     */
    public boolean isMeasured() {
      synchronized (pointList) {
        return (pointList.size() > 0) ? true : false;
      }
    }

  }



}
