package org.opensourcephysics.frames;
import java.util.*;
import java.awt.event.*;
import javax.swing.*;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.display.*;

/**
 * HistogramFrame displays a histogram using a dedicated Histogram object.
 *
 * @author W. Christian
 * @version 1.0
 */
public class HistogramFrame  extends DrawingFrame{

  protected Histogram histogram = new Histogram();
  protected DataTable dataTable = new DataTable();
  protected DataTableFrame tableFrame;

  /**
   * A DrawingFrame with a Histogram as its drawable.
   *
   * @param xlabel String
   * @param ylabel String
   * @param title String
   */
  public HistogramFrame(String xlabel, String ylabel, String title) {
    super(new PlottingPanel( xlabel,  ylabel, null));
    //histogram.setDiscrete(false) ;
    drawingPanel.addDrawable(histogram);
    setTitle(title);
    dataTable.add(histogram);
    setAnimated(true);
    setAutoclear(true);
    addMenuItems();
  }

  /**
 * Adds Views menu items on the menu bar.
 */
  protected void addMenuItems() {
     JMenuBar menuBar = getJMenuBar();
      if (menuBar==null){
         return;
      }
     JMenu helpMenu=this.removeMenu("Help");
     JMenu menu = getMenu("Views");
     if(menu==null) {
        menu = new JMenu("Views");
        menuBar.add(menu);
        menuBar.validate();
     } else { // add a separator if tools already exists
        menu.addSeparator();
     }
      menuBar.add(helpMenu);
     // add a menu item to show the data table
     JMenuItem tableItem = new JMenuItem("Data Table");
     tableItem.setAccelerator(KeyStroke.getKeyStroke('T', MENU_SHORTCUT_KEY_MASK));
     ActionListener tableListener= new ActionListener() {
       public void actionPerformed(ActionEvent e) {
         showDataTable(true);
       }
     };
     tableItem.addActionListener(tableListener);
     menu.add(tableItem);
     // add to popup menu
     JMenuItem item = new JMenuItem("Data Table");
     item.addActionListener(tableListener);
     if(drawingPanel!=null && drawingPanel.getPopupMenu()!=null )drawingPanel.getPopupMenu().add(item);

  }

  /**
   * Removes drawable objects added by the user from this frame.
   */
  public void clearDrawables() {
    drawingPanel.clear(); // removes all drawables
    drawingPanel.addDrawable(histogram); // puts complex dataset back into panel
    showDataTable(false);
  }

  /**
   * Gets Drawable objects added by the user to this frame.
   *
   * @return the list
   */
  public synchronized ArrayList getDrawables() {
    ArrayList list = super.getDrawables();
    list.remove(histogram);
    return list;
  }

  /**
   * Gets Drawable objects added by the user of an assignable type. The list contains
   * objects that are assignable from the class or interface.
   *
   * @param c the type of Drawable object
   *
   * @return the cloned list
   *
   * @see #getObjectOfClass(Class c)
   */
  public synchronized ArrayList getDrawables(Class c) {
    ArrayList list = super.getDrawables(c);
    list.remove(histogram);
    return list;
  }




  /**
   * Clears all the data stored.
   */
  public void clearData() {
    histogram.clear();
    dataTable.refreshTable();
    drawingPanel.invalidateImage();
  }

  /**
   * Appends a data point to the histogram.
   * @param x  Data point's x value
   * @param y  Data point's y value
   */
  public void append(double v) {
    histogram.append(v);
    // this may be slow if the table is large
    if(tableFrame!=null && tableFrame.isShowing())dataTable.refreshTable();
  }

  /**
   * Histogram uses logarithmic scale (true/false)
   */
  public void setLogScale(boolean b) {
    histogram.logScale = b;
  }

  /**
   *  Sets the width of a bin.
   *
   * @param  width
   */
  public void setBinWidth(double width) {
    histogram.setBinWidth(width);
  }

  /**
   * Normalizes the occurences in this histogram to one (true/false).
   */
  public void setNormalizedToOne(boolean b) {
    histogram.setNormalizedToOne(b);
    histogram.adjustForWidth = b;
  }

  /**
   * Makes the x axis positive by default.
   */
  public void positiveX() {
    boolean b = drawingPanel.isAutoscaleX();
    drawingPanel.setPreferredMinMaxX(0, drawingPanel.getPreferredXMax());
    drawingPanel.setAutoscaleX(b);
  }

  /**
   * Shows or hides the data table.
   *
   * @param show boolean
   */
  public synchronized void showDataTable(boolean show){
    if(show){
      if(tableFrame==null){
        tableFrame = new DataTableFrame(getTitle() + " Data", dataTable);
        tableFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
      }
      dataTable.refreshTable();
      dataTable.sort(0);
      tableFrame.setVisible(true);
    }
    else{
      tableFrame.setVisible(false);
      tableFrame.dispose();
      tableFrame=null;
    }
  }

  public static XML.ObjectLoader getLoader(){
return new HistogramFrameLoader();
}

static protected class HistogramFrameLoader extends DrawingFrame.DrawingFrameLoader {
 /**
 * Creates a PlotFame.
 *
 * @param element XMLControl
 * @return Object
 */
public Object createObject(XMLControl control) {
  HistogramFrame frame = new HistogramFrame("x","y","Histogram Frame");
  return frame;
}

/**
 * Loads the object with data from the control.
 *
 * @param element XMLControl
 * @param obj Object
 * @return Object
 */
public Object loadObject(XMLControl control, Object obj) {
  super.loadObject(control,obj);
  HistogramFrame frame = ((HistogramFrame) obj);
  ArrayList list=frame.getObjectOfClass(Histogram.class);
  if(list.size()>0){ // assume the first Histogram belongs to this frame
     frame.histogram=(Histogram)list.get(0);
     frame.histogram.clear();
     frame.dataTable.add(frame.histogram);
  }
  return obj;
}
}


}
