package org.opensourcephysics.frames;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.*;
import org.opensourcephysics.display.DrawingFrame;
import org.opensourcephysics.displayejs.DrawingPanel3D;
import org.opensourcephysics.displayejs.InteractionListener;

/**
 * A DrawingFrame with an empty EJS DrawingPanel3D.
 *
 * Views menu allows the user to project the drawing into two dimensions.
 * ClearData removes all drawables from this frame.
 *
 * @author W. Christian
 * @version 1.0
 */
public class EJSFrame extends DrawingFrame {

   JMenuItem threeDItem, threeDNPItem, xyItem, xzItem, yzItem;

   /**
    * Constructs the EJSFrame with the given title.
    *
    * @param xlabel String
    * @param ylabel String
    * @param title String
    */
   public EJSFrame(String title) {
      super(new DrawingPanel3D());
      setTitle(title);
      addMenuItems();
      setAnimated(true);
      setAutoclear(true);
   }

   /**
    * Adds an EJS InteractionListener.
    *
    * @param listener InteractionListener
    */
   public void addListener(InteractionListener listener){
      ((DrawingPanel3D) drawingPanel).addListener(listener);
   }

   /**
    * Whether the panel should try to remove hidden lines
    * @param value the desired value
    */
   public void setRemoveHiddenLines(boolean value){
      ( (DrawingPanel3D) drawingPanel).setRemoveHiddenLines(value);
   }

   /**
    * Types of decoration displayed.One of the following
    * <ul>
    *   <li><b>DECORATION_NONE</b>: No decoration</li>
    *   <li><b>DECORATION_AXES</b>: Displays labelled axes</li>
    *   <li><b>DECORATION_CUBE</b>: Displays the boundig box</li>
    * </ul>
    *
    * @param value the desired value
    */
   public void setDecorationType(int value) {
      ((DrawingPanel3D) drawingPanel).setDecorationType(value);
   }

   /**
    * Set the extrema in the X, Y and Z coordinates at once
    *
    * @param xmin double
    * @param xmax double
    * @param ymin double
    * @param ymax double
    * @param zmin double
    * @param zmax double
    */
   public void setPreferredMinMax(double xmin, double xmax, double ymin, double ymax, double zmin, double zmax) {
      ((DrawingPanel3D) drawingPanel).setPreferredMinMax(xmin, xmax, ymin, ymax, zmin, zmax);
   }

   /**
    * Adds Views menu items on the menu bar.
    */
   protected void addMenuItems() {
      JMenuBar menuBar = getJMenuBar();
      if (menuBar==null){
         return;
      }
      JMenu helpMenu=this.removeMenu("Help");
      JMenu menu = getMenu("Views");
      if(menu==null) {
         menu = new JMenu("Views");
         menuBar.add(menu);
         menuBar.validate();
      } else { // add a separator if tools already exists
         menu.addSeparator();
      }
      menuBar.add(helpMenu);
      ButtonGroup menubarGroup = new ButtonGroup();
      //3D
      threeDItem = new JRadioButtonMenuItem("3D Perspective");
      menubarGroup.add(threeDItem);
      threeDItem.setSelected(true);
      ActionListener actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertTo3DPerspective();
         }
      };
      threeDItem.addActionListener(actionListener);
      menu.add(threeDItem);
      //3D no perspective
      threeDNPItem = new JRadioButtonMenuItem("3D No Perspective");
      menubarGroup.add(threeDNPItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertTo3D();
         }
      };
      threeDNPItem.addActionListener(actionListener);
      menu.add(threeDNPItem);
      //xy
      xyItem = new JRadioButtonMenuItem("XY");
      menubarGroup.add(xyItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToXY();
         }
      };
      xyItem.addActionListener(actionListener);
      menu.add(xyItem);
      //xz
      xzItem = new JRadioButtonMenuItem("XZ");
      menubarGroup.add(xzItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToXZ();
         }
      };
      xzItem.addActionListener(actionListener);
      menu.add(xzItem);
      // yz
      yzItem = new JRadioButtonMenuItem("YZ");
      menubarGroup.add(yzItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToYZ();
         }
      };
      yzItem.addActionListener(actionListener);
      menu.add(yzItem);
   }

   public void convertTo3DPerspective() {
      ((DrawingPanel3D) drawingPanel).setDisplayMode(DrawingPanel3D.DISPLAY_PERSPECTIVE);
      drawingPanel.repaint();
      threeDItem.setSelected(true);
   }

   public void convertTo3D() {
      ((DrawingPanel3D) drawingPanel).setDisplayMode(DrawingPanel3D.DISPLAY_NO_PERSPECTIVE);
      drawingPanel.repaint();
      threeDNPItem.setSelected(true);
   }

   public void convertToXY() {
      ((DrawingPanel3D) drawingPanel).setDisplayMode(DrawingPanel3D.DISPLAY_PLANAR_XY);
      drawingPanel.repaint();
      xyItem.setSelected(true);
   }

   public void convertToXZ() {
      ((DrawingPanel3D) drawingPanel).setDisplayMode(DrawingPanel3D.DISPLAY_PLANAR_XZ);
      drawingPanel.repaint();
      xzItem.setSelected(true);
   }

   public void convertToYZ() {
      ((DrawingPanel3D) drawingPanel).setDisplayMode(DrawingPanel3D.DISPLAY_PLANAR_YZ);
      drawingPanel.repaint();
      yzItem.setSelected(true);
   }

   /**
    * Set the angle (in radians) to rotate the scene horizontally before projecting. Default is 0.0.
    * @param alpha the desired angle
    */
   public void setAlpha(double alpha){
      ((DrawingPanel3D) drawingPanel).setAlpha(alpha);
   }

   /**
    * Get the angle (in degrees) to rotate the scene horizontally before projecting.
    * @return the current value
    */
   public double getAlpha(){
      return ((DrawingPanel3D) drawingPanel).getAlpha();
   }

   /**
    * Set the angle (in radians) to rotate the scene vertically before projecting. Default is 0.0.
    * @param _beta the desired angle
    */
   public void setBeta(double beta){
      ((DrawingPanel3D) drawingPanel).setBeta(beta);
   }

   /**
    * Get the angle (in radians) to rotate the scene vertically before projecting
    * @return the current value
    */
   public double getBeta(){
      return ((DrawingPanel3D) drawingPanel).getBeta();
   }

   /**
    * Set the angles (in radians) to rotate the scene horizontally and vertically before projecting
    * @param alpha the desired horizontal angle
    * @param beta the desired vertical angle
    */
   public void setAlphaAndBeta(double alpha, double beta){
      ((DrawingPanel3D) drawingPanel).setAlphaAndBeta(alpha,beta);
   }

   /**
    * Set the magnifying factor to apply to the scene. Default is 1.0.
    * @param _zoom the desired value
    */
   public void setZoom(double zoom){
      ((DrawingPanel3D) drawingPanel).setZoom(zoom);
   }

   /**
    * Get the magnifying factor applied to the scene.
    * @return the current value
    */
   public double getZoom(){
      return ((DrawingPanel3D) drawingPanel).getZoom();
   }

   /**
    * The style to display the cursor.One of the following
    * <ul>
    *   <li><b>CURSOR_NONE</b>: No cursor</li>
    *   <li><b>CURSOR_CUBE</b>: A cube from the minima to the cursor point</li>
    *   <li><b>CURSOR_XYZ</b>: An xyz line from the minima to the cursor point</li>
    *   <li><b>CURSOR_CROSHAIR</b>: Lines from the minima to the maxima through the cursor point</li>
    * </ul>
    * @param mode the desired value
    */
   public void setCursorMode(int mode){
     ((DrawingPanel3D) drawingPanel).setCursorMode(mode);
   }

   /**
    * The style to display the cursor
    * @return the current mode
    */
   public int getCursorMode(){
      return ((DrawingPanel3D) drawingPanel).getCursorMode();
   }


}
