package org.opensourcephysics.frames;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import org.opensourcephysics.display.*;
import org.opensourcephysics.display2d.*;

/**
 * A DrawingFrame that displays 2D plots of complex scalar fields.
 *
 * @author W. Christian
 * @version 1.0
 */
public class Complex2DFrame extends DrawingFrame {

   GridData gridData;
   Plot2D plot = new ComplexGridPlot(null);
   SurfacePlotMouseController surfacePlotMC;
   JMenuItem surfaceItem, gridItem, interpolatedItem;
   GridTableFrame tableFrame;

   /**
    * Constructs a Complex2DFrame with the given axes labels and frame title.
    * @param xlabel String
    * @param ylabel String
    * @param frameTitle String
    */
   public Complex2DFrame(String xlabel, String ylabel, String frameTitle) {
      super(new PlottingPanel(xlabel, ylabel, null));
      drawingPanel.setPreferredSize(new Dimension(350, 350));
      setTitle(frameTitle);
      ((PlottingPanel) drawingPanel).getAxes().setShowMajorXGrid(false);
      ((PlottingPanel) drawingPanel).getAxes().setShowMajorYGrid(false);
      drawingPanel.addDrawable(plot);
      addMenuItems();
      setAnimated(true);
      setAutoclear(true);
   }

   /**
    * Sets the autoscale flag and the floor and ceiling values for the intensity.
    *
    * @param isAutoscale
    * @param floor
    * @param ceil
    */
   public void setAutoscaleZ(boolean isAutoscale, double floor, double ceil){
      plot.setAutoscaleZ( isAutoscale,  floor,  ceil);
   }

   /**
    * Constructs a Complex2DFrame with the given frame title but without axes.
    * @param frameTitle String
    */
   public Complex2DFrame(String frameTitle) {
      super(new InteractivePanel());
      setTitle(frameTitle);
      drawingPanel.addDrawable(plot);
      addMenuItems();
      setAnimated(true);
      setAutoclear(true);
   }

   /**
    * Adds Views menu items on the menu bar.
    */
   protected void addMenuItems() {
      JMenuBar menuBar = getJMenuBar();
      if (menuBar==null){
         return;
      }
      JMenu helpMenu=this.removeMenu("Help");
      JMenu menu = getMenu("Views");
      if(menu==null) {
         menu = new JMenu("Views");
         menuBar.add(menu);
         menuBar.validate();
      } else { // add a separator if tools already exists
         menu.addSeparator();
      }
      menuBar.add(helpMenu);
      ButtonGroup menubarGroup = new ButtonGroup();
      //grid plot menu item
      gridItem = new JRadioButtonMenuItem("Grid Plot");
      menubarGroup.add(gridItem);
      gridItem.setSelected(true);
      ActionListener actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToGridPlot();
         }
      };
      gridItem.addActionListener(actionListener);
      menu.add(gridItem);
      //surface plot menu item
      surfaceItem = new JRadioButtonMenuItem("Surface Plot");
      menubarGroup.add(surfaceItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToSurfacePlot();
         }
      };
      surfaceItem.addActionListener(actionListener);
      menu.add(surfaceItem);
      //interpolated plot menu item
      interpolatedItem = new JRadioButtonMenuItem("Interpolated Plot");
      menubarGroup.add(interpolatedItem);
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            convertToInterpolatedPlot();
         }
      };
      interpolatedItem.addActionListener(actionListener);
      menu.add(interpolatedItem);
      // add phase legend to tool menu
      menu.addSeparator();
      JMenuItem legendItem = new JMenuItem("Phase Legend");
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            ComplexColorMapper.showPhaseLegend();
         }
      };
      legendItem.addActionListener(actionListener);
      menu.add(legendItem);
      // a grid data table
      JMenuItem tableItem = new JMenuItem("Data Table");
      tableItem.setAccelerator(KeyStroke.getKeyStroke('T', MENU_SHORTCUT_KEY_MASK));
      actionListener = new ActionListener() {

         public void actionPerformed(ActionEvent e) {
            showDataTable(true);
         }
      };
      tableItem.addActionListener(actionListener);
      menu.add(tableItem);
      // add data table to the popup menu
      if((drawingPanel!=null)&&(drawingPanel.getPopupMenu()!=null)) {
         JMenuItem item = new JMenuItem("Data Table");
         item.addActionListener(actionListener);
         drawingPanel.getPopupMenu().add(item);
      }
   }

   /**
    * Gets Drawable objects added by the user to this frame.
    *
    * @return the list
    */
   public synchronized ArrayList getDrawables() {
      ArrayList list = super.getDrawables();
      list.remove(plot);
      return list;
   }

   /**
    * Gets Drawable objects added by the user of an assignable type. The list contains
    * objects that are assignable from the class or interface.
    *
    * @param c the type of Drawable object
    *
    * @return the cloned list
    *
    * @see #getObjectOfClass(Class c)
    */
   public synchronized ArrayList getDrawables(Class c) {
      ArrayList list = super.getDrawables(c);
      list.remove(plot);
      return list;
   }

   /**
    * Removes drawable objects added by the user from this frame.
    */
   public void clearDrawables() {
      drawingPanel.clear(); // removes all drawables
      drawingPanel.addDrawable(plot);
   }

   /**
    * Clears data by setting the scalar field to zero.
    */
   public void clearData() {
      if(gridData!=null) {
         setAll(new double[2][gridData.getNx()][gridData.getNy()]);
      }
      drawingPanel.invalidateImage();
   }

   /*
    * Converts to an InterpolatedPlot plot.
    */
   public void convertToInterpolatedPlot() {
      if(!(plot instanceof ComplexInterpolatedPlot)) {
         if(surfacePlotMC!=null) {
            drawingPanel.removeMouseListener(surfacePlotMC);
            drawingPanel.removeMouseMotionListener(surfacePlotMC);
            surfacePlotMC = null;
         }
         drawingPanel.removeDrawable(plot);
         plot = new ComplexInterpolatedPlot(gridData);
         drawingPanel.addDrawable(plot);
         drawingPanel.invalidateImage();
         drawingPanel.repaint();
         interpolatedItem.setSelected(true);
      }
   }

   /*
    * Converts to a GridPlot plot.
    */
   public void convertToGridPlot() {
      if(!(plot instanceof ComplexGridPlot)) {
         if(surfacePlotMC!=null) {
            drawingPanel.removeMouseListener(surfacePlotMC);
            drawingPanel.removeMouseMotionListener(surfacePlotMC);
            surfacePlotMC = null;
         }
         drawingPanel.removeDrawable(plot);
         plot = new ComplexGridPlot(gridData);
         drawingPanel.addDrawable(plot);
         drawingPanel.invalidateImage();
         drawingPanel.repaint();
         gridItem.setSelected(true);
      }
   }

   /**
    * Converts to a SurfacePlot plot.
    */
   public void convertToSurfacePlot() {
      if(!(plot instanceof ComplexSurfacePlot)) {
         drawingPanel.removeDrawable(plot);
         try {
            plot = new ComplexSurfacePlot(gridData);
         } catch(IllegalArgumentException ex) {
            plot = null;
            surfaceItem.setEnabled(false);
            convertToGridPlot();
            return;
         }
         drawingPanel.addDrawable(plot);
         drawingPanel.invalidateImage();
         drawingPanel.repaint();
         if(surfacePlotMC==null) {
            surfacePlotMC = new SurfacePlotMouseController(drawingPanel, plot);
         }
         drawingPanel.addMouseListener(surfacePlotMC);
         drawingPanel.addMouseMotionListener(surfacePlotMC);
         surfaceItem.setSelected(true);
      }
   }

   /**
    * Resizes the grid used to store the field using the panel's preferred min/max values.
    *
    * @param nx int
    * @param ny int
    */
   public void resizeGrid(int nx, int ny) {
      double xmin, xmax, ymin, ymax;
      boolean cellScale = false;
      if(gridData==null) {
         xmin = drawingPanel.getPreferredXMin();
         xmax = drawingPanel.getPreferredXMax();
         ymin = drawingPanel.getPreferredYMin();
         ymax = drawingPanel.getPreferredYMax();
      } else {
         xmin = gridData.getLeft();
         xmax = gridData.getRight();
         ymin = gridData.getBottom();
         ymax = gridData.getTop();
         cellScale = gridData.isCellData();
      }
      gridData = new ArrayData(nx, ny, 3); // a grid with three data components
      gridData.setComponentName(0, "magnitude");
      gridData.setComponentName(1, "real");
      gridData.setComponentName(2, "imaginary");
      if(cellScale) {
         gridData.setCellScale(xmin, xmax, ymin, ymax);
      } else {
         gridData.setScale(xmin, xmax, ymin, ymax);
      }
      if(nx!=ny) {
         surfaceItem.setEnabled(false);
         if(plot instanceof ComplexSurfacePlot) {
            convertToGridPlot();
         }
      } else {
         surfaceItem.setEnabled(true);
      }
      plot.setGridData(gridData);
      plot.update();
      if((tableFrame!=null)&&tableFrame.isShowing()) {
         tableFrame.refreshTable();
      }
      drawingPanel.invalidateImage();
      drawingPanel.repaint();
   }

   /*
    * Sets the data in the given row.
    *
    * vals[0][] is assumed to contain the real components of the row.
    * vals[1][] is assumed to contain the imaginary components of the row.
    *
    *
    * @param row  int the index for this row
    * @param vals double[][] complex field values
    * @throws IllegalArgumentException if array length does not match grid size.
    */
   public void setRow(int row, double[][] vals) throws IllegalArgumentException {
      if(gridData.getNx()!=vals.length) {
         throw new IllegalArgumentException("Row data length does not match grid size.");
      }
      double[] re = gridData.getData()[1][row];
      double[] im = gridData.getData()[2][row];
      double[] phase = gridData.getData()[0][row];
      System.arraycopy(vals[0], 0, re, 0, vals.length);
      System.arraycopy(vals[1], 0, im, 0, vals.length);
      for(int j = 0, ny = phase.length; j<ny; j++) {
         phase[j] = Math.atan2(re[j], im[j]);
      }
      plot.update();
      if((tableFrame!=null)&&tableFrame.isShowing()) {
         tableFrame.refreshTable();
      }
      drawingPanel.invalidateImage();
   }

   /**
    * Sets the complex field's values and scale..
    *
    * @param val int[][] the new values
    * @param xmin double
    * @param xmax double
    * @param ymin double
    * @param ymax double
    */
   public void setAll(double[][][] vals, double xmin, double xmax, double ymin, double ymax) {
      setAll(vals);
      if(gridData.isCellData()) {
         gridData.setCellScale(xmin, xmax, ymin, ymax);
      } else {
         gridData.setScale(xmin, xmax, ymin, ymax);
      }
   }

   /**
    * Sets the complex field's values.
    *
    * vals[0][][] is assumed to contain the real components of the field.
    * vals[1][][] is assumed to contain the imaginary components of the field.
    *
    * @param vals double[][][] complex field values
    */
   public void setAll(double[][][] vals) {
      resizeGrid(vals[0].length, vals[0][0].length);
      double[][] mag = gridData.getData()[0];
      double[][] reData = gridData.getData()[1];
      double[][] imData = gridData.getData()[2];
      // current grid has correct size
      int ny = vals[0][0].length;
      for(int i = 0, nx = vals[0].length; i<nx; i++) {
         System.arraycopy(vals[0][i], 0, reData[i], 0, ny);
         System.arraycopy(vals[1][i], 0, imData[i], 0, ny);
         for(int j = 0; j<ny; j++) {
            mag[i][j] = Math.sqrt(vals[0][i][j]*vals[0][i][j]+vals[1][i][j]*vals[1][i][j]);
         }
      }
      plot.update();
      if((tableFrame!=null)&&tableFrame.isShowing()) {
         tableFrame.refreshTable();
      }
      drawingPanel.invalidateImage();
   }

   /**
    * Sets the comples field's data and scale.
    *
    * The array is assumed to contain complex numbers in row-major format.
    *
    * @param val int[][] the new values
    * @param nx
    * @param xmin double
    * @param xmax double
    * @param ymin double
    * @param ymax double
    */
   public void setAll(double[] vals, int nx, double xmin, double xmax, double ymin, double ymax) {
      if((vals.length/2)%nx!=0) {
         throw new IllegalArgumentException("Number of values in grid (nx*ny) must match number of values.");
      }
      resizeGrid(nx, vals.length/nx);
      setAll(vals);
      if(gridData.isCellData()) {
         gridData.setCellScale(xmin, xmax, ymin, ymax);
      } else {
         gridData.setScale(xmin, xmax, ymin, ymax);
      }
   }

   /**
    * Sets the comples field's data using the given array.
    *
    * The array is assumed to contain complex numbers in row-major format.
    *
    * @param vals double[] complex field values
    */
   public void setAll(double[] vals) {
      if(gridData==null) {
         throw new IllegalArgumentException("Grid size must be set before using row-major format.");
      }
      int nx = gridData.getNx(), ny = gridData.getNy();
      if(vals.length!=2*nx*ny) {
         throw new IllegalArgumentException("Grid does not have the correct size.");
      }
      double[][] mag = gridData.getData()[0]; // magnitude maps to color
      double[][] reData = gridData.getData()[1];
      double[][] imData = gridData.getData()[2];
      for(int j = 0; j<ny; j++) {
         int offset = 2*j*nx;
         for(int i = 0; i<nx; i++) {
            double re = vals[offset+2*i];
            double im = vals[offset+2*i+1];
            mag[i][j] = Math.sqrt(re*re+im*im);
            reData[i][j] = re;
            imData[i][j] = im;
         }
      }
      plot.update();
      if((tableFrame!=null)&&tableFrame.isShowing()) {
         tableFrame.refreshTable();
      }
      drawingPanel.invalidateImage();
   }

   /**
    * Shows or hides the data table.
    *
    * @param show boolean
    */
   public synchronized void showDataTable(boolean show) {
      if(show) {
         if(tableFrame==null) {
            if(gridData==null) {
               return;
            }
            tableFrame = new GridTableFrame(gridData);
            tableFrame.setTitle("Complex Field Data");
            tableFrame.setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);
         }
         tableFrame.refreshTable();
         tableFrame.setVisible(true);
      } else {
         tableFrame.setVisible(false);
         tableFrame.dispose();
         tableFrame = null;
      }
   }

   /**
    * Gets the x coordinate for the given index.
    *
    * @param i int
    * @return double the x coordiante
    */
   public double indexToX(int i) {
      if(gridData==null) {
         throw new IllegalStateException("Data has not been set.  Invoke setAll before invoking this method.");
      }
      return gridData.indexToX(i);
   }

   /**
    * Gets the index that is closest to the given x value
    *
    * @param i int
    * @return double the x coordiante
    */
   public int xToIndex(double x) {
      if(gridData==null) {
         throw new IllegalStateException("Data has not been set.  Invoke setAll before invoking this method.");
      }
      return gridData.xToIndex(x);
   }

   /**
    * Gets the index that is closest to the given x value
    *
    * @param i int
    * @return double the x coordiante
    */
   public int yToIndex(double y) {
      if(gridData==null) {
         throw new IllegalStateException("Data has not been set.  Invoke setAll before invoking this method.");
      }
      return gridData.yToIndex(y);
   }

   /**
    * Gets the y coordinate for the given index.
    *
    * @param i int
    * @return double the y coordiante
    */
   public double indexToY(int i) {
      if(gridData==null) {
         throw new IllegalStateException("Data has not been set.  Invoke setAll before invoking this method.");
      }
      return gridData.indexToY(i);
   }
}
