/*
 * The value package contains utilities to work with primitives
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.value;

import org.opensourcephysics.ejs.control.GroupControl;

 /**
  * A <code>ExpressionValue</code> is a <code>Value</code> object that
  * holds an expression is parsed into a double.
  * <p>
  * @see     Value
  */
public class ExpressionValue extends Value {
  private String expression;
  private GroupControl group;
  private ParserSuryono parser;
  private String vars[];
  // Now consider the case when it is an array
  private boolean isArray;
  private ParserSuryono[] arrayParser;
  private String[][] arrayVars;
  private double[] arrayValues;

  public ExpressionValue(String _expression, GroupControl _group) {
    group = _group;
    expression = new String(_expression.trim());
    processExpression();
  }

  public boolean getBoolean() { return (getDouble()!=0); }

  public int     getInteger() { return (int) getDouble(); }

  public double  getDouble()  {
    for (int i=0, n=vars.length; i<n; i++) parser.setVariable(i,group.getDouble(vars[i]));
    return parser.evaluate();
  }

  public String  getString()  { return String.valueOf(getDouble()); }

  public Object  getObject()  {
    if (isArray) {
      for (int k=0, m=arrayVars.length; k<m; k++) {
        for (int i=0, n=arrayVars[k].length; i<n; i++) arrayParser[k].setVariable(i,group.getDouble(arrayVars[k][i]));
        arrayValues[k] = arrayParser[k].evaluate();
      }
      return arrayValues;
    }
    return null;
  }

  public void setExpression(String _expression) {
    expression = new String(_expression.trim());
    processExpression();
  }

  public void copyValue(Value _source) {
    if (_source instanceof ExpressionValue) expression = new String(((ExpressionValue)_source).expression);
    else expression = new String(_source.getString());
    processExpression();
  }

  public Value cloneValue() { return new ExpressionValue(expression,group); }

  private void processExpression () {
    if (expression.startsWith("{") && expression.endsWith("}")) { // An explicit array of constants
      String text  = expression.substring(1,expression.length()-1);
      java.util.StringTokenizer tkn = new java.util.StringTokenizer(text,",");
      int dim = tkn.countTokens();
      arrayParser = new ParserSuryono[dim];
      arrayVars   = new String[dim][];
      arrayValues = new double[dim];
      isArray = true;
      // Prepare the parsers
      int k = 0;
      while (tkn.hasMoreTokens()) {
        String token = tkn.nextToken();
        arrayVars[k] = ParserSuryono.getVariableList (token);
        arrayParser[k] = new ParserSuryono(arrayVars[k].length);
        for (int i=0, n=arrayVars[k].length; i<n; i++) arrayParser[k].defineVariable (i,arrayVars[k][i]);
        arrayParser[k].define(token);
        arrayParser[k].parse();
        k++;
      }
    }
    else { // A single variable or expression. Taken to be double
      vars = ParserSuryono.getVariableList (expression);
      parser = new ParserSuryono(vars.length);
      for (int i=0, n=vars.length; i<n; i++) parser.defineVariable (i,vars[i]);
      parser.define(expression);
      parser.parse();
      isArray = false;
    }
  }
}

