/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.awt.Color;

import javax.swing.JTextField;

import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.StringValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A textfield to display double values. When this value changes,
 * it invokes both the VARIABLE_CHANGED and the ACTION actions.
 */
public class ControlTextField extends ControlSwingElement {
  static private final int VARIABLE  = 0;
  static private final int BACKGROUND  = ControlSwingElement.BACKGROUND+4;

  protected JTextField textfield;
  private StringValue internalValue;
  private boolean defaultValueSet;
  private String defaultValue;
  private Color defaultColor, editingColor, errorColor;

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlTextField (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof JTextField) textfield = (JTextField) _visual;
    else {
      textfield = new JTextField();
      textfield.setText ("");
    }
    defaultValue = textfield.getText();
    textfield.addActionListener (new MyActionListener());
    textfield.addKeyListener    (new MyKeyListener());
    defaultValueSet = false;
    internalValue = new StringValue(defaultValue);
    decideColors (textfield.getBackground());
    return textfield;
  }

  public void reset() {
    if (defaultValueSet) {
      setTheValue (defaultValue);
      setInternalValue (defaultValue);
    }
  }

  private void setTheValue (String _value) {
    if (internalValue.value.equals(_value)) return;
    textfield.setText (internalValue.value = _value);
    setColor (defaultColor);
  }

  private void setInternalValue (String _value) {
    internalValue.value = _value;
    variableChanged (VARIABLE,internalValue);
    invokeActions ();
  }

// ------------------------------------------------
// Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("value");
      infoList.add ("editable");
      infoList.add ("action");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))       return "String VARIABLE_EXPECTED";
    if (_property.equals("value"))          return "String CONSTANT";
    if (_property.equals("editable"))       return "boolean";
    if (_property.equals("action"))         return "Action CONSTANT";
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case VARIABLE : setTheValue(_value.getString()); break;
      case 1 :
        defaultValueSet = true; defaultValue = _value.getString();
        setActive (false); reset (); setActive(true);
        break;
      case 2 : textfield.setEditable(_value.getBoolean()); break;
      case 3 : // action
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;
      case BACKGROUND :
        super.setValue (ControlSwingElement.BACKGROUND,_value);
        decideColors (getVisual().getBackground());
        break;
      default: super.setValue(_index-4,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case VARIABLE : break; // Do nothing
      case 1 : defaultValueSet = false; break;
      case 2 : textfield.setEditable(true); break;
      case 3 : removeAction (ControlElement.ACTION,getProperty("action")); break;
      case BACKGROUND :
        super.setDefaultValue (ControlSwingElement.BACKGROUND);
        decideColors (getVisual().getBackground());
        break;
      default: super.setDefaultValue(_index-4); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case VARIABLE : return internalValue;
      case 1 : case 2 : case 3 :
        return null;
      default: return super.getValue(_index-4);
    }
  }

// -------------------------------------
// Private methods and inner classes
// -------------------------------------

  private void setColor (Color aColor) {
    if (textfield.isEditable()) getVisual().setBackground (aColor);
  }

  private void decideColors (Color aColor) {
    if (aColor==null) return;
    defaultColor = aColor;
    if (defaultColor.equals(Color.yellow)) editingColor = Color.orange;
    else editingColor = Color.yellow;
    if (defaultColor.equals(Color.red)) errorColor = Color.magenta;
    else errorColor = Color.red;
  }

  private class MyActionListener implements java.awt.event.ActionListener {
    public void actionPerformed (java.awt.event.ActionEvent _e) {
      setInternalValue (textfield.getText());
      setColor (defaultColor);
    }
  }

  private class MyKeyListener implements java.awt.event.KeyListener {
    public void keyPressed  (java.awt.event.KeyEvent _e) { processKeyEvent (_e,0); }
    public void keyReleased (java.awt.event.KeyEvent _e) { processKeyEvent (_e,1); }
    public void keyTyped    (java.awt.event.KeyEvent _e) { processKeyEvent (_e,2); }
    private void processKeyEvent (java.awt.event.KeyEvent _e, int _n) {
      if (!textfield.isEditable()) return;
      if (_e.getKeyChar()!='\n') setColor (editingColor);
      if (_e.getKeyCode()==27)   setValue (VARIABLE,internalValue);
    }
  }

} // End of class