/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import javax.swing.JSlider;
import javax.swing.border.EtchedBorder;
import javax.swing.border.TitledBorder;

import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A slider to display double values. When the value is changing it
 * invokes all VARIABLE_CHANGED actions. When the value is finally set,
 * it invokes all ACTION actions.
 */
public class ControlSlider extends ControlSwingElement {
  static private final int RESOLUTION=100000;
  static private final int VARIABLE  = 0;

  protected JSlider slider;
  private DoubleValue internalValue;
  private boolean recalculate = true, defaultValueSet;
  private int ticks=0;
  private double defaultValue, scale, minimum=0.0, maximum=1.0;
  private TitledBorder titledBorder;
  private EtchedBorder etchedBorder;
  private java.text.DecimalFormat format=null, ticksFormat=null;

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlSlider (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof JSlider) slider = (JSlider) _visual;
    else {
      slider = new JSlider();
      slider.setPaintLabels(false);
      slider.setPaintTicks(false);
      slider.setPaintTrack(true);
  }
    slider.setMinimum (0);
    slider.setMaximum (RESOLUTION);
    slider.setValue (0);
    etchedBorder = new EtchedBorder(EtchedBorder.LOWERED);
    titledBorder = new TitledBorder (etchedBorder,"");
    titledBorder.setTitleJustification (TitledBorder.CENTER);
    slider.setBorder (etchedBorder);

    defaultValue  = 0.0;
    defaultValueSet = false;
    internalValue = new DoubleValue(defaultValue);
    minimum = 0.0;
    maximum = 1.0;

    scale = RESOLUTION*(maximum-minimum);
    setMaximum(maximum);
    internalValue.value = minimum + slider.getValue()/scale;
    slider.addChangeListener (new MyChangeListener());
    slider.addMouseListener  (new MyMouseListener());
    return slider;
  }

  private void setTheValue (double val) {
    internalValue.value = val;
    recalculate = false;
    slider.setValue ((int) ((internalValue.value-minimum)*scale));
    recalculate = true;
    if (format!=null) {
      titledBorder.setTitle (format.format (internalValue.value));
      slider.repaint();
    }
  }

  public void reset() {
    if (defaultValueSet) {
      setTheValue (defaultValue);
      variableChanged (VARIABLE,internalValue);
    }
  }

// ------------------------------------------------
// Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("value");
      infoList.add ("minimum");
      infoList.add ("maximum");
      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");

      infoList.add ("format");
      infoList.add ("ticks");
      infoList.add ("ticksFormat");
      infoList.add ("closest");
      infoList.add ("orientation");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))       return "int|double";
    if (_property.equals("value"))          return "int|double";
    if (_property.equals("minimum"))        return "int|double";
    if (_property.equals("maximum"))        return "int|double";
    if (_property.equals("pressaction"))    return "Action CONSTANT";
    if (_property.equals("dragaction"))     return "Action CONSTANT";
    if (_property.equals("action"))         return "Action CONSTANT";
    if (_property.equals("format"))         return "Format|Object";
    if (_property.equals("ticks"))          return "int    BASIC";
    if (_property.equals("ticksFormat"))    return "Format|Object BASIC";
    if (_property.equals("closest"))        return "boolean BASIC";
    if (_property.equals("orientation"))    return "Orientation|int BASIC";
    if (_property.equals("enabled"))        return "boolean"; // Not hidden
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case VARIABLE : if (internalValue.value!=_value.getDouble()) setTheValue(_value.getDouble()); break;
      case 1 :
        defaultValueSet = true; defaultValue = _value.getDouble();
        setActive (false); reset (); setActive(true);
        break;
      case 2 : setMinimum (_value.getDouble()); break;
      case 3 : setMaximum (_value.getDouble()); break;
      case 4 : // pressaction
        removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlSwingElement.ACTION_PRESS,_value.getString());
        break;
      case 5 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        break;
      case 6 : // pressaction
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;
      case 7 :
        if (_value.getObject() instanceof java.text.DecimalFormat) {
          if (format==(java.text.DecimalFormat) _value.getObject()) return;
          format = (java.text.DecimalFormat) _value.getObject();
          titledBorder.setTitle (format.format (internalValue.value));
          slider.setBorder (titledBorder);
        }
        break;
      case 8 : if (_value.getInteger()!=ticks) { ticks = _value.getInteger(); setTicks (); } break;
      case 9 :
        if (_value.getObject() instanceof java.text.DecimalFormat) {
          if (ticksFormat==(java.text.DecimalFormat) _value.getObject()) return;
          ticksFormat = (java.text.DecimalFormat) _value.getObject();
          slider.setPaintLabels(true);
          setTicks();
        }
        break;
      case 10 : slider.setSnapToTicks (_value.getBoolean()); break;
      case 11 :
        if (slider.getOrientation()!=_value.getInteger()) slider.setOrientation(_value.getInteger());
        break;
      default: super.setValue(_index-12,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case VARIABLE : break; // Do nothing
      case 1 : defaultValueSet = false; break;
      case 2 : setMinimum (0.0); break;
      case 3 : setMaximum (1.0); break;
      case 4 : removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction")); break;
      case 5 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));   break;
      case 6 : removeAction (ControlElement.ACTION,getProperty("action"));                 break;
      case 7 : format = null; slider.setBorder (etchedBorder); break;
      case 8 : ticks = 0; setTicks (); break;
      case 9 : ticksFormat = null; slider.setPaintLabels(false); setTicks(); break;
      case 10 : slider.setSnapToTicks (false); break;
      case 11 : slider.setOrientation(JSlider.HORIZONTAL); break;
      default: super.setDefaultValue(_index-12); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case VARIABLE : return internalValue;
      case 1 :  case 2 :  case 3 :  case 4 : case 5 :
      case 6 :  case 7 :  case 8 :  case 9 : case 10 :
      case 11 :
        return null;
      default: return super.getValue(_index-12);
    }
  }


// -------------------------------------
// Private methods
// -------------------------------------

  private void setTicks () {
    if (ticks<2) { slider.setPaintTicks(false); return; }
    int spacing = RESOLUTION/(ticks-1);
    slider.setMinorTickSpacing (spacing);
    slider.setMajorTickSpacing (2*spacing);
    slider.setPaintTicks(true);
    if (ticksFormat!=null) {
      java.util.Hashtable table = new java.util.Hashtable ();
      for (int i=0; i<=RESOLUTION; i+=2*spacing)
        table.put (new Integer(i),new javax.swing.JLabel(ticksFormat.format(minimum+i/scale)));
      slider.setLabelTable (table);
    }
  }

  private void setMinimum (double val) {
    if (val==minimum) return;
    minimum = val;
    if (minimum>=maximum) maximum = minimum+1.0;
//    internalValue.value = minimum;
    scale = 1.0*RESOLUTION/(maximum-minimum);
    setTicks ();
    setTheValue (internalValue.value);
  }

  private void setMaximum (double val) {
    if (val==maximum) return;
    maximum = val;
    if (minimum>=maximum) minimum = maximum-1.0;
//    internalValue.value = minimum;
    scale = 1.0*RESOLUTION/(maximum-minimum);
    setTicks ();
    setTheValue (internalValue.value);
  }

// -------------------------------------
// Inner classes
// -------------------------------------

  private boolean adjusted = false;

  private class MyChangeListener implements javax.swing.event.ChangeListener {
    public void stateChanged(javax.swing.event.ChangeEvent e) {
      if (recalculate) {
        double value = minimum + slider.getValue()/scale;
//        if (internalValue.value==value) return;
        internalValue.value = value;
        if (format!=null) {
          titledBorder.setTitle (format.format (internalValue.value));
          slider.repaint();
        }
        variableChanged (VARIABLE,internalValue);
      }
    }
  }

  private class MyMouseListener extends java.awt.event.MouseAdapter {
    public void mousePressed (java.awt.event.MouseEvent evt) {
      invokeActions (ControlSwingElement.ACTION_PRESS);
    }
    public void mouseReleased (java.awt.event.MouseEvent evt) {
      invokeActions (ControlElement.ACTION);
    }
  }

} // End of class