/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Font;

import org.opensourcephysics.display.Interactive;
import org.opensourcephysics.display.InteractiveMouseHandler;
import org.opensourcephysics.display.InteractivePanel;
import org.opensourcephysics.display.PlottingPanel;
import org.opensourcephysics.display.axes.AxisFactory;
import org.opensourcephysics.display.axes.CartesianAxes;
import org.opensourcephysics.display.axes.CartesianType1;
import org.opensourcephysics.display.axes.CartesianType2;
import org.opensourcephysics.display.axes.CartesianType3;
import org.opensourcephysics.display.axes.DrawableAxes;
import org.opensourcephysics.display.axes.PolarAxes;
import org.opensourcephysics.display.axes.PolarType1;
import org.opensourcephysics.display.axes.PolarType2;
import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionTarget;
import org.opensourcephysics.displayejs.Point3D;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.IntegerValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A configurable plottingPanel.
 */
public class ControlPlottingPanel extends ControlDrawablesParent implements InteractiveMouseHandler {
  static final protected int ADDEDBYPLOTTINGPANEL = 28;
  static final int FONT  = ControlSwingElement.FONT+ADDEDBYPLOTTINGPANEL;
  static private final int[] posIndex = {18,19};

  protected PlottingPanel plottingPanel;
  private String title, titleFontname;

  private double minX, maxX,minY, maxY;
  private java.awt.Rectangle myGutters;
  // All about axes
  private DrawableAxes axes;
  private boolean axisGridX, axisGridY, xaxisLog, yaxisLog;
  private int axesType;
  private double xaxisPos, yaxisPos;
  private double deltaR, deltaTheta;
  private String xLabel, yLabel, labelFontname;

  private DoubleValue[] posValues ={ new DoubleValue(0.0), new DoubleValue(0.0)};

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlPlottingPanel (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof PlottingPanel) {
      plottingPanel = (PlottingPanel) _visual;
    }
    else {
      plottingPanel = new PlottingPanel ("", "", "");
      plottingPanel.enableInspector(false);
      plottingPanel.setSquareAspect (false);
      plottingPanel.setAutoscaleX (true);
      plottingPanel.setAutoscaleY (true);
    }
    plottingPanel.removeOptionController();
    axes = plottingPanel.getAxes();
    axes.setVisible(true);
    axes.setShowMajorXGrid(axisGridX = true);
    axes.setShowMajorYGrid(axisGridY = true);
    xaxisLog = yaxisLog = false;
    deltaR = 1; deltaTheta = Math.PI/8;
    if (axes instanceof CartesianAxes) {
      xaxisLog = ((CartesianAxes) axes).isXLog();
      yaxisLog = ((CartesianAxes) axes).isYLog();
      if      (axes instanceof CartesianType1) axesType = 1;
      else if (axes instanceof CartesianType2) axesType = 2;
      else if (axes instanceof CartesianType3) axesType = 3;
      else axesType = 1; // default for CartesianAxes
    }
    else if (axes instanceof PolarAxes) {
      deltaR     = ((PolarAxes) axes).getDeltaR();
      deltaTheta = ((PolarAxes) axes).getDeltaTheta();
      if      (axes instanceof PolarType1) axesType = 4;
      else if (axes instanceof PolarType2) axesType = 5;
      else axesType = 4; // default for PolarAxes
    }
    else { // Don't touch anything
      axesType = 0;
    }
    minX = plottingPanel.getXMin();
    maxX = plottingPanel.getXMax();
    minY = plottingPanel.getYMin();
    maxY = plottingPanel.getYMax();
    plottingPanel.setInteractiveMouseHandler(this);
    return plottingPanel;
  }

  protected int[] getPosIndex () { return posIndex; } // in case it should be overriden

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("title");
      infoList.add ("titleFont");

      infoList.add ("axesType"); // cartesian1, cartesian2, cartesian3, polar1, polar2
      infoList.add ("titleX");
      infoList.add ("titleY");
      infoList.add ("xaxisType");  // log10 or linear for cartesian types
      infoList.add ("yaxisType");  // log10 or linear for cartesian types
      infoList.add ("deltaR");     // In the case of polar coordinates
      infoList.add ("deltaTheta");
      infoList.add ("interiorBackground");
      infoList.add ("majorTicksX");
      infoList.add ("majorTicksY");

      infoList.add ("autoscaleX");
      infoList.add ("autoscaleY");
      infoList.add ("minimumX");
      infoList.add ("maximumX");
      infoList.add ("minimumY");
      infoList.add ("maximumY");
      infoList.add ("x");
      infoList.add ("y");
      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");
      infoList.add ("square");
      infoList.add ("showCoordinates");
      infoList.add ("gutters");

      infoList.add ("xaxisPos");
      infoList.add ("yaxisPos");

      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("title"))          return "String";
    if (_property.equals("titleFont"))      return "Font|Object BASIC";

    if (_property.equals("axesType"))       return "int|AxesType BASIC";
    if (_property.equals("titleX"))         return "String";
    if (_property.equals("titleY"))         return "String";
    if (_property.equals("xaxisType"))      return "int|CartesianAxisType BASIC";
    if (_property.equals("yaxisType"))      return "int|CartesianAxisType BASIC";
    if (_property.equals("deltaR"))         return "int|double BASIC";
    if (_property.equals("deltaTheta"))     return "int|double BASIC";
    if (_property.equals("interiorBackground")) return "Color|Object BASIC";
    if (_property.equals("majorTicksX"))     return "boolean BASIC";
    if (_property.equals("majorTicksY"))     return "boolean BASIC";

    if (_property.equals("autoscaleX"))     return "boolean";
    if (_property.equals("autoscaleY"))     return "boolean";
    if (_property.equals("minimumX"))       return "int|double";
    if (_property.equals("maximumX"))       return "int|double";
    if (_property.equals("minimumY"))       return "int|double";
    if (_property.equals("maximumY"))       return "int|double";
    if (_property.equals("x"))              return "int|double";
    if (_property.equals("y"))              return "int|double";
    if (_property.equals("action"))         return "Action CONSTANT";
    if (_property.equals("pressaction"))    return "Action CONSTANT";
    if (_property.equals("dragaction"))     return "Action CONSTANT";
    if (_property.equals("square"))         return "boolean BASIC";
    if (_property.equals("showCoordinates"))return "boolean BASIC";
    if (_property.equals("gutters"))        return "Margins|Object BASIC";

    if (_property.equals("xaxisPos"))          return "int|double BASIC";
    if (_property.equals("yaxisPos"))          return "int|double BASIC";

    return super.getPropertyInfo(_property);
  }

  public Value parseConstant (String _propertyType, String _value) {
    if (_value==null) return null;
    if (_propertyType.indexOf("CartesianAxisType")>=0) {
      _value = _value.trim().toLowerCase();
      if (_value.equals("linear")) return new IntegerValue (0);
      if (_value.equals("log10"))  return new IntegerValue (1);
    }
    if (_propertyType.indexOf("AxesType")>=0) {
      _value = _value.trim().toLowerCase();
      if (_value.equals("cartesian1")) return new IntegerValue (1);
      if (_value.equals("cartesian2")) return new IntegerValue (2);
      if (_value.equals("cartesian3")) return new IntegerValue (3);
      if (_value.equals("polar1"))     return new IntegerValue (4);
      if (_value.equals("polar2"))     return new IntegerValue (5);
    }
    return super.parseConstant (_propertyType,_value);
  }

  // Backwards compatibility
  public ControlElement setProperty(String _property, String _value) {
    _property = _property.trim();
    if (_property.equals("xaxis")) return super.setProperty ("xaxisPos",_value);
    if (_property.equals("yaxis")) return super.setProperty ("yaxisPos",_value);
    return super.setProperty(_property,_value);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------


  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 : plottingPanel.setTitle(title=_value.getString(),titleFontname); break;
      case 1 :
        if (_value.getObject() instanceof Font) {
          Font font = (Font) _value.getObject();
          titleFontname = font.getFamily();
          if (font.isPlain()) titleFontname += "-PLAIN";
          else if (font.isItalic()) {
            if (font.isBold()) titleFontname += "-BOLDITALIC";
            else titleFontname += "-ITALIC";
          }
          else if (font.isBold()) titleFontname += "-BOLD";
          titleFontname += "-" + font.getSize();
          if (title!=null) axes.setTitle(title,titleFontname);
        }
        break;
      case 2 :
        if (axesType!=_value.getInteger()) { // Create new axes
          switch (axesType = _value.getInteger()) {
            default :
            case 1 : axes = AxisFactory.createAxesType1(plottingPanel); ((CartesianType1)axes).setXLog(xaxisLog); ((CartesianType1)axes).setYLog(yaxisLog); break;
            case 2 : axes = AxisFactory.createAxesType2(plottingPanel); ((CartesianType2)axes).setXLog(xaxisLog); ((CartesianType2)axes).setYLog(yaxisLog); break;
            case 3 : axes = AxisFactory.createAxesType3(plottingPanel); ((CartesianType3)axes).setXLog(xaxisLog); ((CartesianType3)axes).setYLog(yaxisLog); break;
            case 4 : axes = new PolarType1(plottingPanel); ((PolarAxes)axes).setDeltaR(deltaR); ((PolarAxes)axes).setDeltaTheta(deltaTheta); break;
            case 5 : axes = new PolarType2(plottingPanel); ((PolarAxes)axes).setDeltaR(deltaR); ((PolarAxes)axes).setDeltaTheta(deltaTheta); break;
          }
          if (xLabel!=null) axes.setXLabel(xLabel, labelFontname);
          if (yLabel!=null) axes.setYLabel(yLabel, labelFontname);
          if (title!=null)  axes.setTitle(title, titleFontname);
          axes.setShowMajorXGrid(axisGridX);
          axes.setShowMajorYGrid(axisGridY);
          if (axes instanceof CartesianAxes) {
            ((CartesianAxes) axes).setX(xaxisPos);
            ((CartesianAxes) axes).setY(yaxisPos);
          }
          plottingPanel.setAxes(axes);
        }
        break;

      case 3 : plottingPanel.setXLabel(xLabel=_value.getString(),labelFontname); break;
      case 4 : plottingPanel.setYLabel(yLabel=_value.getString(),labelFontname); break;

      case 5 : if (xaxisLog && _value.getInteger()!=1) plottingPanel.setLogScale(xaxisLog = false, yaxisLog);
               else if (!xaxisLog && _value.getInteger()==1) plottingPanel.setLogScale(xaxisLog = true, yaxisLog);
               break;
      case 6 : if (yaxisLog && _value.getInteger()!=1) plottingPanel.setLogScale(xaxisLog, yaxisLog = false);
               else if (!yaxisLog && _value.getInteger()==1) plottingPanel.setLogScale(xaxisLog, yaxisLog = true);
               break;
      case 7 : if (_value.getDouble()!=deltaR) {
                 deltaR = _value.getDouble();
                 if (axes instanceof PolarAxes) ((PolarAxes)axes).setDeltaR(deltaR);
               }
               break;
      case 8 : if (_value.getDouble()!=deltaTheta) {
                 deltaTheta = _value.getDouble();
                 if (axes instanceof PolarAxes) ((PolarAxes)axes).setDeltaTheta(deltaTheta);
              }
              break;
      case 9 : if (_value.getObject() instanceof Color) axes.setInteriorBackground((Color) _value.getObject()); break;

      case 10 : axes.setShowMajorXGrid(_value.getBoolean()); break;
      case 11 : axes.setShowMajorYGrid(_value.getBoolean()); break;

      case 12 : plottingPanel.setAutoscaleX(_value.getBoolean()); break;
      case 13 : plottingPanel.setAutoscaleY(_value.getBoolean()); break;
      case 14 : if (_value.getDouble()!=minX) plottingPanel.setPreferredMinMaxX(minX=_value.getDouble(),maxX); break;
      case 15 : if (_value.getDouble()!=maxX) plottingPanel.setPreferredMinMaxX(minX,maxX=_value.getDouble()); break;
      case 16 : if (_value.getDouble()!=minY) plottingPanel.setPreferredMinMaxY(minY=_value.getDouble(),maxY); break;
      case 17 : if (_value.getDouble()!=maxY) plottingPanel.setPreferredMinMaxY(minY,maxY=_value.getDouble()); break;

      case 18 : posValues[0].value = _value.getDouble(); break;
      case 19 : posValues[1].value = _value.getDouble(); break;
      case 20 : // pressaction
        removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlSwingElement.ACTION_PRESS,_value.getString());
        break;
      case 21 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        break;
      case 22 : // pressaction
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;

      case 23 : plottingPanel.setSquareAspect(_value.getBoolean()); break;
      case 24 : plottingPanel.setShowCoordinates(_value.getBoolean()); break;
      case 25 :
        if (_value.getObject() instanceof java.awt.Rectangle) {
          java.awt.Rectangle rect = (java.awt.Rectangle) _value.getObject();
          if (rect!=myGutters) {
            plottingPanel.setGutters(rect.x,rect.y,rect.width,rect.height);
            myGutters = rect;
          }
        }
        break;
      case 26 : if (xaxisPos!=_value.getDouble() && axes instanceof CartesianAxes) ((CartesianAxes)axes).setX(xaxisPos=_value.getDouble()); break;
      case 27 : if (yaxisPos!=_value.getDouble() && axes instanceof CartesianAxes) ((CartesianAxes)axes).setY(yaxisPos=_value.getDouble()); break;

      default: super.setValue(_index-ADDEDBYPLOTTINGPANEL,_value); break;
      case FONT :
        if (_value.getObject() instanceof Font) {
          Font font = (Font) _value.getObject();
          labelFontname = font.getFamily();
          if (font.isPlain()) labelFontname += "-PLAIN";
          else if (font.isItalic()) {
            if (font.isBold()) labelFontname += "-BOLDITALIC";
            else labelFontname += "-ITALIC";
          }
          else if (font.isBold()) labelFontname += "-BOLD";
          labelFontname += "-" + font.getSize();
          if (xLabel!=null) axes.setXLabel(xLabel, labelFontname);
          if (yLabel!=null) axes.setYLabel(yLabel, labelFontname);
        }
        super.setValue (ControlSwingElement.FONT,_value);
        break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {

      case 0 : plottingPanel.setTitle(title="",titleFontname); break;
      case 1 : titleFontname = "Helvetica-BOLD-14"; if (title!=null) axes.setTitle(title,titleFontname); break;

      case 2 :
        axesType = 1;
        axes = AxisFactory.createAxesType1(plottingPanel); ((CartesianType1)axes).setXLog(xaxisLog); ((CartesianType1)axes).setYLog(yaxisLog);
        if (xLabel!=null) axes.setXLabel(xLabel, labelFontname);
        if (yLabel!=null) axes.setYLabel(yLabel, labelFontname);
        if (title!=null)  axes.setTitle(title, titleFontname);
        axes.setShowMajorXGrid(axisGridX);
        axes.setShowMajorYGrid(axisGridY);
        // plottingPanel.setAxes(axes);
        break;

      case 3 : plottingPanel.setXLabel(xLabel="",labelFontname); break;
      case 4 : plottingPanel.setYLabel(yLabel="",labelFontname); break;
      case 5 : plottingPanel.setLogScale(xaxisLog = false, yaxisLog); break;
      case 6 : plottingPanel.setLogScale(xaxisLog, yaxisLog = false); break;
      case 7 : deltaR = 1; if (axes instanceof PolarAxes) ((PolarAxes)axes).setDeltaR(deltaR); break;
      case 8 : deltaTheta = Math.PI/8; if (axes instanceof PolarAxes) ((PolarAxes)axes).setDeltaTheta(deltaTheta); break;
      case 9 : axes.setInteriorBackground(Color.white); break;

      case 10 : axes.setShowMajorXGrid(true); break;
      case 11 : axes.setShowMajorYGrid(true); break;

      case 12 : plottingPanel.setAutoscaleX(false); break;
      case 13 : plottingPanel.setAutoscaleY(false); break;
      case 14 : plottingPanel.setPreferredMinMaxX(minX=0.0,maxX); break;
      case 15 : plottingPanel.setPreferredMinMaxX(minX,maxX=1.0); break;
      case 16 : plottingPanel.setPreferredMinMaxY(minY=0.0,maxY); break;
      case 17 : plottingPanel.setPreferredMinMaxY(minY,maxY=1.0); break;

      case 18 : posValues[0].value = (minX+maxX)/2.0; break; // x
      case 19 : posValues[1].value = (minY+maxY)/2.0; break; // y
      case 20 : removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction")); break;
      case 21 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));   break;
      case 22 : removeAction (ControlElement.ACTION,getProperty("action"));                break;

      case 23 : plottingPanel.setSquareAspect(false);  break;
      case 24 : plottingPanel.setShowCoordinates(true); break;
      case 25 : plottingPanel.setGutters(0,0,0,0); myGutters = null; break;

      case 26 : if (axes instanceof CartesianAxes) ((CartesianAxes)axes).setX(xaxisPos=Double.NaN); break;
      case 27 : if (axes instanceof CartesianAxes) ((CartesianAxes)axes).setY(yaxisPos=Double.NaN); break;

      default: super.setDefaultValue(_index-ADDEDBYPLOTTINGPANEL); break;
      case FONT :
        labelFontname = "Helvetica-PLAIN-12";
        if (xLabel!=null) axes.setXLabel(xLabel, labelFontname);
        if (yLabel!=null) axes.setYLabel(yLabel, labelFontname);
        super.setDefaultValue (ControlSwingElement.FONT);
        break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case  0 : case  1 : case  2 : case  3 : case  4 : case  5 :
      case  6 : case  7 : case  8 : case  9 : case 10 : case 11 :
      case 12 : case 13 : case 14 : case 15 : case 16 : case 17 :
        return null;
      case 18 : return posValues[0];
      case 19 : return posValues[1];
      case 20 : case 21 : case 22 : case 23 : case 24 : case 25 :
      case 26 : case 27 :
        return null;
      default: return super.getValue(_index-ADDEDBYPLOTTINGPANEL);
    }
  }

// ---------- Implementation of InteractiveMouseHandler -------------------

  private InteractionTarget targetHit=null;

  public  ControlDrawable getSelectedDrawable() {
    if (targetHit!=null && (targetHit.getSource() instanceof ControlDrawable) )
      return (ControlDrawable) targetHit.getSource();
    return null;
  }

  public void handleMouseAction(InteractivePanel _panel, java.awt.event.MouseEvent _evt) {
    switch (_panel.getMouseAction ()) {
      case InteractivePanel.MOUSE_PRESSED :
        Interactive interactiveDrawable=_panel.getInteractive ();
        if (interactiveDrawable instanceof InteractionTarget) {
          targetHit = (InteractionTarget) interactiveDrawable;
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_PRESSED,null,targetHit));
        }
        else {
          targetHit = null;
          mousePressed(_panel.getMouseX (),_panel.getMouseY ());
        }
        break;
      case InteractivePanel.MOUSE_DRAGGED :
        if (targetHit!=null) {
          Point3D trackerPoint = new Point3D (_panel.getMouseX (),_panel.getMouseY (),0.0);
          targetHit.updateHotspot(_panel,trackerPoint);
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_DRAGGED,null,targetHit));
          _panel.repaint ();
        }
        else mouseDragged(_panel.getMouseX (),_panel.getMouseY ());
        break;
      case InteractivePanel.MOUSE_RELEASED :
        if (targetHit!=null) {
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_RELEASED,null,targetHit));
        }
        else mouseReleased(_panel.getMouseX (),_panel.getMouseY ());
        // Do not break!
      case InteractivePanel.MOUSE_EXITED :
        targetHit = null;
        break;
      case InteractivePanel.MOUSE_MOVED :
        if (_panel.getInteractive ()!=null)
          _panel.setMouseCursor (Cursor.getPredefinedCursor (Cursor.HAND_CURSOR));
        else _panel.setMouseCursor (Cursor.getPredefinedCursor (Cursor.CROSSHAIR_CURSOR));
        break;
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  public void mousePressed (double _x, double _y) {
    invokeActions (ControlSwingElement.ACTION_PRESS);
    mouseDragged (_x,_y);
  }

  public void mouseDragged (double _x, double _y) {
    posValues[0].value = _x;
    posValues[1].value = _y;
//    System.out.println("dragged at "+_x+","+_y);
    variablesChanged (getPosIndex(),posValues);
  }

  public void mouseReleased (double _x, double _y) {
    invokeActions (ControlElement.ACTION);
  }

} // End of class