/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.awt.Color;

import javax.swing.JTextField;

import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A textfield to display double values. When this value changes,
 * it invokes both the VARIABLE_CHANGED and the ACTION actions.
 */
public class ControlNumberField extends ControlSwingElement {
  static private final int VARIABLE  = 0;
  static private final int BACKGROUND  = ControlSwingElement.BACKGROUND+5;

  static private final java.text.DecimalFormat defaultFormat = new java.text.DecimalFormat ("0.000;-0.000");

  protected JTextField textfield;
  private DoubleValue internalValue;
  private double defaultValue;
  private boolean defaultValueSet;
  private java.text.DecimalFormat format;
  private Color defaultColor, editingColor, errorColor;

// ------------------------------------------------
// Visual component
// ------------------------------------------------

  public ControlNumberField (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof JTextField) textfield = (JTextField) _visual;
    else textfield = new JTextField();
    format = defaultFormat;
    defaultValue  = 0.0;
    defaultValueSet = false;
    internalValue = new DoubleValue(defaultValue);
    textfield.setText (format.format (internalValue.value));
    textfield.addActionListener (new MyActionListener());
    textfield.addKeyListener    (new MyKeyListener());
    decideColors (textfield.getBackground());
    return textfield;
  }

  public void reset() {
    if (defaultValueSet) {
      setTheValue (defaultValue);
      setInternalValue (defaultValue);
    }
  }

  private void setTheValue (double _value) {
    if (_value!=internalValue.value) {
      internalValue.value = _value;
      textfield.setText (format.format (_value));
      setColor (defaultColor);
    }
  }

  private void setInternalValue (double _value) {
    internalValue.value = _value;
    variableChanged (VARIABLE,internalValue);
    invokeActions ();
  }


// ------------------------------------------------
// Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("value");
      infoList.add ("editable");
      infoList.add ("format");
      infoList.add ("action");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))       return "int|double";
    if (_property.equals("value"))          return "int|double";
    if (_property.equals("editable"))       return "boolean";
    if (_property.equals("format"))         return "Format|Object";
    if (_property.equals("action"))         return "Action CONSTANT";
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case VARIABLE : setTheValue (_value.getDouble()); break;
      case 1 :
        defaultValueSet = true; defaultValue = _value.getDouble();
        setActive (false); reset (); setActive(true);
        break;
      case 2 : textfield.setEditable(_value.getBoolean()); break;
      case 3 :
        if (_value.getObject() instanceof java.text.DecimalFormat) {
          if (format == (java.text.DecimalFormat) _value.getObject()) return;
          format = (java.text.DecimalFormat) _value.getObject();
          setActive (false);
          try { setInternalValue (format.parse(textfield.getText()).doubleValue()); }
          catch (Exception exc) {}
          setActive (true);
          textfield.setText (format.format (internalValue.value));
        }
        break;
      case 4 : // action
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;
      case BACKGROUND :
        super.setValue (ControlSwingElement.BACKGROUND,_value);
        decideColors (getVisual().getBackground());
        break;
      default: super.setValue(_index-5,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case VARIABLE : break;  // Do nothing
      case 1 : defaultValueSet = false; break;
      case 2 : textfield.setEditable(true); break;
      case 3 :
        format = defaultFormat;
        setActive (false);
        try { setInternalValue (format.parse(textfield.getText()).doubleValue()); }
        catch (Exception exc) {}
        setActive (true);
        textfield.setText (format.format (internalValue.value));
        break;
      case 4 : removeAction (ControlElement.ACTION,getProperty("action")); break;
      case BACKGROUND :
        super.setDefaultValue (ControlSwingElement.BACKGROUND);
        decideColors (getVisual().getBackground());
        break;
      default: super.setDefaultValue(_index-5); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case VARIABLE : return internalValue;
      case 1 : case 2 : case 3 : case 4 :
        return null;
      default: return super.getValue(_index-5);
    }
  }

// -------------------------------------
// Private methods and inner classes
// -------------------------------------

  private void setColor (Color aColor) {
    if (textfield.isEditable()) getVisual().setBackground (aColor);
  }

  private void decideColors (Color aColor) {
    if (aColor==null) return;
    defaultColor = aColor;
    if (defaultColor.equals(Color.yellow)) editingColor = Color.orange;
    else editingColor = Color.yellow;
    if (defaultColor.equals(Color.red)) errorColor = Color.magenta;
    else errorColor = Color.red;
  }


  private class MyActionListener implements java.awt.event.ActionListener {
    public void actionPerformed (java.awt.event.ActionEvent _e) {
//      System.out.println ("Action for "+textfield.getText());
      setColor (defaultColor);
      try { setInternalValue (format.parse(textfield.getText()).doubleValue()); }
      catch (Exception exc) {
        setColor (errorColor);
        exc.printStackTrace(System.err);
      }
    }
  }

  private class MyKeyListener implements java.awt.event.KeyListener {
    public void keyPressed  (java.awt.event.KeyEvent _e) { processKeyEvent (_e,0); }
    public void keyReleased (java.awt.event.KeyEvent _e) { processKeyEvent (_e,1); }
    public void keyTyped    (java.awt.event.KeyEvent _e) { processKeyEvent (_e,2); }
    private void processKeyEvent (java.awt.event.KeyEvent _e, int _n) {
      if (!textfield.isEditable()) return;
//      System.out.println ("Key Event "+_n+" for "+textfield.getText());
      if (_e.getKeyChar()!='\n') setColor (editingColor);
      if (_e.getKeyCode()==27)   setValue (VARIABLE,internalValue);
    }
  }

} // End of class