/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import org.opensourcephysics.displayejs.DrawingPanel3D;
import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionListener;
import org.opensourcephysics.displayejs.Point3D;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.IntegerValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A configurable 3D panel from the displayejs package
 */
public class ControlDrawingPanel3DNew extends ControlDrawablesParent implements InteractionListener {
  static private final double TO_RAD = Math.PI/180.0;
  static private final int [] posIndexes = {9,10,11};
  static private final int [] angleIndexes = {15,16};
  static private final int ZOOM_INDEX=17;
  static private final int [] panIndexes = {18,19};

  protected DrawingPanel3D drawingPanel3D;
  private double minX, maxX, minY, maxY, minZ, maxZ;
  private DoubleValue zoomValue;
  private DoubleValue angleValues[];
  private IntegerValue panValues[];
  private DoubleValue[] posValues ={ new DoubleValue(0.0), new DoubleValue(0.0), new DoubleValue(0.0)};

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlDrawingPanel3DNew (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof DrawingPanel3D) drawingPanel3D = (DrawingPanel3D) _visual;
    else drawingPanel3D = new DrawingPanel3D ();
    // drawingPanel3D.removeOptionController();
    minX = drawingPanel3D.getXMin();
    maxX = drawingPanel3D.getXMax();
    minY = drawingPanel3D.getYMin();
    maxY = drawingPanel3D.getYMax();
    minZ = drawingPanel3D.getZMin();
    maxZ = drawingPanel3D.getZMax();
    angleValues = new DoubleValue[2];
    angleValues[0] = new DoubleValue(drawingPanel3D.getAlpha());
    angleValues[1] = new DoubleValue(drawingPanel3D.getBeta());
    zoomValue = new DoubleValue(drawingPanel3D.getZoom());
    panValues = new IntegerValue[2];
    panValues[0]  = new IntegerValue(drawingPanel3D.getPan().x);
    panValues[1]  = new IntegerValue(drawingPanel3D.getPan().y);
    drawingPanel3D.addListener (this);
    drawingPanel3D.render();
    return drawingPanel3D;
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("autoscaleX");
      infoList.add ("autoscaleY");
      infoList.add ("autoscaleZ");
      infoList.add ("minimumX");
      infoList.add ("maximumX");
      infoList.add ("minimumY");
      infoList.add ("maximumY");
      infoList.add ("minimumZ");
      infoList.add ("maximumZ");
      infoList.add ("x"); // 9
      infoList.add ("y");
      infoList.add ("z");
      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");

      infoList.add ("alpha"); // 15
      infoList.add ("beta");
      infoList.add ("zoom");
      infoList.add ("panx");
      infoList.add ("pany");

      infoList.add ("displayMode");
      infoList.add ("decoration");
      infoList.add ("square");
      infoList.add ("cursorMode");
      infoList.add ("showCoordinates");
      infoList.add ("hideLines");

      infoList.addAll(super.getPropertyList()); // Inherit from ControlSwingElement
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("autoscaleX"))     return "boolean";
    if (_property.equals("autoscaleY"))     return "boolean";
    if (_property.equals("autoscaleZ"))     return "boolean";
    if (_property.equals("minimumX"))       return "int|double";
    if (_property.equals("maximumX"))       return "int|double";
    if (_property.equals("minimumY"))       return "int|double";
    if (_property.equals("maximumY"))       return "int|double";
    if (_property.equals("minimumZ"))       return "int|double";
    if (_property.equals("maximumZ"))       return "int|double";
    if (_property.equals("x"))              return "int|double";
    if (_property.equals("y"))              return "int|double";
    if (_property.equals("z"))              return "int|double";
    if (_property.equals("action"))         return "Action CONSTANT";
    if (_property.equals("pressaction"))    return "Action CONSTANT";
    if (_property.equals("dragaction"))     return "Action CONSTANT";

    if (_property.equals("alpha"))          return "int|double BASIC";
    if (_property.equals("beta"))           return "int|double BASIC";
    if (_property.equals("zoom"))           return "int|double BASIC";
    if (_property.equals("panx"))           return "int|double BASIC HIDDEN";
    if (_property.equals("pany"))           return "int|double BASIC HIDDEN";

    if (_property.equals("displayMode"))    return "int|DisplayMode BASIC";
    if (_property.equals("decoration"))     return "int|Decoration BASIC";
    if (_property.equals("square"))         return "boolean BASIC";
    if (_property.equals("cursorMode"))     return "int|CursorMode BASIC";
    if (_property.equals("showCoordinates"))return "boolean BASIC";
    if (_property.equals("hideLines"))      return "boolean BASIC";

/*
    if (_property.equals("perspective"))    return "boolean BASIC";
    if (_property.equals("showBox"))        return "boolean BASIC";
    if (_property.equals("showAxes"))       return "boolean BASIC";
*/
    return super.getPropertyInfo(_property);
  }

  public Value parseConstant (String _propertyType, String _value) {
    if (_value==null) return null;
    if (_propertyType.indexOf("DisplayMode")>=0) {
      _value = _value.trim().toLowerCase();
      if (_value.equals("perspective")) return new IntegerValue (DrawingPanel3D.DISPLAY_PERSPECTIVE);
      if (_value.equals("no_perspective")) return new IntegerValue (DrawingPanel3D.DISPLAY_NO_PERSPECTIVE);
      if (_value.equals("planar_xy")) return new IntegerValue (DrawingPanel3D.DISPLAY_PLANAR_XY);
      if (_value.equals("planar_xz")) return new IntegerValue (DrawingPanel3D.DISPLAY_PLANAR_XZ);
      if (_value.equals("planar_yz")) return new IntegerValue (DrawingPanel3D.DISPLAY_PLANAR_YZ);
    }
    if (_propertyType.indexOf("Decoration")>=0) {
      _value = _value.trim().toLowerCase();
      if (_value.equals("none")) return new IntegerValue (DrawingPanel3D.DECORATION_NONE);
      if (_value.equals("cube")) return new IntegerValue (DrawingPanel3D.DECORATION_CUBE);
      if (_value.equals("axes")) return new IntegerValue (DrawingPanel3D.DECORATION_AXES);
    }
    if (_propertyType.indexOf("CursorMode")>=0) {
      _value = _value.trim().toLowerCase();
      if (_value.equals("none")) return new IntegerValue (DrawingPanel3D.CURSOR_NONE);
      if (_value.equals("cube")) return new IntegerValue (DrawingPanel3D.CURSOR_CUBE);
      if (_value.equals("crosshair")) return new IntegerValue (DrawingPanel3D.CURSOR_CROSSHAIR);
      if (_value.equals("xyz")) return new IntegerValue (DrawingPanel3D.CURSOR_XYZ);
    }
    return super.parseConstant (_propertyType,_value);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    double angle;
    switch (_index) {
      case 0 : drawingPanel3D.setAutoscaleX(_value.getBoolean()); break;
      case 1 : drawingPanel3D.setAutoscaleY(_value.getBoolean()); break;
      case 2 : drawingPanel3D.setAutoscaleZ(_value.getBoolean()); break;
      case 3 : if (_value.getDouble()!=minX) drawingPanel3D.setPreferredMinMaxX(minX=_value.getDouble(),maxX); break;
      case 4 : if (_value.getDouble()!=maxX) drawingPanel3D.setPreferredMinMaxX(minX,maxX=_value.getDouble()); break;
      case 5 : if (_value.getDouble()!=minY) drawingPanel3D.setPreferredMinMaxY(minY=_value.getDouble(),maxY); break;
      case 6 : if (_value.getDouble()!=maxY) drawingPanel3D.setPreferredMinMaxY(minY,maxY=_value.getDouble()); break;
      case 7 : if (_value.getDouble()!=minZ) drawingPanel3D.setPreferredMinMaxZ(minZ=_value.getDouble(),maxZ); break;
      case 8 : if (_value.getDouble()!=maxZ) drawingPanel3D.setPreferredMinMaxZ(minZ,maxZ=_value.getDouble()); break;

      case  9 : posValues[0].value = _value.getDouble(); break;
      case 10 : posValues[1].value = _value.getDouble(); break;
      case 11 : posValues[2].value = _value.getDouble(); break;
      case 12 : // pressaction
        removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlSwingElement.ACTION_PRESS,_value.getString());
        break;
      case 13 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        break;
      case 14 : // pressaction
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;

      case 15 :
        if (_value instanceof IntegerValue) angle = _value.getInteger()*TO_RAD;
        else angle = _value.getDouble();
        if (angle!=angleValues[0].value) drawingPanel3D.setAlpha(angleValues[0].value=angle);
        break;
      case 16 :
        if (_value instanceof IntegerValue) angle = _value.getInteger()*TO_RAD;
        else angle = _value.getDouble();
        if (angle!=angleValues[1].value) drawingPanel3D.setBeta(angleValues[1].value=angle);
        break;
      case 17 : if (_value.getDouble()!=zoomValue.value) drawingPanel3D.setZoom(zoomValue.value=_value.getDouble()); break;
      case 18 : if (_value.getInteger()!=panValues[0].value) drawingPanel3D.setPan(panValues[0].value=_value.getInteger(),panValues[1].value); break;
      case 19 : if (_value.getInteger()!=panValues[1].value) drawingPanel3D.setPan(panValues[0].value,panValues[1].value=_value.getInteger()); break;

      case 20 : if (_value.getInteger()!=drawingPanel3D.getDisplayMode())    drawingPanel3D.setDisplayMode   (_value.getInteger()); break;
      case 21 : if (_value.getInteger()!=drawingPanel3D.getDecorationType()) drawingPanel3D.setDecorationType(_value.getInteger()); break;
      case 22 : if (_value.getBoolean()!=drawingPanel3D.isSquareAspect())    drawingPanel3D.setSquareAspect  (_value.getBoolean()); break;
      case 23 : if (_value.getInteger()!=drawingPanel3D.getCursorMode())     drawingPanel3D.setCursorMode    (_value.getInteger()); break;
      case 24 : drawingPanel3D.setShowCoordinates(_value.getBoolean()); break;
      case 25 : drawingPanel3D.setRemoveHiddenLines(_value.getBoolean()); break;

      default: super.setValue(_index-26,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case  0 : drawingPanel3D.setAutoscaleX(false); break;
      case  1 : drawingPanel3D.setAutoscaleY(false); break;
      case  2 : drawingPanel3D.setAutoscaleZ(false); break;
      case  3 : drawingPanel3D.setPreferredMinMaxX(minX=0.0,maxX); break;
      case  4 : drawingPanel3D.setPreferredMinMaxX(minX,maxX=1.0); break;
      case  5 : drawingPanel3D.setPreferredMinMaxY(minY=0.0,maxY); break;
      case  6 : drawingPanel3D.setPreferredMinMaxY(minY,maxY=1.0); break;
      case  7 : drawingPanel3D.setPreferredMinMaxZ(minZ=0.0,maxZ); break;
      case  8 : drawingPanel3D.setPreferredMinMaxZ(minZ,maxZ=1.0); break;

      case  9 : posValues[0].value = (minX+maxX)/2.0; break;
      case 10 : posValues[1].value = (minY+maxY)/2.0; break;
      case 11 : posValues[2].value = (minZ+maxZ)/2.0; break;
      case 12 : removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction")); break;
      case 13 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));   break;
      case 14 : removeAction (ControlElement.ACTION,getProperty("action"));                break;

      case 15 : drawingPanel3D.setAlpha(angleValues[0].value=0.0); break;
      case 16 : drawingPanel3D.setBeta(angleValues[1].value=0.0);  break;
      case 17 : drawingPanel3D.setZoom(zoomValue.value=1.0);     break;
      case 18 : drawingPanel3D.setPan(panValues[0].value=0,panValues[1].value);  break;
      case 19 : drawingPanel3D.setPan(panValues[0].value,panValues[1].value=0);  break;

      case 20 : drawingPanel3D.setDisplayMode   (DrawingPanel3D.DISPLAY_PERSPECTIVE); break;
      case 21 : drawingPanel3D.setDecorationType(DrawingPanel3D.DECORATION_CUBE); break;
      case 22 : drawingPanel3D.setSquareAspect  (true); break;
      case 23 : drawingPanel3D.setCursorMode    (DrawingPanel3D.CURSOR_CROSSHAIR); break;
      case 24 : drawingPanel3D.setShowCoordinates(true); break;
      case 25 : drawingPanel3D.setRemoveHiddenLines(true); break;

      default: super.setDefaultValue(_index-26); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case  0 : case  1 : case  2 : case  3 :
      case  4 : case  5 : case  6 : case  7 :
      case  8 :
      case 12 : case 13 : case 14 :
      case 20 : case 21 : case 22 : case 23 :
      case 24 : case 25 :
        return null;

      case  9 : return posValues[0];
      case 10 : return posValues[1];
      case 11 : return posValues[2];

      case 15 : return angleValues[0];
      case 16 : return angleValues[1];
      case 17 : return zoomValue;
      case 18 : return panValues[0];
      case 19 : return panValues[1];

      default: return super.getValue(_index-26);
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  public void interactionPerformed(InteractionEvent _event) {
    if (_event.getTarget()==null) { // Just changing the view point
      if (_event.getID()==InteractionEvent.MOUSE_DRAGGED) { // Update projection data
        if (angleValues[0].value != drawingPanel3D.getAlpha() ||
          angleValues[1].value != drawingPanel3D.getBeta()) {
          angleValues[0].value = drawingPanel3D.getAlpha();
          angleValues[1].value = drawingPanel3D.getBeta();
          variablesChanged(angleIndexes, angleValues);
        }
        if (zoomValue.value != drawingPanel3D.getZoom()) {
          zoomValue.value = drawingPanel3D.getZoom();
          variableChanged(ZOOM_INDEX, zoomValue);
        }
        if (panValues[0].value != drawingPanel3D.getPan().x ||
          panValues[1].value != drawingPanel3D.getPan().y) {
          panValues[0].value = drawingPanel3D.getPan().x;
          panValues[1].value = drawingPanel3D.getPan().y;
          variablesChanged(panIndexes, panValues);
        }
      }
      return;
    }
    if (! (_event.getTarget() instanceof Point3D) ) return; // Not a real click on the panel
    Point3D point = (Point3D) _event.getTarget();
    switch (_event.getID()) {
      case InteractionEvent.MOUSE_PRESSED :
        invokeActions (ControlSwingElement.ACTION_PRESS);
        // Do not break!
      case InteractionEvent.MOUSE_DRAGGED :
        posValues[0].value = point.x;
        posValues[1].value = point.y;
        posValues[2].value = point.z;
        // System.out.println("dragged at "+point.x+","+point.y+","+point.z);
        variablesChanged (posIndexes,posValues);
        break;
      case InteractionEvent.MOUSE_RELEASED :
        invokeActions (ControlElement.ACTION);
        break;
    }
  } // End of interaction method

} // End of class