/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.awt.Cursor;

import org.opensourcephysics.display.Interactive;
import org.opensourcephysics.display.InteractiveMouseHandler;
import org.opensourcephysics.display.InteractivePanel;
import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionTarget;
import org.opensourcephysics.displayejs.InteractiveElement;
import org.opensourcephysics.displayejs.Point3D;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A panel to hold Drawables
 */
public class ControlDrawingPanel extends ControlDrawablesParent implements InteractiveMouseHandler {
  static private final int[] posIndex = {6,7};

  protected InteractivePanel drawingPanel;
  private double minX, maxX,minY, maxY;
  private java.awt.Rectangle myGutters=null;
  private DoubleValue[] posValues ={ new DoubleValue(0.0), new DoubleValue(0.0)};

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlDrawingPanel (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof InteractivePanel) drawingPanel = (InteractivePanel) _visual;
    else {
      drawingPanel = new InteractivePanel ();
      drawingPanel.enableInspector (false); // OSP Update
      drawingPanel.setSquareAspect (false);
    }
    drawingPanel.removeOptionController();
    minX = drawingPanel.getXMin();
    maxX = drawingPanel.getXMax();
    minY = drawingPanel.getYMin();
    maxY = drawingPanel.getYMax();
    drawingPanel.render();
    drawingPanel.setInteractiveMouseHandler(this);
    return drawingPanel;
  }

  protected int[] getPosIndex () { return posIndex; } // in case it should be overriden

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("autoscaleX");
      infoList.add ("autoscaleY");
      infoList.add ("minimumX");
      infoList.add ("maximumX");
      infoList.add ("minimumY");
      infoList.add ("maximumY");
      infoList.add ("x");
      infoList.add ("y");
      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");
      infoList.add ("square");
      infoList.add ("showCoordinates");
      infoList.add ("gutters");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("autoscaleX"))     return "boolean";
    if (_property.equals("autoscaleY"))     return "boolean";
    if (_property.equals("minimumX"))       return "int|double";
    if (_property.equals("maximumX"))       return "int|double";
    if (_property.equals("minimumY"))       return "int|double";
    if (_property.equals("maximumY"))       return "int|double";
    if (_property.equals("x"))              return "int|double";
    if (_property.equals("y"))              return "int|double";
    if (_property.equals("action"))         return "Action CONSTANT";
    if (_property.equals("pressaction"))    return "Action CONSTANT";
    if (_property.equals("dragaction"))     return "Action CONSTANT";
    if (_property.equals("square"))         return "boolean BASIC";
    if (_property.equals("showCoordinates"))return "boolean BASIC";
    if (_property.equals("gutters"))        return "Margins|Object BASIC";
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 : drawingPanel.setAutoscaleX(_value.getBoolean()); break;
      case 1 : drawingPanel.setAutoscaleY(_value.getBoolean()); break;
      case 2 : if (_value.getDouble()!=minX) drawingPanel.setPreferredMinMaxX(minX=_value.getDouble(),maxX); break;
      case 3 : if (_value.getDouble()!=maxX) drawingPanel.setPreferredMinMaxX(minX,maxX=_value.getDouble()); break;
      case 4 : if (_value.getDouble()!=minY) drawingPanel.setPreferredMinMaxY(minY=_value.getDouble(),maxY); break;
      case 5 : if (_value.getDouble()!=maxY) drawingPanel.setPreferredMinMaxY(minY,maxY=_value.getDouble()); break;
      case 6 : posValues[0].value = _value.getDouble(); break;
      case 7 : posValues[1].value = _value.getDouble(); break;
      case 8 : // pressaction
        removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlSwingElement.ACTION_PRESS,_value.getString());
        break;
      case 9 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        break;
      case 10 : // pressaction
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;
      case 11 : drawingPanel.setSquareAspect(_value.getBoolean());    break;
      case 12 : drawingPanel.setShowCoordinates(_value.getBoolean()); break;
      case 13 :
        if (_value.getObject() instanceof java.awt.Rectangle) {
          java.awt.Rectangle rect = (java.awt.Rectangle) _value.getObject();
          if (rect!=myGutters) {
            drawingPanel.setGutters(rect.x,rect.y,rect.width,rect.height);
            myGutters = rect;
          }
        }
        break;
      default: super.setValue(_index-14,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 : drawingPanel.setAutoscaleX(false); break;
      case 1 : drawingPanel.setAutoscaleY(false); break;
      case 2 : drawingPanel.setPreferredMinMaxX(minX=0.0,maxX); break;
      case 3 : drawingPanel.setPreferredMinMaxX(minX,maxX=1.0); break;
      case 4 : drawingPanel.setPreferredMinMaxY(minY=0.0,maxY); break;
      case 5 : drawingPanel.setPreferredMinMaxY(minY,maxY=1.0); break;
      case 6 : posValues[0].value = (minX+maxX)/2.0; break; // x
      case 7 : posValues[1].value = (minY+maxY)/2.0; break; // y
      case 8 : removeAction (ControlSwingElement.ACTION_PRESS,getProperty("pressaction")); break;
      case 9 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));   break;
      case 10 : removeAction (ControlElement.ACTION,getProperty("action"));                break;
      case 11 : drawingPanel.setSquareAspect(false);   break;
      case 12 : drawingPanel.setShowCoordinates(true); break;
      case 13 : drawingPanel.setGutters(0,0,0,0); myGutters = null; break;
      default: super.setDefaultValue(_index-14); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case 0 : case 1 : case  2 : case  3 : case  4 : case  5 :
      case 8 : case 9 : case 10 : case 11 : case 12 : case 13 :
        return null;
      case 6 : return posValues[0];
      case 7 : return posValues[1];
      default: return super.getValue(_index-14);
    }
  }

// ---------- Implementation of InteractiveMouseHandler -------------------

  private InteractionTarget targetHit=null;

  public  ControlDrawable getSelectedDrawable() {
    if (targetHit!=null && (targetHit.getSource() instanceof InteractiveElement) ) {
      Object data = ((InteractiveElement) targetHit.getSource()).getDataObject();
      if (data instanceof ControlDrawable) return (ControlDrawable) data;
    }
    return null;
  }

  public void handleMouseAction(InteractivePanel _panel, java.awt.event.MouseEvent _evt) {
    switch (_panel.getMouseAction ()) {
      case InteractivePanel.MOUSE_PRESSED :
        Interactive interactiveDrawable=_panel.getInteractive ();
        if (interactiveDrawable instanceof InteractionTarget) {
          targetHit = (InteractionTarget) interactiveDrawable;
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_PRESSED,null,targetHit));
        }
        else {
          targetHit = null;
          mousePressed(_panel.getMouseX (),_panel.getMouseY ());
        }
        break;
      case InteractivePanel.MOUSE_DRAGGED :
        if (targetHit!=null) {
          Point3D trackerPoint = new Point3D (_panel.getMouseX (),_panel.getMouseY (),0.0);
          targetHit.updateHotspot(_panel,trackerPoint);
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_DRAGGED,null,targetHit));
          _panel.repaint ();
        }
        else mouseDragged(_panel.getMouseX (),_panel.getMouseY ());
        break;
      case InteractivePanel.MOUSE_RELEASED :
        if (targetHit!=null) {
          targetHit.getSource().invokeActions (new InteractionEvent (targetHit.getSource(),InteractionEvent.MOUSE_RELEASED,null,targetHit));
        }
        else mouseReleased(_panel.getMouseX (),_panel.getMouseY ());
        // Do not break!
      case InteractivePanel.MOUSE_EXITED :
        targetHit = null;
        break;
      case InteractivePanel.MOUSE_MOVED :
        if (_panel.getInteractive ()!=null)
          _panel.setMouseCursor (Cursor.getPredefinedCursor (Cursor.HAND_CURSOR));
        else _panel.setMouseCursor (Cursor.getPredefinedCursor (Cursor.CROSSHAIR_CURSOR));
        break;
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  public void mousePressed (double _x, double _y) {
    invokeActions (ControlSwingElement.ACTION_PRESS);
    mouseDragged (_x,_y);
  }

  public void mouseDragged (double _x, double _y) {
    posValues[0].value = _x;
    posValues[1].value = _y;
//    System.out.println("dragged at "+_x+","+_y);
    variablesChanged (getPosIndex(),posValues);
  }

  public void mouseReleased (double _x, double _y) {
    invokeActions (ControlElement.ACTION);
  }


} // End of class