/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.BooleanValue;

/**
 * A configurable Container
 */
public abstract class ControlContainer extends ControlSwingElement {
  static private final BooleanValue falseValue = new BooleanValue(false);
  protected java.util.Vector radioButtons = new java.util.Vector();
  protected java.util.Vector children     = new java.util.Vector();

  public ControlContainer (Object _visual) { super (_visual); }

  // This is not final since windows may change the default case (this one)
  public java.awt.Container getContainer () { return (java.awt.Container) getVisual(); }

// ------------------------------------------------
// Own methods
// ------------------------------------------------

 /**
  * adds a child control
  * @param _child the child control
  */
  public void add(ControlElement _child) {
    children.add(_child);
    java.awt.Container container = getContainer();
    java.awt.LayoutManager layout = container.getLayout();
    // This is set by Ejs to allow changing the natural order of childhood
    String indexInParent = _child.getProperty("_ejs_indexInParent_");
    int index = -1;
    if (indexInParent!=null) index = Integer.parseInt(indexInParent);
    _child.setProperty("_ejs_indexInParent_",null);
    if (layout instanceof java.awt.BorderLayout) {
      String pos = _child.getProperty("position");
      if (pos!=null) container.add(_child.getComponent(),ConstantParser.constraintsConstant(pos).getString(),index);
      else container.add(_child.getComponent(),java.awt.BorderLayout.CENTER,index);
    }
    else container.add(_child.getComponent(),index);
    adjustSize();
    if (_child instanceof ControlRadioButton) {
      radioButtons.add(_child);
      ((ControlRadioButton)_child).setParent(this);
    }
    // Now propagate my own font, foreground and background;
    propagateProperty (_child,"font"      ,getProperty("font"));
    propagateProperty (_child,"foreground",getProperty("foreground"));
    propagateProperty (_child,"background",getProperty("background"));
  }

  public void adjustSize() {
    getContainer().validate();
    getContainer().repaint();
    resizeContainer (getContainer());
    resizeContainer (getComponent().getParent());
  }

  private static void resizeContainer(java.awt.Container _container) {
    if (_container==null) return;
    java.awt.Rectangle b = _container.getBounds();
    _container.setBounds (b.x,b.y,b.width+1,b.height+1);
    _container.setBounds (b.x,b.y,b.width,b.height);
    _container.validate();
    _container.repaint();
  }

 /**
  * Returns the vector of children
  */
  public java.util.Vector getChildren() { return children; }

 /**
  * removes a child control
  * @param _child the child control
  */
  public void remove(ControlElement _child) {
    children.remove(_child);
    java.awt.Container container = getContainer();
    container.remove(_child.getComponent());
    container.validate();
    container.repaint();
    if (_child instanceof ControlRadioButton) {
      radioButtons.remove(_child);
      ((ControlRadioButton)_child).setParent(null);
    }
  }

  public void informRadioGroup(ControlRadioButton _source, boolean _state) {
    if (_state==false) return;
    for (java.util.Enumeration e = radioButtons.elements(); e.hasMoreElements(); ) {
      ControlRadioButton rb = (ControlRadioButton) e.nextElement();
      if (rb!=_source) {
        boolean wasActive = rb.isActive();
        rb.setActive(false);
        rb.setValue (ControlRadioButton.VARIABLE,falseValue);
        rb.reportChanges ();
        rb.setActive(wasActive);
      }
    }
  }

// ------------------------------------------------
// Private methods
// ------------------------------------------------

  private void propagateProperty (ControlElement _child, String _property, String _value) {
    if (_child.getProperty(_property)==null) _child.setProperty(_property,_value);
  }

  private void propagateProperty (String _property, String _value) {
    for (int i=0; i<children.size(); i++)
      propagateProperty ((ControlElement) children.elementAt(i),_property,_value);
  }

// ------------------------------------------------
// Properties
// ------------------------------------------------

  public String getPropertyInfo(String _property) {
    if (_property.equals("visible")) return "boolean"; // not HIDDEN
    return super.getPropertyInfo(_property);
  }

  public ControlElement setProperty (String _property, String _value) {
    ControlElement returnValue = super.setProperty (_property,_value);
    if (_property.equals("font") || _property.equals("foreground")
     || _property.equals("background")) propagateProperty (_property,_value);
    return returnValue;
  }

} // End of class