/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.awt.Color;
import java.util.StringTokenizer;

import javax.swing.JComboBox;

import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.StringValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * A combobox to display string options. When the selected option changes,
 * it invokes both the VARIABLE_CHANGED and the ACTION actions.
 */
public class ControlComboBox extends ControlSwingElement {
  static private final int VARIABLE  = 0;
  static private final int BACKGROUND  = ControlSwingElement.BACKGROUND+6;
  static private final int FOREGROUND  = ControlSwingElement.FOREGROUND+6;

  protected JComboBox combo;
  private java.awt.Component editorComponent;
  private String optionsString;
  private StringValue internalValue;
  private boolean defaultValueSet, defaultEditable,doNotUpdate=false;
  private String defaultValue;
  private Color defaultColor, editingColor, errorColor;

// ------------------------------------------------
// Visual component
// ------------------------------------------------
  public ControlComboBox (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof JComboBox) combo = (JComboBox) _visual;
    else combo = new JComboBox();
    defaultEditable = combo.isEditable();
    combo.addActionListener (new MyActionListener());
    editorComponent = combo.getEditor().getEditorComponent();
    editorComponent.addKeyListener    (new MyKeyListener());
    defaultValue  = "";
    defaultValueSet = false;
    internalValue = new StringValue(defaultValue);
    decideColors (editorComponent.getBackground());
    return combo;
  }

  public void reset() {
    if (defaultValueSet) {
      setTheValue (defaultValue);
      setInternalValue (defaultValue);
    }
  }

  private void setTheValue (String _value) {
    if (internalValue.value!=null && internalValue.value.equals(_value)) return;
    combo.setSelectedItem (internalValue.value = _value);
    setColor (defaultColor);
  }

  private void setInternalValue (String _value) {
    internalValue.value = _value;
    variableChanged (VARIABLE,internalValue);
    invokeActions ();
  }

  private void setTheOptions(String _options)  {
    if (_options==null) {
      if (optionsString!=null) {
        combo.removeAllItems();
        optionsString = null;
      }
      return;
    }
    if (_options.equals(optionsString)) return;
    doNotUpdate = true;
    combo.removeAllItems();
    StringTokenizer tkn = new StringTokenizer (_options,";");
    while (tkn.hasMoreTokens()) combo.addItem(tkn.nextToken());
    optionsString = _options;
    doNotUpdate = false;
    if (combo.getItemCount()>0) setTheValue (combo.getItemAt(0).toString());
  }

// ------------------------------------------------
// Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("options");
      infoList.add ("value");
      infoList.add ("editable");
      infoList.add ("editBackground");
      infoList.add ("action");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))       return "String VARIABLE_EXPECTED";
    if (_property.equals("options"))        return "String PREVIOUS";
    if (_property.equals("value"))          return "String CONSTANT";
    if (_property.equals("editable"))       return "boolean";
    if (_property.equals("editBackground")) return "Color|Object";
    if (_property.equals("action"))         return "Action CONSTANT";
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case VARIABLE : setTheValue(_value.getString()); break;
      case 1 : setTheOptions(_value.getString()); break;
      case 2 :
        defaultValueSet = true; defaultValue = _value.getString();
        setActive (false); reset (); setActive(true);
        break;
      case 3 : combo.setEditable(_value.getBoolean()); break;
      case 4 :
        if (_value.getObject() instanceof Color) editorComponent.setBackground((Color) _value.getObject());
        decideColors (editorComponent.getBackground());
        break;
      case 5 : // action
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        break;
      default: super.setValue(_index-6,_value); break;
      case FOREGROUND :
        super.setValue (ControlSwingElement.FOREGROUND,_value);
        if (_value.getObject() instanceof Color) editorComponent.setForeground((Color) _value.getObject());
        break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case VARIABLE : break; // Do nothing
      case 1 : setTheOptions(null); break;
      case 2 : defaultValueSet = false; break;
      case 3 : combo.setEditable(defaultEditable); break;
      case 4 :
        editorComponent.setBackground(Color.white);
        decideColors (editorComponent.getBackground());
        break;
      case 5 : removeAction (ControlElement.ACTION,getProperty("action")); break;
      default: super.setDefaultValue(_index-6); break;
      case FOREGROUND :
        super.setDefaultValue (ControlSwingElement.FOREGROUND);
        editorComponent.setForeground(Color.black);
        break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case VARIABLE : return internalValue;
      case 1 : case 2 : case 3 : case 4 : case 5 :
        return null;
      default: return super.getValue(_index-6);
    }
  }

// -------------------------------------
// Private methods and inner classes
// -------------------------------------

  private void setColor (Color aColor) {
    if (combo.isEditable()) editorComponent.setBackground (aColor);
  }

  private void decideColors (Color aColor) {
    if (aColor==null) return;
    defaultColor = aColor;
    if (defaultColor.equals(Color.yellow)) editingColor = Color.orange;
    else editingColor = Color.yellow;
    if (defaultColor.equals(Color.red)) errorColor = Color.magenta;
    else errorColor = Color.red;
  }

  private class MyActionListener implements java.awt.event.ActionListener {
    public void actionPerformed (java.awt.event.ActionEvent _e) {
      if (doNotUpdate) return;
      setInternalValue ((String) combo.getSelectedItem());
      setColor (defaultColor);
    }
  }

  private class MyKeyListener implements java.awt.event.KeyListener {
    public void keyPressed  (java.awt.event.KeyEvent _e) { processKeyEvent (_e,0); }
    public void keyReleased (java.awt.event.KeyEvent _e) { processKeyEvent (_e,1); }
    public void keyTyped    (java.awt.event.KeyEvent _e) { processKeyEvent (_e,2); }
    private void processKeyEvent (java.awt.event.KeyEvent _e, int _n) {
      if (!combo.isEditable()) return;
      if (_e.getKeyChar()!='\n') setColor (editingColor);
      if (_e.getKeyCode()==27)   setValue (VARIABLE,internalValue);
    }
  }


} // End of class