/*
 * The control.swing package contains subclasses of control.ControlElement
 * that create visuals using Java's Swing library
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.swing;

import java.text.DecimalFormat;

import javax.swing.JProgressBar;

import org.opensourcephysics.ejs.control.value.Value;

/**
 * A bar that display double values. The value cannot be changed
 */
public class ControlBar extends ControlSwingElement {
  static private final int RESOLUTION=100000;

  protected JProgressBar bar;
  private double scale, minimum=0.0, maximum=1.0, variable=0.0;
  private DecimalFormat format;

// ------------------------------------------------
// Visual component
// ------------------------------------------------

  public ControlBar (Object _visual) { super (_visual); }

  protected java.awt.Component createVisual (Object _visual) {
    if (_visual instanceof JProgressBar) bar = (JProgressBar) _visual;
    else {
      bar = new JProgressBar(JProgressBar.HORIZONTAL);
      bar.setBorderPainted(true);
      bar.setStringPainted(false);
    }
    bar.setMinimum (0);
    bar.setMaximum (RESOLUTION);
    minimum  = 0.0;
    maximum  = 1.0;
    variable = bar.getValue();
    scale    = RESOLUTION*(maximum-minimum);
    format   = null;
    bar.setValue ((int) ((variable-minimum)*scale));
    return bar;
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("minimum");
      infoList.add ("maximum");
      infoList.add ("format");
      infoList.add ("orientation");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))       return "int|double";
    if (_property.equals("minimum"))        return "int|double";
    if (_property.equals("maximum"))        return "int|double";
    if (_property.equals("format"))         return "Format|Object";
    if (_property.equals("orientation"))    return "Orientation|int";
    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 : setValue   (_value.getDouble()); break;
      case 1 : setMinimum (_value.getDouble()); break;
      case 2 : setMaximum (_value.getDouble()); break;
      case 3 :
        {
          DecimalFormat newFormat;
          if (_value.getObject() instanceof DecimalFormat) newFormat = (DecimalFormat) _value.getObject();
          else newFormat = null;
          if (format==newFormat) return; // and save time
          format = newFormat;
          if (format!=null) {
            bar.setString (format.format(variable));
            bar.setStringPainted(true);
          }
          else bar.setStringPainted(false);
        }
        break;
      case 4 :
        if (bar.getOrientation()!=_value.getInteger()) bar.setOrientation(_value.getInteger());
        break;
      default: super.setValue(_index-5,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 : break; // Do nothing
      case 1 : setMinimum (0.0); break;
      case 2 : setMaximum (1.0); break;
      case 3 : format = null; bar.setStringPainted(false); break;
      case 4 : bar.setOrientation(JProgressBar.HORIZONTAL); break;
      default: super.setDefaultValue(_index-5); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case 0 : case 1 : case 2 : case 3 :
      case 4 :
        return null;
      default: return super.getValue(_index-5);
    }
  }

// -------------- private methods -----------

  private void setValue (double val) {
    if (val==variable) return;
    variable = val;
    bar.setValue ((int) ((variable-minimum)*scale));
    if (format!=null) bar.setString (format.format (variable));
  }

  private void setMinimum (double val) {
    if (val==minimum) return;
    minimum = val;
    if (minimum>=maximum) maximum = minimum+1.0;
    scale = 1.0*RESOLUTION/(maximum-minimum);
    bar.setValue ((int) ((variable-minimum)*scale));
    if (format!=null) bar.setString (format.format (variable));
  }

  private void setMaximum (double val) {
    if (val==maximum) return;
    maximum = val;
    if (minimum>=maximum) minimum = maximum-1.0;
    scale = 1.0*RESOLUTION/(maximum-minimum);
    bar.setValue ((int) ((variable-minimum)*scale));
    if (format!=null) bar.setString (format.format (variable));
  }

}