/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) October 2003 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.displayejs.ElementSet;
import org.opensourcephysics.displayejs.InteractiveParticle;
import org.opensourcephysics.displayejs.InteractiveTrace;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * An interactive set of particles
 */
public class ControlTraceSet extends ControlElementSet {
  static final protected int TRACESET_ADDED=9;
  static protected final int INPUT_X = POSITION_X+TRACESET_ADDED;
  static protected final int INPUT_Y = POSITION_Y+TRACESET_ADDED;
  static protected final int INPUT_Z = POSITION_Z+TRACESET_ADDED;
  static protected final int MY_STYLE = STYLE + TRACESET_ADDED;

  protected boolean norepeat;
  protected int elements=0;
  protected double[] x, y, z;
  protected double[] xOne, yOne, zOne;
  private boolean xIsConstant=true, xIsSet=false,
                  yIsConstant=true, yIsSet=false,
                  zIsConstant=true, zIsSet=false;

  public ControlTraceSet (Object _visual) { super (_visual); }

  protected Drawable createDrawable (Object _drawable) {
    x = y = z = null;
    xOne = new double[]{0.0}; yOne = new double[]{0.0}; zOne = new double[]{0.0};
    xIsConstant = yIsConstant = zIsConstant = true;
    xIsSet = yIsSet = zIsSet = false;
    norepeat = false;
    if (_drawable instanceof ElementSet) elementSet = (ElementSet) _drawable;
    else elementSet = new ElementSet(1,InteractiveTrace.class);
/*
    allposValues     = new ObjectValue[3];
    allposValues[0]  = new ObjectValue(theXs = elementSet.getXs());
    allposValues[1]  = new ObjectValue(theYs = elementSet.getYs());
    allposValues[2]  = new ObjectValue(theZs = elementSet.getZs());
    allsizesValue    = new ObjectValue[3];
    allsizesValue[0] = new ObjectValue(theSizeXs = elementSet.getSizeXs());
    allsizesValue[1] = new ObjectValue(theSizeYs = elementSet.getSizeYs());
    allsizesValue[2] = new ObjectValue(theSizeZs = elementSet.getSizeZs());
    elementSet.addListener(this);
    defaultFont = font = elementSet.elementAt(0).getStyle().getFont();
*/
    return elementSet;
  }

  public void initialize () { // Overwrites default initialize
    for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).clear();
    xIsSet = yIsSet = zIsSet = false;
  }

  public void reset () { // Overwrites default reset
    for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).clear();
    xIsSet = yIsSet = zIsSet = false;
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("maxpoints");
      infoList.add ("skippoints");
      infoList.add ("active");
      infoList.add ("norepeat");
      infoList.add ("connected");
      infoList.add ("markersize");
      infoList.add ("positionx");
      infoList.add ("positiony");
      infoList.add ("positionz");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("maxpoints"))   return "int|int[] PREVIOUS";
    if (_property.equals("skippoints"))  return "int|int[]";
    if (_property.equals("active"))      return "boolean|boolean[] BASIC";
    if (_property.equals("norepeat"))    return "boolean|boolean[] BASIC";
    if (_property.equals("connected"))   return "boolean|boolean[] BASIC";
    if (_property.equals("style"))       return "MarkerShape|int|int[] BASIC";
    if (_property.equals("markersize"))  return "int|int[] BASIC";

    if (_property.equals("positionx"))   return "int|double|double[] BASIC";
    if (_property.equals("positiony"))   return "int|double|double[] BASIC";
    if (_property.equals("positionz"))   return "int|double|double[] BASIC";

    if (_property.equals("sizex"))       return "int|double BASIC HIDDEN";
    if (_property.equals("sizey"))       return "int|double BASIC HIDDEN";
    if (_property.equals("sizez"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scalex"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scaley"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scalez"))       return "int|double BASIC HIDDEN";
    if (_property.equals("enabledSecondary")) return "boolean HIDDEN";
    if (_property.equals("resolution"))       return "Resolution BASIC HIDDEN";
    if (_property.equals("font"))             return "Font|Object  BASIC HIDDEN";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Variable properties
// ------------------------------------------------

  public ControlElement setProperty(String _property, String _value) {
    _property = _property.trim();
    if (_property.equals("x")) {
      if (_value==null) xIsConstant = true;
      else {
        Value constant = Value.parseConstantOrArray(_value,true);
        xIsConstant = (constant!=null);
        xOne[0] = 0.0;
      }
//        else xIsConstant = false;
    }
    else if (_property.equals("y")) {
      if (_value==null) yIsConstant = true;
      else {
        Value constant = Value.parseConstantOrArray(_value,true);
        yIsConstant = (constant!=null);
        yOne[0] = 0.0;
      }
//        else yIsConstant = false;
    }
    else if (_property.equals("z")) {
      if (_value==null) zIsConstant = true;
      else {
        Value constant = Value.parseConstantOrArray(_value,true);
        zIsConstant = (constant!=null);
        zOne[0] = 0.0;
      }
//        else zIsConstant = false;
    }
    // Backwards compatibility
    if (_property.equals("makershape"))  return super.setProperty ("style",_value);
    if (_property.equals("linecolor"))   return super.setProperty ("color",_value);
    if (_property.equals("markercolor")) return super.setProperty ("secondaryColor",_value);
    return super.setProperty(_property,_value);
  }

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case INPUT_X :
        if (_value.getObject() instanceof double[]) x = (double[]) _value.getObject();
        else { xOne[0] = _value.getDouble(); x = xOne; }
        if (yIsConstant) return;
        if (yIsSet && (zIsConstant || zIsSet)) {
          if (zIsConstant) addAllPoints(x,y,zOne);
          else             addAllPoints(x,y,z);
          xIsSet = yIsSet = zIsSet = false;
        }
        else xIsSet = true;
        break;
      case INPUT_Y :
        if (_value.getObject() instanceof double[]) y = (double[]) _value.getObject();
        else { yOne[0] = _value.getDouble(); y = yOne; }
        if (xIsConstant) return;
        if (xIsSet && (zIsConstant || zIsSet)) {
          if (zIsConstant) addAllPoints(x,y,zOne);
          else             addAllPoints(x,y,z);
          xIsSet = yIsSet = zIsSet = false;
        }
        else yIsSet = true;
        break;
      case INPUT_Z :
        if (_value.getObject() instanceof double[]) z = (double[]) _value.getObject();
        else { zOne[0] = _value.getDouble(); z = zOne; }
        if (xIsConstant || yIsConstant) return;
        if (xIsSet && yIsSet) {
          addAllPoints(x,y,z);
          xIsSet = yIsSet = zIsSet = false;
        }
        else zIsSet = true;
        break;

    case 0 :
      if (_value.getObject() instanceof int[]) {
          int[] val = (int[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setMaximumPoints(val[i]);
        }
        else {
          int val = _value.getInteger();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setMaximumPoints(val);
        }
      break;
    case 1 :
      if (_value.getObject() instanceof int[]) {
          int[] val = (int[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setSkip(val[i]);
        }
        else {
          int val = _value.getInteger();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setSkip(val);
        }
      break;

    case 2 :
      if (_value.getObject() instanceof boolean[]) {
          boolean[] val = (boolean[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setActive(val[i]);
        }
        else {
          boolean val = _value.getBoolean();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setActive(val);
        }
      break;
    case 3 :
      if (_value.getObject() instanceof boolean[]) {
          boolean[] val = (boolean[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setIgnoreEqualPoints(val[i]);
        }
        else {
          boolean val = _value.getBoolean();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setIgnoreEqualPoints(val);
        }
      break;
    case 4 :
      if (_value.getObject() instanceof boolean[]) {
          boolean[] val = (boolean[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setConnected(val[i]);
        }
        else {
          boolean val = _value.getBoolean();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setConnected(val);
        }
      break;
    case 5 :
      if (_value.getObject() instanceof int[]) {
          int[] val = (int[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeSize(val[i]);
        }
        else {
          int val = _value.getInteger();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeSize(val);
        }
      break;
    case 6 : super.setValue(POSITION_X,_value);              break;
    case 7 : super.setValue(POSITION_Y,_value);              break;
    case 8 : super.setValue(POSITION_Z,_value);              break;

    case MY_STYLE :
      if (_value.getObject() instanceof int[]) {
          int[] val = (int[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeType(val[i]);
        }
        else {
          int val = _value.getInteger();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeType(val);
        }
      break;
    default: super.setValue(_index-TRACESET_ADDED,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case INPUT_X : xIsConstant = true; break;
      case INPUT_Y : yIsConstant = true; break;
      case INPUT_Z : zIsConstant = true; break;

      case 0 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setMaximumPoints(0); break;
      case 1 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setSkip(0);          break;
      case 2 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setActive(true);     break;
      case 3 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setIgnoreEqualPoints(false); break;
      case 4 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setConnected(true);          break;
      case 5 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeSize(5); break;
      case 6 : super.setDefaultValue(POSITION_X);  break;
      case 7 : super.setDefaultValue(POSITION_Y);  break;
      case 8 : super.setDefaultValue(POSITION_Z);  break;

      case MY_STYLE : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) ((InteractiveTrace)elementSet.elementAt(i)).setShapeType(InteractiveParticle.NONE); break;
      default: super.setDefaultValue(_index-TRACESET_ADDED); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case INPUT_X : case INPUT_Y : case INPUT_Z :
      case 0 : case 1 : case 2 : case 3 :  case 4 :
      case 5 :
        return null;

      case 6 : return super.getValue(POSITION_X);
      case 7 : return super.getValue(POSITION_Y);
      case 8 : return super.getValue(POSITION_Z);

      default: return super.getValue (_index-TRACESET_ADDED);
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  static private final int[] posSpot = {6,7,8};
  static private final int[] sizeSpot = {SIZE_X+TRACESET_ADDED,SIZE_Y+TRACESET_ADDED,SIZE_Z+TRACESET_ADDED};

  int[] getPosSpot ()  { return posSpot; }
  int[] getSizeSpot ()  { return sizeSpot; }

// -------------------------------------
// Private methods
// -------------------------------------

  private void addAllPoints(double [] a, double [] b, double [] c) {
    int maxa = a.length-1, maxb = b.length-1, maxc = c.length-1;
    for (int i=0,ia=0,ib=0,ic=0,n = elementSet.getNumberOfElements(); i<n; i++) {
      ( (InteractiveTrace) elementSet.elementAt(i)).addPoint(a[ia], b[ib], c[ic]);
      if (ia<maxa) ia++;
      if (ib<maxb) ib++;
      if (ic<maxc) ic++;
    }
  }

} // End of class