/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) October 2003 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.displayejs.InteractiveParticle;
import org.opensourcephysics.displayejs.InteractiveTrace;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.Value;


/**
 * An interactive particle
 */
public class ControlTrace extends ControlInteractiveElement {
  static final protected int TRACE_ADDED=9;
  static protected final int INPUT_X = POSITION_X+TRACE_ADDED;
  static protected final int INPUT_Y = POSITION_Y+TRACE_ADDED;
  static protected final int INPUT_Z = POSITION_Z+TRACE_ADDED;
  static protected final int MY_STYLE = STYLE + TRACE_ADDED;

  protected InteractiveTrace trace;
  protected double x, y, z;

  // Trace is very peculiar, because the order in which you set
  // a property DOES affectthe output. This is because Trace is
  // a class in which new graphical elements are ADDED
  static final protected int CANBESET=9;
  private boolean[] isConstant, isSet;


  public ControlTrace (Object _visual) { super (_visual); }

  protected Drawable createDrawable (Object _drawable) {
    x = y = z = 0.0;
    isConstant = new boolean[CANBESET];
    isSet      = new boolean[CANBESET];
    for (int i = 0; i<CANBESET; i++) { isConstant[i] = true; isSet[i] = false; }
    if (_drawable instanceof InteractiveTrace) trace = (InteractiveTrace) _drawable;
    else trace = new InteractiveTrace();
    return trace;
  }

  public void initialize () { // Overwrites default initialize
    for (int i = 0; i<CANBESET; i++) isSet[i] = false;
  }

  public void reset () { // Overwrites default reset
    trace.clear();
    for (int i = 0; i<CANBESET; i++) isSet[i] = false;
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("maxpoints");
      infoList.add ("skippoints");
      infoList.add ("active");
      infoList.add ("norepeat");
      infoList.add ("connected");
      infoList.add ("markersize");
      infoList.add ("positionx");
      infoList.add ("positiony");
      infoList.add ("positionz");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("maxpoints"))   return "int PREVIOUS";
    if (_property.equals("skippoints"))  return "int";
    if (_property.equals("active"))      return "boolean BASIC";
    if (_property.equals("norepeat"))    return "boolean BASIC";
    if (_property.equals("connected"))   return "boolean BASIC";
    if (_property.equals("style"))          return "MarkerShape|int BASIC";
    if (_property.equals("markersize"))  return "int BASIC";
    if (_property.equals("positionx"))   return "int|double BASIC";
    if (_property.equals("positiony"))   return "int|double BASIC";
    if (_property.equals("positionz"))   return "int|double BASIC";

    if (_property.equals("sizex"))       return "int|double BASIC HIDDEN";
    if (_property.equals("sizey"))       return "int|double BASIC HIDDEN";
    if (_property.equals("sizez"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scalex"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scaley"))       return "int|double BASIC HIDDEN";
    if (_property.equals("scalez"))       return "int|double BASIC HIDDEN";
    if (_property.equals("enabledSecondary")) return "boolean HIDDEN";
    if (_property.equals("resolution"))     return "Resolution BASIC HIDDEN";
    if (_property.equals("font"))           return "Font|Object  BASIC HIDDEN";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public ControlElement setProperty(String _property, String _value) {
    _property = _property.trim();
    int canBeSet = -1;
    if      (_property.equals("maxpoints"))  canBeSet = 0;
    else if (_property.equals("skippoints")) canBeSet = 1;
    else if (_property.equals("active"))     canBeSet = 2;
    else if (_property.equals("norepeat"))   canBeSet = 3;
    else if (_property.equals("connected"))  canBeSet = 4;
    else if (_property.equals("markersize")) canBeSet = 5;
    else if (_property.equals("x"))          canBeSet = 6;
    else if (_property.equals("y"))          canBeSet = 7;
    else if (_property.equals("z"))          canBeSet = 8;
    if (canBeSet>=0) {
      if (_value==null) isConstant[canBeSet] = true;
      else {
        Value constant = Value.parseConstant(_value,true);
        isConstant[canBeSet] = (constant!=null);
      }
    }

    // Backwards compatibility
    if (_property.equals("makershape"))  return super.setProperty ("style",_value);
    if (_property.equals("linecolor"))   return super.setProperty ("color",_value);
    if (_property.equals("markercolor")) return super.setProperty ("secondaryColor",_value);
    return super.setProperty(_property,_value);
  }

  private void doIt () {
    for (int i=0; i<CANBESET; i++) if (!isConstant[i] && !isSet[i]) return; // wait for all of them to be set
    trace.addPoint(x,y,z);
    for (int i=0; i<CANBESET; i++) isSet[i] = false;
  }

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case INPUT_X : x = _value.getDouble(); isSet[6] = true; doIt(); break;
      case INPUT_Y : y = _value.getDouble(); isSet[7] = true; doIt(); break;
      case INPUT_Z : z = _value.getDouble(); isSet[8] = true; doIt(); break;

      case 0 : trace.setMaximumPoints(_value.getInteger());     isSet[0] = true; doIt(); break;
      case 1 : trace.setSkip(_value.getInteger());              isSet[1] = true; doIt(); break;
      case 2 : trace.setActive(_value.getBoolean());            isSet[2] = true; doIt(); break;
      case 3 : trace.setIgnoreEqualPoints(_value.getBoolean()); isSet[3] = true; doIt(); break;
      case 4 : trace.setConnected(_value.getBoolean());         isSet[4] = true; doIt(); break;
      case 5 : trace.setShapeSize(_value.getInteger());         isSet[5] = true; doIt(); break;
      case 6 : super.setValue(POSITION_X,_value); break; // These affect the whole trace,
      case 7 : super.setValue(POSITION_Y,_value); break; // not just the added point
      case 8 : super.setValue(POSITION_Z,_value); break;

      case MY_STYLE : trace.setShapeType(_value.getInteger());         break;
      default: super.setValue(_index-TRACE_ADDED,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case INPUT_X : isConstant[6] = true; break;
      case INPUT_Y : isConstant[7] = true; break;
      case INPUT_Z : isConstant[8] = true; break;

      case 0 : trace.setMaximumPoints(0);         isConstant[0] = true; break;
      case 1 : trace.setSkip(0);                  isConstant[1] = true; break;
      case 2 : trace.setActive(true);             isConstant[2] = true; break;
      case 3 : trace.setIgnoreEqualPoints(false); isConstant[3] = true; break;
      case 4 : trace.setConnected(true);          isConstant[4] = true; break;
      case 5 : trace.setShapeSize(5);             isConstant[5] = true; break;
      case 6 : super.setDefaultValue(POSITION_X);  break;
      case 7 : super.setDefaultValue(POSITION_Y);  break;
      case 8 : super.setDefaultValue(POSITION_Z);  break;

      case MY_STYLE : trace.setShapeType(InteractiveParticle.NONE); break;
      default: super.setDefaultValue(_index-TRACE_ADDED); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case INPUT_X : case INPUT_Y : case INPUT_Z :
      case 0 : case 1 : case 2 : case 3 :  case 4 :
      case 5 :
        return null;

      case 6 : return super.getValue(POSITION_X);
      case 7 : return super.getValue(POSITION_Y);
      case 8 : return super.getValue(POSITION_Z);

      default: return super.getValue (_index-TRACE_ADDED);
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  static private final int[] posSpot = {6,7,8};
  static private final int[] sizeSpot = {3+TRACE_ADDED,4+TRACE_ADDED,5+TRACE_ADDED};


  int[] getPosSpot ()  { return posSpot; }
  int[] getSizeSpot ()  { return sizeSpot; }

} // End of class