/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) October 2003 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionTargetElementSize;
import org.opensourcephysics.displayejs.InteractionTargetPoligonPoint;
import org.opensourcephysics.displayejs.InteractivePoligon;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.ObjectValue;
import org.opensourcephysics.ejs.control.value.Value;


/**
 * An interactive particle
 */
public class ControlPoligon extends ControlInteractiveElement {
  static protected final int POLIGON_ADDED = 7;
  static protected final int INPUT_X = POSITION_X+POLIGON_ADDED;
  static protected final int INPUT_Y = POSITION_Y+POLIGON_ADDED;
  static protected final int INPUT_Z = POSITION_Z+POLIGON_ADDED;
  static protected final int MY_SIZE_X = SIZE_X+POLIGON_ADDED;
  static protected final int MY_SIZE_Y = SIZE_Y+POLIGON_ADDED;
  static protected final int MY_SIZE_Z = SIZE_Z+POLIGON_ADDED;

  static protected final int MY_LINE_COLOR = SECONDARY_COLOR+POLIGON_ADDED;
  static protected final int MY_FILL_COLOR = PRIMARY_COLOR+POLIGON_ADDED;

  protected InteractivePoligon poligon;
  protected ObjectValue[] coordinatesValues;
  protected int insensitiveCorner=-1;

  public ControlPoligon (Object _visual) { super (_visual); }

  protected Drawable createDrawable (Object _drawable) {
    if (_drawable instanceof InteractivePoligon) poligon = (InteractivePoligon) _drawable;
    else poligon = new InteractivePoligon();
    double[][]data = poligon.getData();
    coordinatesValues     = new ObjectValue[3];
    if (data!=null) {
      coordinatesValues[0] = new ObjectValue(data[0]);
      coordinatesValues[1] = new ObjectValue(data[1]);
      coordinatesValues[2] = new ObjectValue(data[2]);
    }
    return poligon;
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("maxpoints");
      infoList.add ("connected");
      infoList.add ("closed");
      infoList.add ("fixed");
      infoList.add ("positionx");
      infoList.add ("positiony");
      infoList.add ("positionz");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("maxpoints")) return "int Previous";
    if (_property.equals("x"))         return "int|double|double[]";
    if (_property.equals("y"))         return "int|double|double[]";
    if (_property.equals("z"))         return "int|double|double[]";
    if (_property.equals("connected")) return "boolean[] BASIC";
    if (_property.equals("closed"))    return "boolean BASIC";

    if (_property.equals("fixed"))       return "int|boolean[] BASIC";

    if (_property.equals("positionx"))   return "int|double BASIC";
    if (_property.equals("positiony"))   return "int|double BASIC";
    if (_property.equals("positionz"))   return "int|double BASIC";

    if (_property.equals("scalex"))         return "int|double BASIC HIDDEN";
    if (_property.equals("scaley"))         return "int|double BASIC HIDDEN";
    if (_property.equals("scalez"))         return "int|double BASIC HIDDEN";

    if (_property.equals("style"))          return "MarkerShape|int BASIC HIDDEN";
    if (_property.equals("elementposition"))return "ElementPosition|int BASIC HIDDEN";
    if (_property.equals("angle"))          return "int|double BASIC HIDDEN";
    if (_property.equals("resolution"))     return "Resolution BASIC HIDDEN";

    if (_property.equals("font"))           return "Font|Object  BASIC HIDDEN";

    return super.getPropertyInfo(_property);
  }

  // Backwards compatibility
  public ControlElement setProperty(String _property, String _value) {
    _property = _property.trim();
    if (_property.equals("linecolor")) return super.setProperty ("secondaryColor",_value);
    return super.setProperty(_property,_value);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 :
        if (poligon.getNumberOfPoints()!=_value.getInteger()) {
          poligon.setNumberOfPoints (_value.getInteger());
          double[][] data = poligon.getData();
          if (data!=null) {
            coordinatesValues[0] = new ObjectValue(data[0]);
            coordinatesValues[1] = new ObjectValue(data[1]);
            coordinatesValues[2] = new ObjectValue(data[2]);
          }
          if (insensitiveCorner >-1 && insensitiveCorner<poligon.getNumberOfPoints()) poligon.setPointSizeEnabled(insensitiveCorner,false);
          else insensitiveCorner = -1;
        }
        break;
      case 1 : if (_value.getObject() instanceof boolean[]) poligon.setConnections((boolean[])_value.getObject()); break;
      case 2 : poligon.setClosed (_value.getBoolean());               break;
      case 3 : // fixed corner(s)
        if (_value.getObject() instanceof boolean[]) {
          boolean[] val = (boolean[]) _value.getObject();
          for (int i=0, n=Math.min(val.length,poligon.getNumberOfPoints()); i<n; i++)  poligon.setPointSizeEnabled(i,!val[i]);
          insensitiveCorner = -1;
        }
        else {
          if (insensitiveCorner<0) for (int i=0, n=poligon.getNumberOfPoints(); i<n; i++) poligon.setPointSizeEnabled(i,true);
          else poligon.setPointSizeEnabled(insensitiveCorner,true);
          poligon.setPointSizeEnabled(insensitiveCorner=_value.getInteger(),false);
        }
      break;
      case 4 : super.setValue(POSITION_X,_value);              break;
      case 5 : super.setValue(POSITION_Y,_value);              break;
      case 6 : super.setValue(POSITION_Z,_value);              break;

      case INPUT_X :
        if (_value.getObject() instanceof double[]) poligon.setXs((double[])_value.getObject());
        else poligon.setXs(_value.getDouble());
        // if (insensitiveCorner>-1) poligon.setPointSizeEnabled(insensitiveCorner,false);
        break;
      case INPUT_Y :
        if (_value.getObject() instanceof double[]) poligon.setYs((double[])_value.getObject());
        else poligon.setYs(_value.getDouble());
        // if (insensitiveCorner>-1) poligon.setPointSizeEnabled(insensitiveCorner,false);
        break;
      case INPUT_Z :
        if (_value.getObject() instanceof double[]) poligon.setZs((double[])_value.getObject());
        else poligon.setZs(_value.getDouble());
        // if (insensitiveCorner>-1) poligon.setPointSizeEnabled(insensitiveCorner,false);
        break;

      case MY_FILL_COLOR : super.setValue(SECONDARY_COLOR,_value) ; break;
      case MY_LINE_COLOR : super.setValue(PRIMARY_COLOR,_value) ; break;
      default: super.setValue(_index-7,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 : poligon.setNumberOfPoints (0); break;
      case 1 : poligon.setConnections(null);  break;
      case 2 : poligon.setClosed (true);      break;
      case 3 : for (int i=0, n=poligon.getNumberOfPoints(); i<n; i++) poligon.setPointSizeEnabled(i,true); insensitiveCorner = -1; break;
      case 4 : super.setDefaultValue(POSITION_X);     break;
      case 5 : super.setDefaultValue(POSITION_Y);     break;
      case 6 : super.setDefaultValue(POSITION_Z);     break;

      case INPUT_X : poligon.setXs(0.0); break;
      case INPUT_Y : poligon.setYs(0.0); break;
      case INPUT_Z : poligon.setZs(0.0); break;

      case  MY_SIZE_X : sizeValues[0].value=1.0; myElement.setSizeX(sizeValues[0].value*scalex); break;
      case  MY_SIZE_Y : sizeValues[1].value=1.0; myElement.setSizeY(sizeValues[1].value*scaley); break;
      case  MY_SIZE_Z : sizeValues[2].value=1.0; myElement.setSizeZ(sizeValues[2].value*scalez); break;

      case MY_FILL_COLOR : super.setDefaultValue(SECONDARY_COLOR) ; break;
      case MY_LINE_COLOR : super.setDefaultValue(PRIMARY_COLOR) ; break;
      default: super.setDefaultValue(_index-7); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case INPUT_X : return coordinatesValues[0];
      case INPUT_Y : return coordinatesValues[1];
      case INPUT_Z : return coordinatesValues[2];

      case 0 : case 1 : case 2 : case 3 :  return null;

      case 4 : return super.getValue(POSITION_X);
      case 5 : return super.getValue(POSITION_Y);
      case 6 : return super.getValue(POSITION_Z);

      default: return super.getValue (_index-7);
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  static private final int[] posSpot = {4,5,6};
  static private final int[] sizeSpot = {MY_SIZE_X,MY_SIZE_Y,MY_SIZE_Z};
  static private final int[] coordinatesSpot = {INPUT_X,INPUT_Y,INPUT_Z};

  int[] getPosSpot ()  { return posSpot; }
  int[] getSizeSpot ()  { return sizeSpot; }
  int[] getCoordinatesSpot ()  { return coordinatesSpot; }

  public void interactionPerformed(InteractionEvent _event) {
  switch (_event.getID()) {
    case InteractionEvent.MOUSE_PRESSED :
      invokeActions (ControlInteractiveElement.ACTION_PRESS);
      // Do not break!
    case InteractionEvent.MOUSE_DRAGGED :
      if (_event.getTarget().getClass()==InteractionTargetPoligonPoint.class) {
        variablesChanged (getCoordinatesSpot(),coordinatesValues);
      }
      else if (_event.getTarget().getClass()==InteractionTargetElementSize.class) {
        if (scalex!=0.0) sizeValues[0].value = myElement.getSizeX()/scalex; else sizeValues[0].value = myElement.getSizeX();
        if (scaley!=0.0) sizeValues[1].value = myElement.getSizeY()/scaley; else sizeValues[1].value = myElement.getSizeY();
        if (scalez!=0.0) sizeValues[2].value = myElement.getSizeZ()/scalez; else sizeValues[2].value = myElement.getSizeZ();
// System.out.println("Size is now "+sizeValues[0].value+", "+sizeValues[1].value);
        variablesChanged (getSizeSpot(),sizeValues);
      }
      else {
        posValues[0].value = myElement.getX();
        posValues[1].value = myElement.getY();
        posValues[2].value = myElement.getZ();
        variablesChanged (getPosSpot(),posValues);
      }
      break;
    case InteractionEvent.MOUSE_RELEASED :
      invokeActions (ControlElement.ACTION);
      break;
  }
} // End of interaction method




} // End of class