/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) October 2003 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import java.awt.Stroke;

import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionListener;
import org.opensourcephysics.displayejs.InteractionTargetElementSize;
import org.opensourcephysics.displayejs.InteractiveElement;
import org.opensourcephysics.displayejs.Resolution;
import org.opensourcephysics.displayejs.Style;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.IntegerValue;
import org.opensourcephysics.ejs.control.value.Value;

public abstract class ControlInteractiveElement extends ControlDrawable3D implements InteractionListener {
  static protected final int ACTION_PRESS   = 10;

  static protected final int POSITION_X = 0;
  static protected final int POSITION_Y = 1;
  static protected final int POSITION_Z = 2;
  static protected final int SIZE_X = 3;
  static protected final int SIZE_Y = 4;
  static protected final int SIZE_Z = 5;
  static protected final int ENABLED = 7;
  static protected final int ENABLED_SECONDARY = 8;
  static protected final int STYLE = 14;
  static protected final int PRIMARY_COLOR = 18;
  static protected final int SECONDARY_COLOR = 19;

  protected InteractiveElement myElement;
  protected DoubleValue[] posValues ={ new DoubleValue(0.0), new DoubleValue(0.0), new DoubleValue(0.0)};
  protected DoubleValue[] sizeValues; // ={ new DoubleValue(0.1), new DoubleValue(0.1), new DoubleValue(0.1)};
  protected double scalex=1.0, scaley=1.0, scalez=1.0, lineWidth=1.0;
//  protected java.awt.Stroke stroke=null;
  protected java.awt.Font font, defaultFont;

  public ControlInteractiveElement (Object _visual) {
    super (_visual);
    if (sizeValues==null)
      sizeValues = new DoubleValue[] { new DoubleValue(0.1), new DoubleValue(0.1), new DoubleValue(0.1)};
    myElement = (InteractiveElement) getDrawable();
    myElement.addListener(this);
    myElement.setDataObject(this);
    defaultFont = font = myElement.getStyle().getFont();
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("x");
      infoList.add ("y");
      infoList.add ("z");
      infoList.add ("sizex");
      infoList.add ("sizey");
      infoList.add ("sizez");

      infoList.add ("visible");
      infoList.add ("enabled");
      infoList.add ("enabledSecondary");

      infoList.add ("scalex");
      infoList.add ("scaley");
      infoList.add ("scalez");

      infoList.add ("group");
      infoList.add ("groupEnabled");

      infoList.add ("style");
      infoList.add ("elementposition");
      infoList.add ("angle");
      infoList.add ("resolution");

      infoList.add ("color");
      infoList.add ("secondaryColor");
      infoList.add ("stroke");
      infoList.add ("font");

      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("x"))           return "int|double";
    if (_property.equals("y"))           return "int|double";
    if (_property.equals("z"))           return "int|double";
    if (_property.equals("sizex"))       return "int|double";
    if (_property.equals("sizey"))       return "int|double";
    if (_property.equals("sizez"))       return "int|double";

    if (_property.equals("visible"))        return "boolean";
    if (_property.equals("enabled"))        return "boolean";
    if (_property.equals("enabledSecondary")) return "boolean";

    if (_property.equals("scalex"))         return "int|double BASIC";
    if (_property.equals("scaley"))         return "int|double BASIC";
    if (_property.equals("scalez"))         return "int|double BASIC";

    if (_property.equals("group"))          return "String BASIC HIDDEN";
    if (_property.equals("groupEnabled"))   return "boolean BASIC HIDDEN";

    if (_property.equals("style"))          return "MarkerShape|int BASIC";
    if (_property.equals("elementposition"))return "ElementPosition|int BASIC";
    if (_property.equals("angle"))          return "int|double BASIC";
    if (_property.equals("resolution"))     return "Resolution|String|Object BASIC";

    if (_property.equals("color"))          return "Color|Object BASIC";
    if (_property.equals("secondaryColor")) return "Color|Object BASIC";
    if (_property.equals("stroke"))         return "int|double|Object BASIC";
    if (_property.equals("font"))           return "Font|Object  BASIC";

    if (_property.equals("action"))      return "Action CONSTANT";
    if (_property.equals("pressaction")) return "Action CONSTANT";
    if (_property.equals("dragaction"))  return "Action CONSTANT";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case  POSITION_X : myElement.setX(posValues[0].value=_value.getDouble());        break;
      case  POSITION_Y : myElement.setY(posValues[1].value=_value.getDouble());        break;
      case  POSITION_Z : myElement.setZ(posValues[2].value=_value.getDouble());        break;
      case  SIZE_X : sizeValues[0].value=_value.getDouble(); myElement.setSizeX(sizeValues[0].value*scalex); break;
      case  SIZE_Y : sizeValues[1].value=_value.getDouble(); myElement.setSizeY(sizeValues[1].value*scaley); break;
      case  SIZE_Z : sizeValues[2].value=_value.getDouble(); myElement.setSizeZ(sizeValues[2].value*scalez); break;
      case  6 : myElement.setVisible(_value.getBoolean()); break;
        // Some elements may change this. For instance, arrows
      case  ENABLED           : myElement.setEnabled(InteractiveElement.TARGET_POSITION,_value.getBoolean()); break;
      case  ENABLED_SECONDARY : myElement.setEnabled(InteractiveElement.TARGET_SIZE,_value.getBoolean()); break;

      case  9 : scalex = _value.getDouble(); myElement.setSizeX(sizeValues[0].value*scalex); break;
      case 10 : scaley = _value.getDouble(); myElement.setSizeY(sizeValues[1].value*scaley); break;
      case 11 : scalez = _value.getDouble(); myElement.setSizeZ(sizeValues[2].value*scalez); break;

      case 12 : break; // Groups... how to implement this???
      case 13 : myElement.setGroupEnabled(_value.getBoolean()); break;

      case STYLE : break; // To be implemented by subclasses
      case 15 : myElement.getStyle().setPosition(_value.getInteger()); break;
      case 16 :
        if (_value instanceof IntegerValue) myElement.getStyle().setAngle(_value.getInteger()*ControlDrawingPanel3D.TO_RAD);
        else myElement.getStyle().setAngle(_value.getDouble());
        break;
      case 17 :
        if (_value.getObject() instanceof Resolution) myElement.setResolution((Resolution)_value.getObject());
        else {
          Resolution res = ControlDrawable3D.decodeResolution (_value.toString());
          if (res!=null) myElement.setResolution(res);
        }
        break;

      case PRIMARY_COLOR     : myElement.getStyle().setEdgeColor((java.awt.Color) _value.getObject()); break;
      case SECONDARY_COLOR   :
        {
        java.awt.Paint fill = (java.awt.Paint) _value.getObject();
        if (fill==NULL_COLOR) fill = null;
        myElement.getStyle().setFillPattern(fill);
        }
        break;
      case 20 :
        if (_value.getObject() instanceof Stroke) myElement.getStyle().setEdgeStroke((Stroke) _value.getObject());
        else if (lineWidth!=_value.getDouble()) {
          lineWidth = _value.getDouble();
          myElement.getStyle().setEdgeStroke(new java.awt.BasicStroke((float) lineWidth));
        }
        break;
      case 21 : if (_value.getObject() instanceof java.awt.Font) {
          java.awt.Font newFont = (java.awt.Font) _value.getObject();
          if (newFont!=font) myElement.getStyle().setFont(font=newFont);
        }
        break;
      case 22 : // pressaction
        removeAction (ControlInteractiveElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlInteractiveElement.ACTION_PRESS,_value.getString());
        return;
      case 23 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        return;
      case 24 : // action
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        return;

      default: super.setValue(_index-25,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case  POSITION_X : myElement.setX(posValues[0].value=0.0);        break;
      case  POSITION_Y : myElement.setY(posValues[1].value=0.0);        break;
      case  POSITION_Z : myElement.setZ(posValues[2].value=0.0);        break;
      case  SIZE_X : sizeValues[0].value=0.1; myElement.setSizeX(sizeValues[0].value*scalex); break;
      case  SIZE_Y : sizeValues[1].value=0.1; myElement.setSizeY(sizeValues[1].value*scaley); break;
      case  SIZE_Z : sizeValues[2].value=0.1; myElement.setSizeZ(sizeValues[2].value*scalez); break;
      case  6 : myElement.setVisible(true); break;

        // Some elements may change this. For instance, arrows
      case  ENABLED           : myElement.setEnabled(InteractiveElement.TARGET_POSITION,true); break;
      case  ENABLED_SECONDARY : myElement.setEnabled(InteractiveElement.TARGET_SIZE,false); break;

      case  9 : scalex = 1.0; myElement.setSizeX(sizeValues[0].value); break;
      case 10 : scaley = 1.0; myElement.setSizeY(sizeValues[1].value); break;
      case 11 : scalez = 1.0; myElement.setSizeZ(sizeValues[2].value); break;

      case 12 : break; // Groups... how to implement this???
      case 13 : myElement.setGroupEnabled(true); break;

      case STYLE : break; // To be implemented by subclasses
      case 15 : myElement.getStyle().setPosition(Style.CENTERED); break;
      case 16 : myElement.getStyle().setAngle(0.0); break;
      case 17 : myElement.setResolution(null); break;

      case PRIMARY_COLOR     : myElement.getStyle().setEdgeColor(java.awt.Color.black); break;
      case SECONDARY_COLOR   : myElement.getStyle().setFillPattern(java.awt.Color.blue); break;
      case 20 : myElement.getStyle().setEdgeStroke(new java.awt.BasicStroke()); break;
      case 21 : myElement.getStyle().setFont(font=defaultFont); break;

      case 22 : removeAction (ControlInteractiveElement.ACTION_PRESS,getProperty("pressaction")); return;
      case 23 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));         return;
      case 24 : removeAction (ControlElement.ACTION,getProperty("action"));                       return;

      default: super.setDefaultValue(_index-25); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case POSITION_X : return posValues[0];
      case POSITION_Y : return posValues[1];
      case POSITION_Z : return posValues[2];
      case SIZE_X : return sizeValues[0];
      case SIZE_Y : return sizeValues[1];
      case SIZE_Z : return sizeValues[2];
      default: return null; // Doesn't inherit
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  static private final int[] posSpot = {0,1,2};
  static private final int[] sizeSpot = {3,4,5};

  int[] getPosSpot ()  { return posSpot; }
  int[] getSizeSpot ()  { return sizeSpot; }

  public void interactionPerformed(InteractionEvent _event) {
    switch (_event.getID()) {
      case InteractionEvent.MOUSE_PRESSED :
        if (myParent instanceof ControlDrawingPanel3D) {
          ((ControlDrawingPanel3D) myParent).setSelectedDrawable (this);
        }
        invokeActions (ControlInteractiveElement.ACTION_PRESS);
        // Do not break!
      case InteractionEvent.MOUSE_DRAGGED :
        if (_event.getTarget().getClass()==InteractionTargetElementSize.class) {
          if (scalex!=0.0) sizeValues[0].value = myElement.getSizeX()/scalex; else sizeValues[0].value = myElement.getSizeX();
          if (scaley!=0.0) sizeValues[1].value = myElement.getSizeY()/scaley; else sizeValues[1].value = myElement.getSizeY();
          if (scalez!=0.0) sizeValues[2].value = myElement.getSizeZ()/scalez; else sizeValues[2].value = myElement.getSizeZ();
// System.out.println("Size is now "+sizeValues[0].value+", "+sizeValues[1].value);
          variablesChanged (getSizeSpot(),sizeValues);
        }
        else {
          posValues[0].value = myElement.getX();
          posValues[1].value = myElement.getY();
          posValues[2].value = myElement.getZ();
          variablesChanged (getPosSpot(),posValues);
        }
        break;
      case InteractionEvent.MOUSE_RELEASED :
        if (myParent instanceof ControlDrawingPanel3D) {
          ((ControlDrawingPanel3D) myParent).setSelectedDrawable (null);
        }
        invokeActions (ControlElement.ACTION);
        break;
    }
  } // End of interaction method


} // End of interface
