/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) October 2003 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Font;
import java.awt.Paint;
import java.awt.Stroke;

import org.opensourcephysics.displayejs.ElementSet;
import org.opensourcephysics.displayejs.InteractionEvent;
import org.opensourcephysics.displayejs.InteractionListener;
import org.opensourcephysics.displayejs.InteractionTargetElementSize;
import org.opensourcephysics.displayejs.InteractionTargetSetElement;
import org.opensourcephysics.displayejs.InteractiveElement;
import org.opensourcephysics.displayejs.Resolution;
import org.opensourcephysics.displayejs.Style;
import org.opensourcephysics.ejs.control.ControlElement;
import org.opensourcephysics.ejs.control.value.IntegerValue;
import org.opensourcephysics.ejs.control.value.ObjectValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * An interactive set of particles
 */
public abstract class ControlElementSet extends ControlDrawable3D implements InteractionListener {
  static protected final int IMAGE = 1;
  static protected final int TEXT = 2;
  static protected final int RADIUS = 3;

  static protected final int POSITION_X = 4;
  static protected final int POSITION_Y = 5;
  static protected final int POSITION_Z = 6;

  static protected final int SIZE_X = 7;
  static protected final int SIZE_Y = 8;
  static protected final int SIZE_Z = 9;

  static protected final int ENABLED = 11;
  static protected final int ENABLED_SECONDARY = 12;
  static protected final int STYLE = 18;
  static protected final int PRIMARY_COLOR = 22;
  static protected final int SECONDARY_COLOR = 23;

  protected ElementSet elementSet;

  protected ObjectValue[] allposValues, allsizesValue;

  protected double[] theXs, theYs, theZs;
  protected double[] theSizeXs, theSizeYs, theSizeZs;
  protected double scalex=1.0, scaley=1.0, scalez=1.0, lineWidth=1.0;
  protected java.awt.Font font, defaultFont;

  public ControlElementSet (Object _visual) {
    super (_visual);
    allposValues     = new ObjectValue[3];
    allposValues[0]  = new ObjectValue(theXs = elementSet.getXs());
    allposValues[1]  = new ObjectValue(theYs = elementSet.getYs());
    allposValues[2]  = new ObjectValue(theZs = elementSet.getZs());
    allsizesValue    = new ObjectValue[3];
    allsizesValue[0] = new ObjectValue(theSizeXs = elementSet.getSizeXs());
    allsizesValue[1] = new ObjectValue(theSizeYs = elementSet.getSizeYs());
    allsizesValue[2] = new ObjectValue(theSizeZs = elementSet.getSizeZs());
    elementSet.addListener(this);
    defaultFont = font = elementSet.elementAt(0).getStyle().getFont();
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("elementnumber");

      // Next two must be here because nothing can appear before elementnumber!!!

      infoList.add ("image");
      infoList.add ("text");
      infoList.add ("radius");

      infoList.add ("x");
      infoList.add ("y");
      infoList.add ("z");
      infoList.add ("sizex");
      infoList.add ("sizey");
      infoList.add ("sizez");

      infoList.add ("visible");
      infoList.add ("enabled");
      infoList.add ("enabledSecondary");

      infoList.add ("scalex");
      infoList.add ("scaley");
      infoList.add ("scalez");

      infoList.add ("group");
      infoList.add ("groupEnabled");

      infoList.add ("style");
      infoList.add ("elementposition");
      infoList.add ("angle");
      infoList.add ("resolution");

      infoList.add ("color");
      infoList.add ("secondaryColor");
      infoList.add ("stroke");
      infoList.add ("font");

      infoList.add ("pressaction");
      infoList.add ("dragaction");
      infoList.add ("action");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("elementnumber")) return "int BASIC PREVIOUS";
    if (_property.equals("image"))       return "HIDDEN";
    if (_property.equals("text"))       return "HIDDEN";
    if (_property.equals("radius"))       return "HIDDEN";

    if (_property.equals("x"))           return "int|double|double[]";
    if (_property.equals("y"))           return "int|double|double[]";
    if (_property.equals("z"))           return "int|double|double[]";
    if (_property.equals("sizex"))       return "int|double|double[]";
    if (_property.equals("sizey"))       return "int|double|double[]";
    if (_property.equals("sizez"))       return "int|double|double[]";

    if (_property.equals("visible"))        return "boolean|boolean[]";
    if (_property.equals("enabled"))        return "boolean|boolean[]";
    if (_property.equals("enabledSecondary")) return "boolean|boolean[]";

    if (_property.equals("scalex"))         return "int|double BASIC";
    if (_property.equals("scaley"))         return "int|double BASIC";
    if (_property.equals("scalez"))         return "int|double BASIC";

    if (_property.equals("group"))          return "String BASIC HIDDEN";
    if (_property.equals("groupEnabled"))   return "boolean BASIC HIDDEN";

    if (_property.equals("style"))          return "MarkerShape|int|int[] BASIC";
    if (_property.equals("elementposition"))return "ElementPosition|int|int[] BASIC";
    if (_property.equals("angle"))          return "int|int[]|double|double[] BASIC";
    if (_property.equals("resolution"))     return "Resolution BASIC";

    if (_property.equals("color"))          return "Color|Object|Object[] BASIC";
    if (_property.equals("secondaryColor")) return "Color|Object|Object[] BASIC";
    if (_property.equals("stroke"))         return "int|double|Object BASIC";
    if (_property.equals("font"))           return "Font|Object  BASIC";

    if (_property.equals("action"))      return "Action CONSTANT";
    if (_property.equals("pressaction")) return "Action CONSTANT";
    if (_property.equals("dragaction"))  return "Action CONSTANT";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Variable properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 :
        if (_value.getInteger()!=elementSet.getNumberOfElements()) {
          elementSet.setNumberOfElements(_value.getInteger());
          allposValues[0].value  = theXs = elementSet.getXs();
          allposValues[1].value  = theYs = elementSet.getYs();
          allposValues[2].value  = theZs = elementSet.getZs();
          allsizesValue[0].value = theSizeXs = elementSet.getSizeXs();
          allsizesValue[1].value = theSizeYs = elementSet.getSizeYs();
          allsizesValue[2].value = theSizeZs = elementSet.getSizeZs();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) {
            if (scalex!=0.0) theSizeXs[i] = elementSet.elementAt(i).getSizeX()/scalex; else theSizeXs[i] = elementSet.elementAt(i).getSizeX();
            if (scaley!=0.0) theSizeYs[i] = elementSet.elementAt(i).getSizeY()/scaley; else theSizeYs[i] = elementSet.elementAt(i).getSizeY();
            if (scalez!=0.0) theSizeZs[i] = elementSet.elementAt(i).getSizeZ()/scalez; else theSizeZs[i] = elementSet.elementAt(i).getSizeZ();
          }
        }
        break;
      case IMAGE : break; // Subclasses will use this
      case TEXT : break; // Subclasses will use this
      case RADIUS : break; // Subclasses will use this

      case POSITION_X :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theXs.length,val.length); i<n; i++) elementSet.elementAt(i).setX(theXs[i] = val[i]);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theXs.length; i<n; i++) elementSet.elementAt(i).setX(theXs[i]=val);
          }
        break;
      case POSITION_Y :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theYs.length,val.length); i<n; i++) elementSet.elementAt(i).setY(theYs[i] = val[i]);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theYs.length; i<n; i++) elementSet.elementAt(i).setY(theYs[i]=val);
          }
        break;
      case POSITION_Z :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theZs.length,val.length); i<n; i++) elementSet.elementAt(i).setZ(theZs[i] = val[i]);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theZs.length; i<n; i++) elementSet.elementAt(i).setZ(theZs[i]=val);
          }
        break;
      case 7 :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theSizeXs.length,val.length); i<n; i++) elementSet.elementAt(i).setSizeX((theSizeXs[i]=val[i])*scalex);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theSizeXs.length; i<n; i++) elementSet.elementAt(i).setSizeX((theSizeXs[i]=val)*scalex);
          }
        break;
      case 8 :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theSizeYs.length,val.length); i<n; i++) elementSet.elementAt(i).setSizeY((theSizeYs[i]=val[i])*scaley);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theSizeYs.length; i<n; i++) elementSet.elementAt(i).setSizeY((theSizeYs[i]=val)*scaley);
          }
        break;
      case 9 :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(theSizeZs.length,val.length); i<n; i++) elementSet.elementAt(i).setSizeZ((theSizeZs[i]=val[i])*scalez);
          }
          else {
            double val = _value.getDouble();
            for (int i=0, n=theSizeZs.length; i<n; i++) elementSet.elementAt(i).setSizeZ((theSizeZs[i]=val)*scalez);
          }
        break;
      case 10 :
        if (_value.getObject() instanceof boolean[]) elementSet.setVisibles((boolean[]) _value.getObject());
        else elementSet.setVisible(_value.getBoolean());
        break;
        // Some elements may change this. For instance, arrows
      case  ENABLED :
        if (_value.getObject() instanceof boolean[]) elementSet.setEnableds(InteractiveElement.TARGET_POSITION,(boolean[]) _value.getObject());
        else elementSet.setEnabled(InteractiveElement.TARGET_POSITION,_value.getBoolean());
        break;
      case  ENABLED_SECONDARY :
        if (_value.getObject() instanceof boolean[]) elementSet.setEnableds(InteractiveElement.TARGET_SIZE,(boolean[]) _value.getObject());
        else elementSet.setEnabled(InteractiveElement.TARGET_SIZE,_value.getBoolean());
        break;

      case 13 :
        scalex = _value.getDouble();
        for (int i=0, n=theSizeXs.length; i<n; i++) elementSet.elementAt(i).setSizeX(theSizeXs[i]*scalex);
        break;
      case 14 :
        scaley = _value.getDouble();
        for (int i=0, n=theSizeYs.length; i<n; i++) elementSet.elementAt(i).setSizeY(theSizeYs[i]*scaley);
        break;
      case 15 :
        scalez = _value.getDouble();
        for (int i=0, n=theSizeZs.length; i<n; i++) elementSet.elementAt(i).setSizeZ(theSizeZs[i]*scalez);
        break;

      case 16 : break; // Groups... how to implement this???
      case 17 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).setGroupEnabled(_value.getBoolean()); break;

      case STYLE : break; // To be implemented by subclasses

      case 19 :
        if (_value.getObject() instanceof int[]) {
            int[] val = (int[]) _value.getObject();
            for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) elementSet.elementAt(i).getStyle().setPosition(val[i]);
          }
          else {
            int val = _value.getInteger();
            for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setPosition(val);
          }
        break;

      case 20 :
        if (_value.getObject() instanceof double[]) {
            double[] val = (double[]) _value.getObject();
            for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++)
              elementSet.elementAt(i).getStyle().setAngle(val[i]);
          }
        else if (_value.getObject() instanceof double[]) {
          int[] val = (int[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++)
            elementSet.elementAt(i).getStyle().setAngle(val[i]*ControlDrawingPanel3D.TO_RAD);
        }
        else if (_value instanceof IntegerValue) {
          double val = _value.getInteger()*ControlDrawingPanel3D.TO_RAD;
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setAngle(val);
        }
        else {
          double val = _value.getDouble();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setAngle(val);
        }
        break;

      case 21 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).setResolution((Resolution)_value.getObject()); break;

      case PRIMARY_COLOR :
        if (_value.getObject() instanceof Object[]) {
          Object[] val = (Object[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeColor((Color)val[i]);
        }
        else if (_value.getObject() instanceof Color) {
          Color val = (Color) _value.getObject();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeColor(val);
        }
        break;

      case SECONDARY_COLOR :
        if (_value.getObject() instanceof Object[]) {
          Object[] val = (Object[]) _value.getObject();
          for (int i=0, n=Math.min(elementSet.getNumberOfElements(),val.length); i<n; i++) elementSet.elementAt(i).getStyle().setFillPattern((Paint)val[i]);
        }
        else if (_value.getObject() instanceof Color) {
          java.awt.Paint fill = (java.awt.Paint) _value.getObject();
          if (fill==NULL_COLOR) fill = null;
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setFillPattern(fill);
        }
        break;

      case 24 :
        if (_value.getObject() instanceof Stroke) {
          Stroke val = (Stroke) _value.getObject();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeStroke(val);
        }
        else if (lineWidth!=_value.getDouble()) {
          lineWidth = _value.getDouble();
          BasicStroke stroke = new java.awt.BasicStroke((float) lineWidth);
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeStroke(stroke);
        }
        break;

      case 25 : if (_value.getObject() instanceof java.awt.Font) {
          Font newFont = (Font) _value.getObject();
          if (newFont!=font) {
            font = newFont;
            for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setFont(newFont);
          }
        }
        break;

      case 26 : // pressaction
        removeAction (ControlInteractiveElement.ACTION_PRESS,getProperty("pressaction"));
        addAction(ControlInteractiveElement.ACTION_PRESS,_value.getString());
        return;
      case 27 : // dragaction
        removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));
        addAction(ControlElement.VARIABLE_CHANGED,_value.getString());
        return;
      case 28 : // action
        removeAction (ControlElement.ACTION,getProperty("action"));
        addAction(ControlElement.ACTION,_value.getString());
        return;

      default: super.setValue(_index-29,_value); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 :
        elementSet.setNumberOfElements(1);
        allposValues[0].value  = theXs = elementSet.getXs();
        allposValues[1].value  = theYs = elementSet.getYs();
        allposValues[2].value  = theZs = elementSet.getZs();
        allsizesValue[0].value = theSizeXs = elementSet.getSizeXs();
        allsizesValue[1].value = theSizeYs = elementSet.getSizeYs();
        allsizesValue[2].value = theSizeZs = elementSet.getSizeZs();
        break;

      case IMAGE : break; // Subclasses will use this
      case TEXT : break; // Subclasses will use this
      case RADIUS : break; // Subclasses will use this

      case POSITION_X : for (int i=0, n=theXs.length; i<n; i++) elementSet.elementAt(i).setX(theXs[i]=0.0);  break;
      case POSITION_Y : for (int i=0, n=theYs.length; i<n; i++) elementSet.elementAt(i).setY(theYs[i]=0.0);  break;
      case POSITION_Z : for (int i=0, n=theZs.length; i<n; i++) elementSet.elementAt(i).setZ(theZs[i]=0.0);  break;

      case 7 : for (int i=0, n=theSizeXs.length; i<n; i++) elementSet.elementAt(i).setSizeX((theSizeXs[i]=0.1)*scalex); break;
      case 8 : for (int i=0, n=theSizeYs.length; i<n; i++) elementSet.elementAt(i).setSizeY((theSizeYs[i]=0.1)*scaley); break;
      case 9 : for (int i=0, n=theSizeZs.length; i<n; i++) elementSet.elementAt(i).setSizeZ((theSizeZs[i]=0.1)*scalez); break;

      case 10 : elementSet.setVisible(true); break;
        // Some elements may change this. For instance, arrows
      case  ENABLED : elementSet.setEnabled(InteractiveElement.TARGET_POSITION,true); break;
      case  ENABLED_SECONDARY : elementSet.setEnabled(InteractiveElement.TARGET_SIZE,false); break;

      case 13 : scalex = 1.0; for (int i=0, n=theSizeXs.length; i<n; i++) elementSet.elementAt(i).setSizeX(theSizeXs[i]); break;
      case 14 : scaley = 1.0; for (int i=0, n=theSizeYs.length; i<n; i++) elementSet.elementAt(i).setSizeY(theSizeYs[i]); break;
      case 15 : scalez = 1.0; for (int i=0, n=theSizeZs.length; i<n; i++) elementSet.elementAt(i).setSizeZ(theSizeZs[i]); break;

      case 16 : break; // Groups... how to implement this???
      case 17 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).setGroupEnabled(true); break;

      case STYLE : break; // To be implemented by subclasses

      case 19 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setPosition(Style.CENTERED); break;
      case 20 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setAngle(0.0); break;
      case 21 : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).setResolution(null); break;

      case PRIMARY_COLOR : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeColor(Color.black); break;
      case SECONDARY_COLOR : for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setFillPattern(Color.blue); break;

      case 24 :
        {
          BasicStroke stroke = new java.awt.BasicStroke();
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setEdgeStroke(stroke);
        }
        break;

      case 25 :
        {
          font = defaultFont;
          for (int i=0, n=elementSet.getNumberOfElements(); i<n; i++) elementSet.elementAt(i).getStyle().setFont(font);
        }
        break;

      case 26 : removeAction (ControlInteractiveElement.ACTION_PRESS,getProperty("pressaction")); return;
      case 27 : removeAction (ControlElement.VARIABLE_CHANGED,getProperty("dragaction"));         return;
      case 28 : removeAction (ControlElement.ACTION,getProperty("action"));                       return;

      default : super.setDefaultValue (_index-29); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case POSITION_X : return allposValues[0];
      case POSITION_Y : return allposValues[1];
      case POSITION_Z : return allposValues[2];
      case 7 : return allsizesValue[0];
      case 8 : return allsizesValue[1];
      case 9 : return allsizesValue[2];
      default: if (_index<29) return null;
               else return super.getValue(_index-29);
    }
  }

// -------------------------------------
// Respond to interaction
// -------------------------------------

  static private final int[] posSpot = {POSITION_X,POSITION_Y,POSITION_Z};
  static private final int[] sizeSpot = {7,8,9};

  int[] getPosSpot ()  { return posSpot; }
  int[] getSizeSpot ()  { return sizeSpot; }

  public void interactionPerformed(InteractionEvent _event) {
    switch (_event.getID()) {
      case InteractionEvent.MOUSE_PRESSED :
        invokeActions (ControlInteractiveElement.ACTION_PRESS);
        // Do not break!
      case InteractionEvent.MOUSE_DRAGGED :
        InteractionTargetSetElement elementTarget = (InteractionTargetSetElement) _event.getTarget();
        int i = elementTarget.getElementIndex();
        if (elementTarget.getElementTarget().getClass()==InteractionTargetElementSize.class) { // Size
          if (scalex!=0.0) theSizeXs[i] = elementSet.elementAt(i).getSizeX()/scalex; else theSizeXs[i] = elementSet.elementAt(i).getSizeX();
          if (scaley!=0.0) theSizeYs[i] = elementSet.elementAt(i).getSizeY()/scaley; else theSizeYs[i] = elementSet.elementAt(i).getSizeY();
          if (scalez!=0.0) theSizeZs[i] = elementSet.elementAt(i).getSizeZ()/scalez; else theSizeZs[i] = elementSet.elementAt(i).getSizeZ();
          variablesChanged (getSizeSpot(),allsizesValue);
        }
        else { // Position
          theXs[i] = elementSet.elementAt(i).getX();
          theYs[i] = elementSet.elementAt(i).getY();
          theZs[i] = elementSet.elementAt(i).getZ();
          variablesChanged (getPosSpot(),allposValues);
        }
        break;
      case InteractionEvent.MOUSE_RELEASED :
        invokeActions (ControlElement.ACTION);
        break;
    }
  } // End of interaction method

} // End of class