/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) Feb 2004 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.displayejs.InteractiveSurface;
import org.opensourcephysics.ejs.control.swing.NeedsPreUpdate;
import org.opensourcephysics.ejs.control.value.DoubleValue;
import org.opensourcephysics.ejs.control.value.ParserSuryono;
import org.opensourcephysics.ejs.control.value.Value;
import org.opensourcephysics.numerics.SuryonoParser;

/**
 * An analytic surface
 */
public class ControlAnalyticSurface extends ControlInteractiveTile implements NeedsPreUpdate {
  static final int PROPERTIES_SURFACE=PROPERTIES_ADDED+12;
  static final int MY_PRIMARY_COLOR=PRIMARY_COLOR+PROPERTIES_SURFACE;
  static final int MY_SECONDARY_COLOR=SECONDARY_COLOR+PROPERTIES_SURFACE;

  protected InteractiveSurface surface;
  protected String variable1, variable2;
  protected double min1 = -1.0, max1 = 1.0;
  protected double min2 = -1.0, max2 = 1.0;
  protected int points1 = 20, points2 = 20;
  protected String functionX, functionY, functionZ;
  protected double[][][] data=new double[points1][points2][3];
  protected boolean useJavaSyntax=true;

  // Implementation variables
  protected ParserSuryono parserX, parserY, parserZ;
  protected SuryonoParser BparserX, BparserY, BparserZ;
  protected String[] varsX, varsY, varsZ;
  protected int indexX1, indexX2, indexY1, indexY2, indexZ1, indexZ2;

  public ControlAnalyticSurface (Object _visual) { super (_visual); }

  protected Drawable createDrawable (Object _drawable) {
    if (_drawable instanceof InteractiveSurface) surface = (InteractiveSurface) _drawable;
    else surface = new InteractiveSurface();
//    surface.setEnabled(false);
    variable1 = "u"; variable2 = "v";
    functionX = null; varsX = new String[0]; indexX1 = -1; indexX2 = -1;
    functionY = null; varsY = new String[0]; indexY1 = -1; indexY2 = -1;
    functionZ = null; varsZ = new String[0]; indexZ1 = -1; indexZ2 = -1;
    surface.setOrigin(0,0,0,true);
    sizeValues = new DoubleValue[] { new DoubleValue(1), new DoubleValue(1), new DoubleValue(1)};
    return surface;
  }

  private void setfunctionX (String _function) {
    if (useJavaSyntax) {
      varsX = ParserSuryono.getVariableList (functionX = _function);
      parserX = new ParserSuryono(varsX.length);
      for (int i = 0, n = varsX.length; i < n; i++) parserX.defineVariable(i,varsX[i]);
      parserX.define(functionX);
      parserX.parse();
    }
    else {
      try {
        BparserX = new SuryonoParser(0);
        varsX = BparserX.parseUnknown(functionX = _function);
      }
      catch (Exception _exc) {
        varsX = ParserSuryono.getVariableList (functionX = _function);
        parserX = new ParserSuryono(varsX.length);
        for (int i = 0, n = varsX.length; i < n; i++) BparserX.defineVariable(i+1,varsX[i]);
      }
      BparserX.define(functionX);
      BparserX.parse();
    }
    indexX1 = ControlAnalyticCurve.indexOf (variable1,varsX);
    indexX2 = ControlAnalyticCurve.indexOf (variable2,varsX);
  }

  private void setfunctionY (String _function) {
    if (useJavaSyntax) {
      varsY = ParserSuryono.getVariableList (functionY = _function);
      parserY = new ParserSuryono(varsY.length);
      for (int i=0, n=varsY.length; i<n; i++) parserY.defineVariable (i,varsY[i]);
      parserY.define(functionY);
      parserY.parse();
    }
    else {
      try {
        BparserY = new SuryonoParser(0);
        varsY = BparserY.parseUnknown(functionY = _function);
      }
      catch (Exception _exc) {
        varsY = ParserSuryono.getVariableList (functionY = _function);
        parserY = new ParserSuryono(varsY.length);
        for (int i = 0, n = varsY.length; i < n; i++) BparserY.defineVariable(i+1,varsY[i]);
      }
      BparserY.define(functionY);
      BparserY.parse();
    }
    indexY1 = ControlAnalyticCurve.indexOf (variable1,varsY);
    indexY2 = ControlAnalyticCurve.indexOf (variable2,varsY);
  }

  private void setfunctionZ (String _function) {
    if (useJavaSyntax) {
      varsZ = ParserSuryono.getVariableList (functionZ = _function);
      parserZ = new ParserSuryono(varsZ.length);
      for (int i=0, n=varsZ.length; i<n; i++) parserZ.defineVariable (i,varsZ[i]);
      parserZ.define(functionZ);
      parserZ.parse();
    }
    else {
      try {
        BparserZ = new SuryonoParser(0);
        varsZ = BparserZ.parseUnknown(functionZ = _function);
      }
      catch (Exception _exc) {
        varsZ = ParserSuryono.getVariableList (functionZ = _function);
        parserZ = new ParserSuryono(varsZ.length);
        for (int i = 0, n = varsZ.length; i < n; i++) BparserZ.defineVariable(i+1,varsZ[i]);
      }
      BparserZ.define(functionZ);
      BparserZ.parse();
    }
    indexZ1 = ControlAnalyticCurve.indexOf (variable1,varsZ);
    indexZ2 = ControlAnalyticCurve.indexOf (variable2,varsZ);
  }


  private void updateIndexes1 () {
    indexX1 = ControlAnalyticCurve.indexOf (variable1,varsX);
    indexY1 = ControlAnalyticCurve.indexOf (variable1,varsY);
    indexZ1 = ControlAnalyticCurve.indexOf (variable1,varsZ);
  }
  private void updateIndexes2 () {
    indexX2 = ControlAnalyticCurve.indexOf (variable2,varsX);
    indexY2 = ControlAnalyticCurve.indexOf (variable2,varsY);
    indexZ2 = ControlAnalyticCurve.indexOf (variable2,varsZ);
  }

  private void updateFunctions () {
    if (functionX != null) setfunctionX(functionX);
    if (functionY != null) setfunctionY(functionY);
    if (functionZ != null) setfunctionZ(functionZ);
  }

// -------------------------------------
// Update the curve
// -------------------------------------

  public void preupdate() {
    if (useJavaSyntax) preupdateJavaSyntax();
    else preupdateFreeSyntax();
  }

  public synchronized void preupdateJavaSyntax() {
    if (myGroup!=null) { // get the necessary values from the ControlGroup
      for (int i=0, n=varsX.length; i<n; i++) if (i!=indexX1 && i!=indexX2) parserX.setVariable(i,myGroup.getDouble(varsX[i]));
      for (int i=0, n=varsY.length; i<n; i++) if (i!=indexY1 && i!=indexY2) parserY.setVariable(i,myGroup.getDouble(varsY[i]));
      for (int i=0, n=varsZ.length; i<n; i++) if (i!=indexZ1 && i!=indexZ2) parserZ.setVariable(i,myGroup.getDouble(varsZ[i]));
    }
    if (data.length!=points1 || data[0].length!=points2) data =new double[points1][points2][3];
    for (int i=0, n=points1-1; i<=n; i++) {
      double u = ((n-i)*min1 + i*max1)/n;
      if (indexX1>=0) parserX.setVariable(indexX1,u);
      if (indexY1>=0) parserY.setVariable(indexY1,u);
      if (indexZ1>=0) parserZ.setVariable(indexZ1,u);
      for (int j=0, m=points2-1; j<=m; j++) {
        double v = ((m-j)*min2 + j*max2)/m;
        if (functionX==null) data[i][j][0] = 0.0;
        else {
          if (indexX2>=0) parserX.setVariable(indexX2,v);
          data[i][j][0] = parserX.evaluate();
        }
        if (functionY==null) data[i][j][1] = 0.0;
        else {
          if (indexY2>=0) parserY.setVariable(indexY2,v);
          data[i][j][1] = parserY.evaluate();
        }
        if (functionZ==null) data[i][j][2] = 0.0;
        else {
          if (indexZ2>=0) parserZ.setVariable(indexZ2,v);
          data[i][j][2] = parserZ.evaluate();
        }
      }
    }
    surface.setData(data);
  }

  public synchronized void preupdateFreeSyntax() {
    if (myGroup!=null) { // get the necessary values from the ControlGroup
      for (int i=0, n=varsX.length; i<n; i++) if (i!=indexX1 && i!=indexX2) BparserX.setVariable(i+1,myGroup.getDouble(varsX[i]));
      for (int i=0, n=varsY.length; i<n; i++) if (i!=indexY1 && i!=indexY2) BparserY.setVariable(i+1,myGroup.getDouble(varsY[i]));
      for (int i=0, n=varsZ.length; i<n; i++) if (i!=indexZ1 && i!=indexZ2) BparserZ.setVariable(i+1,myGroup.getDouble(varsZ[i]));
    }
    if (data.length!=points1 || data[0].length!=points2) data =new double[points1][points2][3];
    for (int i=0, n=points1-1; i<=n; i++) {
      double u = ((n-i)*min1 + i*max1)/n;
      if (indexX1>=0) BparserX.setVariable(indexX1+1,u);
      if (indexY1>=0) BparserY.setVariable(indexY1+1,u);
      if (indexZ1>=0) BparserZ.setVariable(indexZ1+1,u);
      for (int j=0, m=points2-1; j<=m; j++) {
        double v = ((m-j)*min2 + j*max2)/m;
        if (functionX==null) data[i][j][0] = 0.0;
        else {
          if (indexX2>=0) BparserX.setVariable(indexX2+1,v);
          data[i][j][0] = BparserX.evaluate();
        }
        if (functionY==null) data[i][j][1] = 0.0;
        else {
          if (indexY2>=0) BparserY.setVariable(indexY2+1,v);
          data[i][j][1] = BparserY.evaluate();
        }
        if (functionZ==null) data[i][j][2] = 0.0;
        else {
          if (indexZ2>=0) BparserZ.setVariable(indexZ2+1,v);
          data[i][j][2] = BparserZ.evaluate();
        }
      }
    }
    surface.setData(data);
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable1");
      infoList.add ("min1");
      infoList.add ("max1");
      infoList.add ("points1");
      infoList.add ("variable2");
      infoList.add ("min2");
      infoList.add ("max2");
      infoList.add ("points2");
      infoList.add ("functionx");
      infoList.add ("functiony");
      infoList.add ("functionz");
      infoList.add ("javaSyntax");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable1"))  return "String";
    if (_property.equals("variable2"))  return "String";
    if (_property.equals("min1"))       return "int|double";
    if (_property.equals("max1"))       return "int|double";
    if (_property.equals("min2"))       return "int|double";
    if (_property.equals("max2"))       return "int|double";
    if (_property.equals("points1"))    return "int";
    if (_property.equals("points2"))    return "int";
    if (_property.equals("functionx"))  return "String";
    if (_property.equals("functiony"))  return "String";
    if (_property.equals("functionz"))  return "String";
    if (_property.equals("javaSyntax")) return "boolean";

//    if (_property.equals("enabled"))        return "boolean HIDDEN";
    if (_property.equals("enabledSecondary")) return "boolean HIDDEN";

    if (_property.equals("resolution"))     return "Resolution BASIC HIDDEN";

//    if (_property.equals("action"))      return "Action CONSTANT HIDDEN";
//    if (_property.equals("pressaction")) return "Action CONSTANT HIDDEN";
//    if (_property.equals("dragaction"))  return "Action CONSTANT HIDDEN";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 : variable1 = _value.getString(); updateIndexes1(); break;
      case 1 : min1 = _value.getDouble();   break;
      case 2 : max1 = _value.getDouble();   break;
      case 3 : points1 = _value.getInteger(); break;
      case 4 : variable2 = _value.getString(); updateIndexes2(); break;
      case 5 : min2 = _value.getDouble();   break;
      case 6 : max2 = _value.getDouble();   break;
      case 7 : points2 = _value.getInteger(); break;
      case  8 : setfunctionX(_value.getString()); break;
      case  9 : setfunctionY(_value.getString()); break;
      case 10 : setfunctionZ(_value.getString()); break;
      case 11: useJavaSyntax = _value.getBoolean(); updateFunctions(); break;
      default: super.setValue(_index-12,_value); break;
      case MY_PRIMARY_COLOR   :
      {
      java.awt.Paint fill = (java.awt.Paint) _value.getObject();
      if (fill==NULL_COLOR) fill = null;
      myElement.getStyle().setFillPattern(fill);
      }
      break;
      case MY_SECONDARY_COLOR : myElement.getStyle().setEdgeColor((java.awt.Color) _value.getObject()); break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 : variable1 = "u"; updateIndexes1(); break;
      case 1 : min1 = -1.0;   break;
      case 2 : max1 = 1.0;   break;
      case 3 : points1 = 20; break;
      case 4 : variable2 = "v"; updateIndexes2(); break;
      case 5 : min2 = -1.0;   break;
      case 6 : max2 = 1.0;   break;
      case 7 : points2 = 20; break;
      case  8 : functionX = null; varsX = new String[0]; indexX1 = -1; indexX2 = -1; break;
      case  9 : functionY = null; varsY = new String[0]; indexY1 = -1; indexY2 = -1; break;
      case 10 : functionZ = null; varsZ = new String[0]; indexZ1 = -1; indexZ2 = -1; break;
      case 11 : useJavaSyntax = true; updateFunctions(); break;
      default: super.setDefaultValue(_index-12); break;
      case MY_PRIMARY_COLOR : myElement.getStyle().setFillPattern(java.awt.Color.blue); break;
      case MY_SECONDARY_COLOR : myElement.getStyle().setEdgeColor(java.awt.Color.black); break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case 0 : case 1 : case 2 : case 3 :
      case 4 : case 5 : case 6 : case 7 :
      case 8 : case 9 : case 10 : case 11 :
        return null;
      default: return super.getValue (_index-12);
    }
  }


} // End of class
