/*
 * The control.displayejs package contains subclasses of
 * control.ControlElement that deal with the displayejs package
 * Copyright (c) Feb 2004 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control.displayejs;

import org.opensourcephysics.display.Drawable;
import org.opensourcephysics.displayejs.InteractivePoligon;
import org.opensourcephysics.ejs.control.swing.NeedsPreUpdate;
import org.opensourcephysics.ejs.control.value.ParserSuryono;
import org.opensourcephysics.ejs.control.value.Value;
import org.opensourcephysics.numerics.SuryonoParser;

/**
 * An analytic curve
 */
public class ControlAnalyticCurve extends ControlInteractiveElement implements NeedsPreUpdate {
  static protected final int CURVE_ADDED = 8;
  static protected final int MY_LINE_COLOR = SECONDARY_COLOR+CURVE_ADDED;

  // Configuration variables
  protected InteractivePoligon poligon;
  protected String variable;
  protected double min = -1.0, max = 1.0;
  protected String functionX, functionY, functionZ;
  protected boolean useJavaSyntax=true;

  // Implementation variables
  protected ParserSuryono parserX, parserY, parserZ;
  protected SuryonoParser BparserX, BparserY, BparserZ;
  protected String[] varsX, varsY, varsZ;
  protected int indexX, indexY, indexZ;

  public ControlAnalyticCurve (Object _visual) { super (_visual); }

  protected Drawable createDrawable (Object _drawable) {
    if (_drawable instanceof InteractivePoligon) poligon = (InteractivePoligon) _drawable;
    else poligon = new InteractivePoligon();
    poligon.setNumberOfPoints(200);
//    poligon.setEnabled(false);
    poligon.setClosed(false);
    variable = "t";
    functionX = null; varsX = new String[0]; indexX = -1;
    functionY = null; varsY = new String[0]; indexY = -1;
    functionZ = null; varsZ = new String[0]; indexZ = -1;
    return poligon;
  }

  private void setfunctionX (String _function) {
    // if (_function==null) { functionX = null; varsX = new String[0]; return; }
    // if (_function.equals(functionX)) return;
    if (useJavaSyntax) {
      varsX = ParserSuryono.getVariableList (functionX = _function);
      parserX = new ParserSuryono(varsX.length);
      for (int i = 0, n = varsX.length; i < n; i++) parserX.defineVariable(i,varsX[i]);
      parserX.define(functionX);
      parserX.parse();
    }
    else {
      try {
        BparserX = new SuryonoParser(0);
        varsX = BparserX.parseUnknown(functionX = _function);
      }
      catch (Exception _exc) {
        varsX = ParserSuryono.getVariableList (functionX = _function);
        parserX = new ParserSuryono(varsX.length);
        for (int i = 0, n = varsX.length; i < n; i++) BparserX.defineVariable(i+1,varsX[i]);
      }
      BparserX.define(functionX);
      BparserX.parse();
    }
    indexX = indexOf (variable,varsX);
  }

  private void setfunctionY (String _function) {
    if (useJavaSyntax) {
      varsY = ParserSuryono.getVariableList (functionY = _function);
      parserY = new ParserSuryono(varsY.length);
      for (int i=0, n=varsY.length; i<n; i++) parserY.defineVariable (i,varsY[i]);
      parserY.define(functionY);
      parserY.parse();
    }
    else {
      try {
        BparserY = new SuryonoParser(0);
        varsY = BparserY.parseUnknown(functionY = _function);
      }
      catch (Exception _exc) {
        varsY = ParserSuryono.getVariableList (functionY = _function);
        parserY = new ParserSuryono(varsY.length);
        for (int i = 0, n = varsY.length; i < n; i++) BparserY.defineVariable(i+1,varsY[i]);
      }
      BparserY.define(functionY);
      BparserY.parse();
    }
    indexY = indexOf (variable,varsY);
  }

  private void setfunctionZ (String _function) {
    if (useJavaSyntax) {
      varsZ = ParserSuryono.getVariableList (functionZ = _function);
      parserZ = new ParserSuryono(varsZ.length);
      for (int i=0, n=varsZ.length; i<n; i++) parserZ.defineVariable (i,varsZ[i]);
      parserZ.define(functionZ);
      parserZ.parse();
    }
    else {
      try {
        BparserZ = new SuryonoParser(0);
        varsZ = BparserZ.parseUnknown(functionZ = _function);
      }
      catch (Exception _exc) {
        varsZ = ParserSuryono.getVariableList (functionZ = _function);
        parserZ = new ParserSuryono(varsZ.length);
        for (int i = 0, n = varsZ.length; i < n; i++) BparserZ.defineVariable(i+1,varsZ[i]);
      }
      BparserZ.define(functionZ);
      BparserZ.parse();
    }
    indexZ = indexOf (variable,varsZ);
  }

  private void updateIndexes () {
    indexX = indexOf (variable,varsX);
    indexY = indexOf (variable,varsY);
    indexZ = indexOf (variable,varsZ);
  }

  private void updateFunctions () {
    if (functionX != null) setfunctionX(functionX);
    if (functionY != null) setfunctionY(functionY);
    if (functionZ != null) setfunctionZ(functionZ);
  }

  static int indexOf (String _var, String[] _vars) {
    for (int i=0, n=_vars.length; i<n; i++) if (_var.equals(_vars[i])) return i;
    return -1;
  }

// -------------------------------------
// Update the curve
// -------------------------------------

  public void preupdate() {
    if (useJavaSyntax) preupdateJavaSyntax();
    else preupdateFreeSyntax();
  }


  public synchronized void preupdateJavaSyntax() {
    if (myGroup!=null) { // get the necessary values from the ControlGroup
     for (int i = 0, n = varsX.length; i < n; i++)if (i != indexX) parserX.setVariable(i, myGroup.getDouble(varsX[i]));
     for (int i = 0, n = varsY.length; i < n; i++)if (i != indexY) parserY.setVariable(i, myGroup.getDouble(varsY[i]));
     for (int i = 0, n = varsZ.length; i < n; i++)if (i != indexZ) parserZ.setVariable(i, myGroup.getDouble(varsZ[i]));
    }
    double[][] data = poligon.getData();
    for (int i = 0, n = poligon.getNumberOfPoints() - 1; i <= n; i++) {
      double t = ( (n - i) * min + i * max) / n;
      if (functionX == null) data[0][i] = 0.0;
      else {
        if (indexX >= 0) parserX.setVariable(indexX, t);
        data[0][i] = parserX.evaluate();
      }
      if (functionY == null) data[1][i] = 0.0;
      else {
        if (indexY >= 0) parserY.setVariable(indexY, t);
        data[1][i] = parserY.evaluate();
      }
      if (functionZ == null) data[2][i] = 0.0;
      else {
        if (indexZ >= 0) parserZ.setVariable(indexZ, t);
        data[2][i] = parserZ.evaluate();
      }
    }
    poligon.needsToProject(null); // force poligon to update
  }

  public synchronized void preupdateFreeSyntax() {
    if (myGroup!=null) { // get the necessary values from the ControlGroup
      for (int i = 0, n = varsX.length; i < n; i++)if (i != indexX) BparserX.setVariable(i+1, myGroup.getDouble(varsX[i]));
      for (int i = 0, n = varsY.length; i < n; i++)if (i != indexY) BparserY.setVariable(i+1, myGroup.getDouble(varsY[i]));
      for (int i = 0, n = varsZ.length; i < n; i++)if (i != indexZ) BparserZ.setVariable(i+1, myGroup.getDouble(varsZ[i]));
    }
    double[][] data = poligon.getData();
    for (int i = 0, n = poligon.getNumberOfPoints() - 1; i <= n; i++) {
      double t = ( (n - i) * min + i * max) / n;
      if (functionX == null) data[0][i] = 0.0;
      else {
        if (indexX >= 0) BparserX.setVariable(indexX+1, t);
        data[0][i] = BparserX.evaluate();
      }
      if (functionY == null) data[1][i] = 0.0;
      else {
        if (indexY >= 0) BparserY.setVariable(indexY+1, t);
        data[1][i] = BparserY.evaluate();
      }
      if (functionZ == null) data[2][i] = 0.0;
      else {
        if (indexZ >= 0) BparserZ.setVariable(indexZ+1, t);
        data[2][i] = BparserZ.evaluate();
      }
    }
    poligon.needsToProject(null); // force poligon to update
  }

// ------------------------------------------------
// Definition of Properties
// ------------------------------------------------

  static private java.util.ArrayList infoList=null;

  public java.util.ArrayList getPropertyList() {
    if (infoList==null) {
      infoList = new java.util.ArrayList ();
      infoList.add ("variable");
      infoList.add ("min");
      infoList.add ("max");
      infoList.add ("points");
      infoList.add ("functionx");
      infoList.add ("functiony");
      infoList.add ("functionz");
      infoList.add ("javaSyntax");
      infoList.addAll(super.getPropertyList());
    }
    return infoList;
  }

  public String getPropertyInfo(String _property) {
    if (_property.equals("variable"))  return "String";
    if (_property.equals("min"))       return "int|double";
    if (_property.equals("max"))       return "int|double";
    if (_property.equals("points"))    return "int";
    if (_property.equals("functionx")) return "String";
    if (_property.equals("functiony")) return "String";
    if (_property.equals("functionz")) return "String";
    if (_property.equals("javaSyntax"))return "boolean";

    if (_property.equals("scalex"))         return "int|double BASIC HIDDEN";
    if (_property.equals("scaley"))         return "int|double BASIC HIDDEN";
    if (_property.equals("scalez"))         return "int|double BASIC HIDDEN";

//    if (_property.equals("enabled"))        return "boolean HIDDEN";
    if (_property.equals("enabledSecondary")) return "boolean HIDDEN";

    if (_property.equals("style"))          return "MarkerShape|int BASIC HIDDEN";
    if (_property.equals("elementposition"))return "ElementPosition|int BASIC HIDDEN";
    if (_property.equals("angle"))          return "int|double BASIC HIDDEN";
    if (_property.equals("resolution"))     return "Resolution BASIC HIDDEN";

    if (_property.equals("secondaryColor")) return "Color|Object BASIC HIDDEN";
    if (_property.equals("font"))           return "Font|Object  BASIC HIDDEN";

//    if (_property.equals("action"))      return "Action CONSTANT HIDDEN";
//    if (_property.equals("pressaction")) return "Action CONSTANT HIDDEN";
//    if (_property.equals("dragaction"))  return "Action CONSTANT HIDDEN";

    return super.getPropertyInfo(_property);
  }

// ------------------------------------------------
// Set and Get the values of the properties
// ------------------------------------------------

  public void setValue (int _index, Value _value) {
    switch (_index) {
      case 0 : variable = _value.getString(); updateIndexes(); break;
      case 1 : min = _value.getDouble();   break;
      case 2 : max = _value.getDouble();   break;
      case 3 : poligon.setNumberOfPoints (_value.getInteger()); break;
      case 4 : setfunctionX(_value.getString()); break;
      case 5 : setfunctionY(_value.getString()); break;
      case 6 : setfunctionZ(_value.getString()); break;
      case 7 : useJavaSyntax = _value.getBoolean(); updateFunctions(); break;
      default: super.setValue(_index-CURVE_ADDED,_value); break;
      case MY_LINE_COLOR : super.setValue(PRIMARY_COLOR,_value) ; break;
    }
  }

  public void setDefaultValue (int _index) {
    switch (_index) {
      case 0 : variable = "t"; updateIndexes(); break;
      case 1 : min = -1.0;   break;
      case 2 : max = 1.0;   break;
      case 3 : poligon.setNumberOfPoints (200); break;
      case 4 : functionX = null; varsX = new String[0];  break;
      case 5 : functionY = null; varsY = new String[0];  break;
      case 6 : functionZ = null; varsZ = new String[0];  break;
      case 7 : useJavaSyntax = true; updateFunctions();   break;
      default: super.setDefaultValue(_index-CURVE_ADDED); break;
      case MY_LINE_COLOR : super.setDefaultValue(PRIMARY_COLOR) ; break;
    }
  }

  public Value getValue (int _index) {
    switch (_index) {
      case 0 : case 1 : case 2 : case 3 :
      case 4 : case 5 : case 6 : case 7 :
        return null;
      default: return super.getValue (_index-CURVE_ADDED);
    }
  }


} // End of class
