/*
 * The control package contains utilities to build and control
 * simulations using a central control.
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.ejs.control;

import java.awt.Color;
import java.awt.Font;
import java.awt.Rectangle;
import java.util.StringTokenizer;

import org.opensourcephysics.ejs.control.value.BooleanValue;
import org.opensourcephysics.ejs.control.value.ObjectValue;
import org.opensourcephysics.ejs.control.value.Value;

/**
 * This class provides static methods that parse a string and return
 * a Value with the corresponding type and value, ready to be used by
 * the setValue() method of ControlElements
 */
public class ConstantParser {

// -------------- public variables and methods -----------

  public static final java.awt.Color NULL_COLOR = new java.awt.Color(0,0,0,0);
  static private Font defaultFont = new Font ("Dialog",12,Font.PLAIN);

  static public Value fontConstant (Font _currentFont, String _value) {
    if (_value.indexOf(',')<0) return null; // No commas, not a valid constant
    if (_currentFont==null) _currentFont = defaultFont;
    int style = _currentFont.getStyle();
    int size  = _currentFont.getSize();
    String name=null;
    StringTokenizer t = new StringTokenizer(_value,",",true);
    if (t.hasMoreTokens()) {
      name = t.nextToken();
      if (name.equals(",")) name = _currentFont.getName();
      else if (t.hasMoreTokens()) t.nextToken(); // read out next ','
    }
    else name = _currentFont.getName();
    if (t.hasMoreTokens()) {
      String styleStr = t.nextToken().toLowerCase();
      style = _currentFont.getStyle();
      if (!styleStr.equals(",")) {
        if (styleStr.indexOf("plain")!=-1)  style = java.awt.Font.PLAIN;
        if (styleStr.indexOf("bold")!=-1)   style = java.awt.Font.BOLD;
        if (styleStr.indexOf("italic")!=-1) style = style | java.awt.Font.ITALIC;
        if (t.hasMoreTokens()) t.nextToken(); // read out next ','
      }
    }
    if (t.hasMoreTokens()) try { size = Integer.parseInt(t.nextToken());}
                           catch (Exception exc) { size = _currentFont.getSize(); }
    return new ObjectValue(new Font (name,style,size));
  }

  static public Value booleanConstant (String _value) {
    if (_value.equals("true"))  return new BooleanValue(true);
    if (_value.equals("false")) return new BooleanValue(false);
    return null; // Not a valid constant
  }

  static public Value colorConstant (String _value) {
    if (_value.indexOf(',')>=0) { // format is red,green,blue
      try {
        StringTokenizer t = new StringTokenizer(_value,":,");
        int r = Integer.parseInt(t.nextToken());
        int g = Integer.parseInt(t.nextToken());
        int b = Integer.parseInt(t.nextToken());
        int alpha;
        if (t.hasMoreTokens()) alpha = Integer.parseInt(t.nextToken());
        else alpha = 255;
        if (r<0) r = 0; else if (r>255) r = 255;
        if (g<0) g = 0; else if (g>255) g = 255;
        if (b<0) b = 0; else if (b>255) b = 255;
        if (alpha<0) alpha = 0; else if (alpha>255) alpha = 255;
        return new ObjectValue(new Color (r,g,b,alpha));
      } catch (Exception exc) { exc.printStackTrace(); return null; }
    }
    if (_value.equals("null")      || _value.equals("none"))            return new ObjectValue(NULL_COLOR);
    if (_value.equals("black")     || _value.equals("Color.black"))     return new ObjectValue(Color.black);
    if (_value.equals("blue")      || _value.equals("Color.blue"))      return new ObjectValue(Color.blue);
    if (_value.equals("cyan")      || _value.equals("Color.cyan"))      return new ObjectValue(Color.cyan);
    if (_value.equals("darkGray")  || _value.equals("Color.darkGray"))  return new ObjectValue(Color.darkGray);
    if (_value.equals("gray")      || _value.equals("Color.gray"))      return new ObjectValue(Color.gray);
    if (_value.equals("green")     || _value.equals("Color.green"))     return new ObjectValue(Color.green);
    if (_value.equals("lightGray") || _value.equals("Color.lightGray")) return new ObjectValue(Color.lightGray);
    if (_value.equals("magenta")   || _value.equals("Color.magenta"))   return new ObjectValue(Color.magenta);
    if (_value.equals("orange")    || _value.equals("Color.orange"))    return new ObjectValue(Color.orange);
    if (_value.equals("pink")      || _value.equals("Color.pink"))      return new ObjectValue(Color.pink);
    if (_value.equals("red")       || _value.equals("Color.red"))       return new ObjectValue(Color.red);
    if (_value.equals("white")     || _value.equals("Color.white"))     return new ObjectValue(Color.white);
    if (_value.equals("yellow")    || _value.equals("Color.yellow"))    return new ObjectValue(Color.yellow);
    return null; // Not a valid constant
  }

  static public Value formatConstant (String _value) {
    if(_value.indexOf(";")==-1){//FKH 021103
      int id1=_value.indexOf("0"),id2=_value.indexOf("#"),id=-1;
      if(id1>0&&id2>0)id=id1<id2? id1:id2;
      else if(id1>0)id=id1;
      else if(id2>0)id=id2;
      if(id>0)_value=_value+";"+_value.substring(0,id)+"-"+_value.substring(id);
    }//endFKH 021103
    try { return new ObjectValue (new java.text.DecimalFormat(_value)); }
    catch (IllegalArgumentException _exc) { return null; }
  }

  static public Value rectangleConstant (String _value) {
    if (_value.indexOf(',')<0) return null; // No commas, not a valid constant
    try { // x,y,w,h
      StringTokenizer t = new StringTokenizer(_value,",");
      int x = Integer.parseInt(t.nextToken());
      int y = Integer.parseInt(t.nextToken());
      int w = Integer.parseInt(t.nextToken());
      int h = Integer.parseInt(t.nextToken());
      return new ObjectValue(new Rectangle(x,y,w,h));
    } catch (Exception exc) { exc.printStackTrace(); return null; }
  }

} // end of class