/*
 * The display package contains drawing classes and drawables
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.displayejs;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;

import org.opensourcephysics.display.DrawingPanel;

public class LightBulb implements Drawable3D, Measurable3D {
  static private int LEVELS = 256;

  // Configuration variables
  private boolean visible = true;
  private int intensity=0; // From 0 to LEVELS
  private Color lightColor = Color.yellow, lineColor = Color.black;
  private double radius,height;

  // Implementation variables
  private Color[] lights = new Color[LEVELS];
  private double pixel[] = new double[5];
  private double coordinates[] = new double[6];
  private Object3D[] objects    = new Object3D[] { new Object3D(this,0) };

  public LightBulb () {
    for (int i=0; i<6; i++) coordinates[i] = 0;
    objects[0].distance = 0.0;
    setRadius(0.1);
    setColor (Color.yellow);
  }

  public void setVisible (boolean _draw) {  visible = _draw; }
  public boolean isVisible () {  return visible; }

  public void setColor (Color _aColor) {
    if (_aColor==null) lightColor = Color.yellow;
    else lightColor = _aColor;
    int r = lightColor.getRed();
    int g = lightColor.getGreen();
    int b = lightColor.getBlue();
    for (int i=0; i<LEVELS; i++) {
      lights[i] = new Color (r,g,b,i); // 0 is transparent, 255 is opaque
    }
  }
  public Color getColor () { return lightColor; }

  public void setLineColor (Color _aColor) {
    if (_aColor==null) lineColor = Color.black;
    else lineColor = _aColor;
  }
  public Color getLineColor () { return lineColor; }

  public void setX (double _x) { coordinates[0] = _x; }
  public double getX () {  return coordinates[0]; }

  public void setY (double _y) { coordinates[1] = _y; }
  public double getY () {  return coordinates[1]; }

  public void setZ (double _z) { coordinates[2] = _z; }
  public double getZ () {  return coordinates[2]; }

  public void setRadius (double _r) {
    radius = Math.abs(_r);
    height = 3.0*radius;
    coordinates[3] = coordinates[4] = coordinates[5] = radius;
  }
  public double getRadius () { return radius; }

  public void setIntensity (int _int) {
    if (_int<0) intensity = 0;
    else if (_int>=LEVELS) intensity = LEVELS-1;
    else intensity = _int;
  }
  public int getIntensity () {  return intensity; }

  public Object3D[] getObjects3D(DrawingPanel3D panel) {
    if (!visible) return null;
    pixel = panel.project(coordinates,pixel);
    objects[0].distance = pixel[4];
    return objects;
  }

  public void draw (DrawingPanel3D _panel, Graphics2D _g, int _index) { draw (_panel,_g); }

  public void drawQuickly (DrawingPanel3D _panel, Graphics2D _g2) { draw (_panel, _g2); }

  public void draw (DrawingPanel panel, Graphics g) {
    if (!visible) return;
    Color theColor = lights[intensity];
    if (panel instanceof DrawingPanel3D) {
      // projection has already been computed in getObjects3D
      if      (pixel[4]<0.9) theColor = theColor.brighter();
      else if (pixel[4]>1.1) theColor = theColor.darker();
    }
    else pixel = panel.project(coordinates,pixel);
    int xpix = (int) pixel[0], ypix = (int) pixel[1],
        lenx = (int) pixel[2], leny = (int) pixel[3];
    g.setColor (theColor);
    // Next computations could be optimised a little bit
    g.fillOval (xpix-lenx, ypix-3*leny, 2*lenx, 2*leny);  // draw the bulb
    g.setColor (lineColor);
    g.drawRect (xpix-lenx/2,ypix-leny,   lenx,       leny);         // draw the base
    g.drawOval (xpix-lenx,  ypix-3*leny, 2*lenx,     2*leny);       // draw the contour
    g.drawLine (xpix-lenx/2,ypix-2*leny, xpix+lenx/2,ypix-2*leny);  // draw the filament
  }

  public void needsToProject(DrawingPanel _panel)   {
  }

  // Implementation of Measured3D
  public boolean isMeasured () { return true;  }
  public double getXMin () { return coordinates[0]-radius; }
  public double getXMax () { return coordinates[0]+radius; }
  public double getYMin () { return coordinates[1]+height; }
  public double getYMax () { return coordinates[1];}
  public double getZMin () { return coordinates[2]+height; }
  public double getZMax () { return coordinates[2];}

}
