/*
 * The display package contains drawing classes and drawables
 * Copyright (c) June 2002 F. Esquembre
 * @author F. Esquembre (http://fem.um.es).
 */

package org.opensourcephysics.displayejs;


import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RectangularShape;
import java.awt.geom.RoundRectangle2D;
import java.util.ArrayList;
import java.util.Iterator;

import org.opensourcephysics.display.DrawingPanel;

public class InteractiveTrace extends AbstractInteractiveElement {
  // Configuration variables
  protected boolean connected=true, ignore=false, active=true;
  protected int maxPoints=0, skip=0, shapeSize;
  private int shapeType = -1;  // Make sure an shape is created (if requested to)

  // Implementation variables
  private int counter=0;
//  private double xlast=Double.NaN, ylast=Double.NaN, zlast=Double.NaN;
  private double point[] = new double[3];
  protected ArrayList list, displayList;
  private OnePoint lastPoint=new OnePoint(this,Double.NaN,Double.NaN,Double.NaN,false);
  private Object3D[] minimalObjects = new Object3D[1];
  DrawingPanel panelWithValidProjection = null;
  private AffineTransform transform = new AffineTransform();

  /**
   * Default constructor
   */
  public InteractiveTrace () {
    list = new ArrayList();
    setXYZ(0,0,0);
    setSizeXYZ(1,1,1);
//    style.displayObject = new Ellipse2D.Float();
  }

  public void copyFrom (InteractiveElement _element) {
    super.copyFrom(_element);
    if (_element instanceof InteractiveTrace) {
      setMaximumPoints(((InteractiveTrace)_element).getMaximumPoints());
      setConnected(((InteractiveTrace)_element).isConnected());
      setIgnoreEqualPoints(((InteractiveTrace)_element).isIgnoreEqualPoints());
      setActive(((InteractiveTrace)_element).isActive());
      setSkip(((InteractiveTrace)_element).getSkip());
    }
  }

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void setMaximumPoints (int _n) {
    if (_n<0 || _n==maxPoints) return;
    maxPoints = _n;
    counter   = 0;
    synchronized(list) { list.clear(); }
    hasChanged = true;
  }
  public int getMaximumPoints () {  return maxPoints; }

  public void setConnected (boolean connect) { this.connected = connect; }
  public boolean isConnected () { return this.connected; }

  public void setIgnoreEqualPoints (boolean ignoreEqual) { this.ignore = ignoreEqual; }
  public boolean isIgnoreEqualPoints () { return this.ignore; }

  public void setActive (boolean acceptInput) {  this.active = acceptInput; }
  public boolean isActive () {  return this.active; }

  public void setSkip (int howMany) { this.skip = howMany; counter = 0; }
  public int  getSkip () { return this.skip; }

  // Methods for operation

  public void clear () { synchronized(list) { list.clear(); } }

  public void addPoint (double xInput, double yInput) { addPoint (xInput,yInput,0.0); }

  public void addPoint (double xInput, double yInput, double zInput) {
    if (!active) return;
//    System.out.println ("Adding "+xInput+","+yInput+" connected = "+connected);
    if (skip>0) {
      if (counter>0) {
        counter++;
        if (counter>=skip) counter = 0;
        return;
      }
      counter++;
    }
    if (ignore && xInput==lastPoint.coordinates[0] && yInput==lastPoint.coordinates[1] && zInput==lastPoint.coordinates[2]) return;
    synchronized(list) {
      if (maxPoints>0 && (list.size()>=maxPoints) ) list.remove(0);
      list.add (lastPoint = new OnePoint (this,xInput,yInput,zInput,connected));
    }
    hasChanged = true;
  }

  /**
   * Set the type of the marker
   */
  public void setShapeType (int _type) {
    if (shapeType==_type) return;
    shapeType = _type;
    switch (shapeType) {
      default :
      case InteractiveParticle.NONE            : style.displayObject = null; break;
      case InteractiveParticle.ELLIPSE         : style.displayObject = new Ellipse2D.Float(); break;
      case InteractiveParticle.RECTANGLE       : style.displayObject = new Rectangle2D.Float(); break;
      case InteractiveParticle.ROUND_RECTANGLE : style.displayObject = new RoundRectangle2D.Float(); break;
    }
  }

  /**
   * Set the size of the marker
   */
  public void setShapeSize (int _size) { shapeSize = _size; }

// -------------------------------------
// Implementation of Drawable3D
// -------------------------------------

  public org.opensourcephysics.display.Interactive findInteractive (DrawingPanel _panel, int _xpix, int _ypix) {
    if (!visible) return null;
    if (hasChanged || _panel!=panelWithValidProjection) projectPoints (_panel);
//    if (sizeEnabled      && Math.abs(a2-_xpix)<SENSIBILITY               && Math.abs(b2-_ypix)<SENSIBILITY)               return sizeTarget;
    if (positionEnabled  && Math.abs(lastPoint.pixel[0]-_xpix)<SENSIBILITY && Math.abs(lastPoint.pixel[1]-_ypix)<SENSIBILITY) {
      return new InteractionTargetTracePoint(this, new Point3D (lastPoint.coordinates[0],lastPoint.coordinates[1],lastPoint.coordinates[2]));
    }
    return null;
  }

  public Object3D[] getObjects3D (DrawingPanel3D _panel) {
    if (list.size()<=0 || !visible) return null;
    if (hasChanged || _panel!=panelWithValidProjection) projectPoints(_panel);
    return (Object3D[]) displayList.toArray(minimalObjects);
  }

  public void draw (DrawingPanel3D _panel, Graphics2D _g, int _index) {
    try {
      Graphics2D g2 = (Graphics2D) _g;
      OnePoint point = (OnePoint) displayList.get(_index);
      Color theColor = ( (DrawingPanel3D) _panel).projectColor(style.edgeColor, point.distance);
      if (_index>0 && point.connected) {
        g2.setColor(theColor);
        g2.setStroke(style.edgeStroke);
        OnePoint pointPrev = (OnePoint) displayList.get(_index-1);
        g2.drawLine((int)point.pixel[0],(int)point.pixel[1],(int)pointPrev.pixel[0],(int)pointPrev.pixel[1]);
      }
      if (style.displayObject!=null) {
        Paint theFillPattern = style.fillPattern;
        if (theFillPattern instanceof Color) theFillPattern = ( (DrawingPanel3D) _panel).projectColor( (Color) theFillPattern, point.distance);
        drawMarker (g2,point,theColor,theFillPattern);
      }
    } catch (Exception _e) { } //System.out.println ("Aha at drawing number "+_index +" / "+list.size()); } // Ignore it
  }

  public void draw (DrawingPanel _panel, Graphics _g) {
    if (list.size()<=0 || !visible) return;
//    if (hasChanged || _panel!=panelWithValidProjection)
    projectPoints(_panel);
    boolean notFirstTime=false;
    int aprev=0, bprev=0;
    Graphics2D g2 = (Graphics2D) _g;
    for (Iterator it=displayList.iterator(); it.hasNext(); ) { // Draw everything
      OnePoint point = (OnePoint) it.next();
      if (style.edgeColor==null) continue;
      g2.setColor (style.edgeColor);
      g2.setStroke(style.edgeStroke);
      if (notFirstTime && point.connected) g2.drawLine ((int)point.pixel[0],(int)point.pixel[1],aprev,bprev);
      if (style.displayObject!=null) drawMarker (g2,point,style.edgeColor,style.fillPattern);
      aprev = (int)point.pixel[0];
      bprev = (int)point.pixel[1];
      notFirstTime = true;
    }
  }

// -------------------------------------
// Implementation of Measured3D
// -------------------------------------

  public boolean isMeasured () { return canBeMeasured && list.size()>0;  }

  public double getXMin () {
    double min = Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) min = Math.min(min,((OnePoint) it.next()).coordinates[0]);
    }
    if (group==null) return x + min*sizex;
    else return group.x + (x + min*sizex)*group.sizex;
  }
  public double getXMax () {
    double max = -Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) max = Math.max(max,((OnePoint) it.next()).coordinates[0]);
    }
    if (group==null) return x + max*sizex;
    else return group.x + (x + max*sizex)*group.sizex;
  }
  public double getYMin () {
    double min = Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) min = Math.min(min,((OnePoint) it.next()).coordinates[1]);
    }
    if (group==null) return y + min*sizey;
    else return group.y + (y + min*sizey)*group.sizey;
  }
  public double getYMax () {
    double max = -Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) max = Math.max(max,((OnePoint) it.next()).coordinates[1]);
    }
    if (group==null) return y + max*sizey;
    else return group.y + (y + max*sizey)*group.sizey;
  }
  public double getZMin () {
    double min = Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) min = Math.min(min,((OnePoint) it.next()).coordinates[2]);
    }
    if (group==null) return z + min*sizez;
    else return group.z + (z + min*sizez)*group.sizez;
  }
  public double getZMax () {
    double max = -Double.MAX_VALUE;
    synchronized(list) {
      for (Iterator it = list.iterator(); it.hasNext(); ) max = Math.max(max,((OnePoint) it.next()).coordinates[2]);
    }
    if (group==null) return z + max*sizez;
    else return group.z + (z + max*sizez)*group.sizez;
  }

// -------------------------------------
//  Private methods and classes
// -------------------------------------

  private void projectPoints (DrawingPanel _panel) {
    synchronized(list)
    {
      displayList = (ArrayList) list.clone(); // Or face synchronization problems!
    }
    for (int i=0, n=displayList.size(); i<n; i++) ((OnePoint) displayList.get(i)).project(_panel,i);
    panelWithValidProjection = _panel;
  }

  private void drawMarker (Graphics2D _g2, OnePoint _point, Color _color, Paint _fill) {
    if (! (style.displayObject instanceof RectangularShape) ) {
      _g2.setColor (_color);
      _g2.drawOval ((int) _point.pixel[0], (int) _point.pixel[1],1,1);  // draw a point
      return;
    }
    RectangularShape shape = (RectangularShape) style.displayObject;
    AffineTransform originalTransform = _g2.getTransform();
    transform.setTransform(originalTransform);
    transform.rotate(-style.angle,_point.pixel[0],_point.pixel[1]);
    _g2.setTransform(transform);
    shape.setFrame(_point.a1,_point.b1,shapeSize, shapeSize); // shape.getWidth(),shape.getHeight());
    if (_fill!=null) { // First fill the inside
      _g2.setPaint(_fill);
      _g2.fill(shape);
    }
    _g2.setColor (_color);
    _g2.setStroke(style.edgeStroke);
    _g2.draw(shape); // Second, draw the edge
    _g2.setTransform(originalTransform);
  }


private class OnePoint extends Object3D {
  boolean connected;
  double[] coordinates = new double[3];
  double[] pixel = new double[3];
  int a1,b1;

  OnePoint (Drawable3D _drawable, double _x,double _y,double _z, boolean _c) {
    super (_drawable,-1);
    coordinates[0] = _x; coordinates[1] = _y;  coordinates[2] = _z;
    connected = _c;
    }

  protected void project (DrawingPanel _panel, int _index) {
    if (group==null) {
      point[0] = x + coordinates[0]*sizex;
      point[1] = y + coordinates[1]*sizey;
      point[2] = z + coordinates[2]*sizez;
    }
    else {
      point[0] = group.x + (x + coordinates[0]*sizex)*group.sizex;
      point[1] = group.y + (y + coordinates[1]*sizey)*group.sizey;
      point[2] = group.z + (z + coordinates[2]*sizez)*group.sizez;
    }
    _panel.project(point,pixel);
    distance = pixel[2];
    index = _index;
    if (style.displayObject instanceof RectangularShape) {
      RectangularShape shape = (RectangularShape) style.displayObject;
      double dx, dy;
      switch (style.position) {
        default :
        case Style.CENTERED:   dx = shape.getWidth()/2.0; dy = shape.getHeight()/2.0; break;
        case Style.NORTH:      dx = shape.getWidth()/2.0; dy = 0.0;                   break;
        case Style.SOUTH:      dx = shape.getWidth()/2.0; dy = shape.getHeight();     break;
        case Style.EAST:       dx = shape.getWidth();     dy = shape.getHeight()/2.0; break;
        case Style.SOUTH_EAST: dx = shape.getWidth();     dy = shape.getHeight();     break;
        case Style.NORTH_EAST: dx = shape.getWidth();     dy = 0.0;                   break;
        case Style.WEST:       dx = 0.0;                  dy = shape.getHeight()/2.0; break;
        case Style.SOUTH_WEST: dx = 0.0;                  dy = shape.getHeight();     break;
        case Style.NORTH_WEST: dx = 0.0;                  dy = 0.0;                   break;
      }
      a1 = (int) (pixel[0] - dx);
      b1 = (int) (pixel[1] - dy);
    }
  }

}  // End of class OnePoint

}  // End of main class
