/*
 * The display3D package contains 3D drawing classes and drawables
 * @author F. Esquembre (http://fem.um.es).
 * Last version : July 2003
 */


package org.opensourcephysics.displayejs;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;

import org.opensourcephysics.display.DrawingPanel;

public class InteractivePoligon extends AbstractInteractiveElement {
  // Configuration variables
  protected boolean closed = true;
  protected int numPoints=-1;
  protected double coordinates[][] = null;
  protected boolean connect[] = null, pointSizeEnabled[] = null;

  // Implementation variables
  protected int sides = 0;
  protected int aPoints[]=null, bPoints[]=null;
  protected double center[] = new double[3]; // The center of the poligon
  protected double pixel[]  = new double[3], pixelOrigin[] = new double[3]; // Output of panel's projections
  protected double point[]  = new double[3], origin[] = new double[3], size[] = new double[3]; // Auxiliary arrays

  protected Object3D[] lineObjects=null;  // Objects3D for each of the lines
  protected Object3D[] closedObject= new Object3D[] { new Object3D(this,-1) }; // A special object for a closed poligon
  protected InteractionTargetPoligonPoint targetPoint = new InteractionTargetPoligonPoint(this,-1);

  public InteractivePoligon () {
    setXYZ(0.0,0.0,0.0);
    setSizeXYZ(1.0,1.0,1.0);
  }

  public void copyFrom (InteractiveElement _element) {
    super.copyFrom(_element);
    if (_element instanceof InteractivePoligon) {
      InteractivePoligon old = (InteractivePoligon) _element;
      setNumberOfPoints(old.getNumberOfPoints());
      setClosed(old.isClosed());
      setData(old.getData());
      for (int i=0; i<numPoints; i++) connect[i] = old.connect[i];
      for (int i=0; i<numPoints; i++) pointSizeEnabled[i] = old.pointSizeEnabled[i];
    }
  }

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void setNumberOfPoints (int _n) {
    if (_n==numPoints) return;
    if (_n<1) return;
    numPoints = _n;
    sides = numPoints-1;
    coordinates  = new double [3][numPoints];
    connect = new boolean [numPoints];
    pointSizeEnabled = new boolean [numPoints];
    aPoints = new int [numPoints];
    bPoints = new int [numPoints];
    lineObjects = new Object3D[numPoints];
    for (int i=0; i<numPoints; i++) {
      coordinates[0][i] = coordinates[1][i] = coordinates[2][i] = 0.0;
      connect[i] = true;
      pointSizeEnabled[i] = true;
      // each of the lines, including the closing line
      lineObjects[i] = new Object3D(this,i);
    }
    connect[sides] = closed;
    hasChanged = true;
  }
  public int getNumberOfPoints () {  return numPoints; }

  public void setClosed (boolean _closed) {
    closed = _closed;
    if (sides>0) connect[sides] = closed;
    hasChanged = true;
  }
  public boolean isClosed () {  return closed; }

  public void setData (double[][] _data) {
    if (numPoints!=_data.length) setNumberOfPoints(_data.length);
    int maxPoints = Math.min(_data.length,numPoints);
    int n = Math.min(_data[0].length,3);
    for (int i=0; i<maxPoints; i++)  for (int k=0; k<n; k++) coordinates[k][i] = _data[i][k];
    hasChanged = true;
  }

  /**
   * Be warned! Data is stored as coordinates[3][numPoints] or coordinates[2][numPoints]
   * @return double[][]
   */
  public double[][] getData () { return coordinates; }

  public double[] getPoint (int _index) {
    double[] point = new double[coordinates.length];
    for (int i = 0; i < coordinates.length; i++) point[i] = coordinates[i][_index];
    return point;
  }

  public void setXs (double[] _data) {
    for (int i=0, n=Math.min(_data.length,numPoints); i<n; i++)  coordinates[0][i] = _data[i];
    hasChanged = true;
  }
  public void setXs (double _data) {
    for (int i=0; i<numPoints; i++)  coordinates[0][i] = _data;
    hasChanged = true;
  }

  public void setYs (double[] _data) {
    for (int i=0, n=Math.min(_data.length,numPoints); i<n; i++)  coordinates[1][i] = _data[i];
    hasChanged = true;
  }
  public void setYs (double _data) {
    for (int i=0; i<numPoints; i++)  coordinates[1][i] = _data;
    hasChanged = true;
  }

  public void setZs (double[] _data) {
    for (int i=0, n=Math.min(_data.length,numPoints); i<n; i++)  coordinates[2][i] = _data[i];
    hasChanged = true;
  }
  public void setZs (double _data) {
    for (int i=0; i<numPoints; i++)  coordinates[2][i] = _data;
    hasChanged = true;
  }

  public void setConnections (boolean[] _c) {
    if (_c==null) for (int i=0; i<numPoints; i++)  connect[i] = true;
    else for (int i=0; i<numPoints; i++)  connect[i] = _c[i];
    connect[sides] = closed;
    hasChanged = true;
  }

  public void setPointSizeEnableds (boolean[] _enabled) {
    for (int i=0, n=Math.min(_enabled.length,numPoints); i<n; i++)  pointSizeEnabled[i] = _enabled[i];
  }

  public void setPointSizeEnabled (int _index, boolean _enabled) {
    if (_index>-1 && _index<numPoints)  pointSizeEnabled[_index] = _enabled;
  }

// ----------------------------------------------
// Implementation of Interactive and Drawable3D
// ----------------------------------------------


  public org.opensourcephysics.display.Interactive findInteractive (DrawingPanel _panel, int _xpix, int _ypix) {
    if (!visible) return null;
    if (hasChanged || _panel!=panelWithValidProjection) projectPoints(_panel);
    if (sizeEnabled) {
      for (int i=0; i<numPoints; i++)
        if (pointSizeEnabled[i] && Math.abs(aPoints[i]-_xpix)<SENSIBILITY && Math.abs(bPoints[i]-_ypix)<SENSIBILITY) return new InteractionTargetPoligonPoint(this,i);
    }
    if (positionEnabled) {
      for (int i=0; i<numPoints; i++)
        if (Math.abs(aPoints[i]-_xpix)<SENSIBILITY && Math.abs(bPoints[i]-_ypix)<SENSIBILITY) return new InteractionTargetPoligonMovingPoint(this,i);
      // if (closed && Math.abs(pixelOrigin[0]-_xpix)<SENSIBILITY && Math.abs(pixelOrigin[1]-_ypix)<SENSIBILITY) return new InteractionTargetElementPosition(this);
    }
    return null;
   }

  public Object3D[] getObjects3D(DrawingPanel3D _panel) {
    if (!(numPoints>0 && visible)) return null;
    if (hasChanged || _panel!=panelWithValidProjection) projectPoints(_panel);
    if (closed && style.fillPattern!=null) return closedObject;
    else return lineObjects;
  }

  public void draw (DrawingPanel3D _panel, Graphics2D _g2, int _index) {
    if (_index<0) { // Interior ==> closed = true and fillPattern!=null
      java.awt.Paint theFillPattern = style.fillPattern;
      if (theFillPattern instanceof Color) theFillPattern = ((DrawingPanel3D)_panel).projectColor((Color) theFillPattern,closedObject[0].distance);
      _g2.setPaint(theFillPattern);
      _g2.fillPolygon(aPoints,bPoints,numPoints);
      if (style.edgeColor!=null) {
        java.awt.Color theColor = ((DrawingPanel3D)_panel).projectColor(style.edgeColor,closedObject[0].distance);
        _g2.setStroke(style.edgeStroke);
        _g2.setColor (theColor);
        for (int i=0; i<sides; i++) if (connect[i]) _g2.drawLine(aPoints[i],bPoints[i],aPoints[i+1],bPoints[i+1]);
        _g2.drawLine(aPoints[sides],bPoints[sides],aPoints[0],bPoints[0]);
      }
      return;
    }
    if (_index<sides) { // all regular segments
      if (connect[_index] && style.edgeColor!=null) {
        java.awt.Color theColor = ((DrawingPanel3D)_panel).projectColor(style.edgeColor,lineObjects[_index].distance);
        _g2.setStroke(style.edgeStroke);
        _g2.setColor (theColor);
        _g2.drawLine(aPoints[_index],bPoints[_index],aPoints[_index+1],bPoints[_index+1]);
      }
    }
    else { // if (_index==sides) { // Last closing segment
      if (closed && style.edgeColor!=null) {
        java.awt.Color theColor = ((DrawingPanel3D)_panel).projectColor(style.edgeColor,lineObjects[_index].distance);
        _g2.setStroke(style.edgeStroke);
        _g2.setColor (theColor);
        _g2.drawLine(aPoints[sides],bPoints[sides],aPoints[0],bPoints[0]);
      }
    }
  }

  public void draw (DrawingPanel _panel, Graphics _g) {
    if (!(numPoints>0 && visible)) return;
    Graphics2D g2 = (Graphics2D) _g;
    // if (hasChanged || _panel!=panelWithValidProjection)
    projectPoints(_panel);
    if (closed && style.fillPattern!=null) {
      g2.setPaint(style.fillPattern);
      g2.fillPolygon(aPoints,bPoints,numPoints);
    }
    if (style.edgeColor!=null) {
      g2.setColor (style.edgeColor);
      g2.setStroke(style.edgeStroke);
      for (int i=0; i<sides; i++) if (connect[i]) g2.drawLine(aPoints[i],bPoints[i],aPoints[i+1],bPoints[i+1]);
      if (closed) g2.drawLine(aPoints[sides],bPoints[sides],aPoints[0],bPoints[0]);
    }
  }

// ----------------------------------------------
// Implementation of Measured3D
// ----------------------------------------------

  public boolean isMeasured () { return canBeMeasured && numPoints>0;  }

  public double getXMin () {
    if (numPoints<=0) return 0.0;
    double min = Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[0][i]<min) min = coordinates[0][i];
    if (group==null) return x+min*sizex;
    else return group.x + (x + min*sizex)*group.sizex;
  }
  public double getXMax () {
    if (numPoints<=0) return 0.0;
    double max = -Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[0][i]>max) max = coordinates[0][i];
    if (group==null) return x+max*sizex;
    else return group.x + (x + max*sizex)*group.sizex;
  }
  public double getYMin () {
    if (numPoints<=0) return 0.0;
    double min = Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[1][i]<min) min = coordinates[1][i];
    if (group==null) return y+min*sizey;
    else return group.y + (y + min*sizey)*group.sizey;
  }
  public double getYMax () {
    if (numPoints<=0) return 0.0;
    double max = -Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[1][i]>max) max = coordinates[1][i];
    if (group==null) return y+max*sizey;
    else return group.y + (y + max*sizey)*group.sizey;
  }
  public double getZMin () {
    if (numPoints<=0) return 0.0;
    double min = Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[2][i]<min) min = coordinates[2][i];
    if (group==null) return z+min*sizez;
    else return group.z + (z + min*sizez)*group.sizez;
  }
  public double getZMax () {
    if (numPoints<=0) return 0.0;
    double max = -Double.MAX_VALUE;
    for (int i=0; i< numPoints; i++) if (coordinates[2][i]>max) max = coordinates[2][i];
    if (group==null) return z+max*sizez;
    else return group.z + (z + max*sizez)*group.sizez;
  }

// -------------------------------------
//  Private methods
// -------------------------------------

  protected void projectPoints (DrawingPanel _panel) {
//    System.out.println("Projecting poligon");
    // Project all points and compute the center of the poligon
    center[0] = 0.0;   center[1] = 0.0;   center[2] = 0.0;
    if (group==null) {
      origin[0] = x;     origin[1] = y;     origin[2] = z;
      size[0]   = sizex; size[1]   = sizey; size[2] = sizez;
    }
    else {
      origin[0] = group.x + x*group.sizex;   origin[1] = group.y + y*group.sizey;   origin[2] = group.z + z*group.sizez;
      size[0]   = sizex*group.sizex;         size[1]   = sizey*group.sizey;         size[2]   = sizez*group.sizez;
    }
    _panel.project(origin,pixelOrigin);
    for (int i=0; i<numPoints; i++) {
      for (int k=0; k<3; k++) {
        double delta = coordinates[k][i]*size[k];
        center[k] += delta;
        point[k] = origin[k] + delta;
      }
      _panel.project (point,pixel);
      aPoints[i] = (int) pixel[0];
      bPoints[i] = (int) pixel[1];
      if (connect[i]) lineObjects[i].distance = pixel[2];
      else lineObjects[i].distance = Double.NaN; // Will not be drawn
    }
    for (int k=0; k<3; k++) center[k] = origin[k] + center[k]/numPoints;
    // The interior
    if (closed && style.fillPattern!=null) {
      _panel.project(center,pixel);
      closedObject[0].distance = pixel[2];
    }
    else closedObject[0].distance = Double.NaN; // Will not be drawn
    hasChanged = false;
    panelWithValidProjection = _panel;
  }

/*
      protected void projectCenter (DrawingPanel _panel) {
    origin[0] = x;   origin[1] = y;   origin[2] = z;
    for (int k=0; k<3; k++) {
      center[k] = 0.0;
      for (int i = 0; i < numPoints; i++)
        center[k] += coordinates[k][i] * size[k];
      center[k] = origin[k] + center[k] / numPoints;
    }
    // The interior
    if (closed && style.fillPattern!=null) {
      _panel.project(center,pixelCenter);
      objects[numPoints].distance = pixelCenter[2];
    }
    else objects[numPoints].distance = Double.NaN; // Will not be drawn
  }
*/
}
