/*
 * The display3D package contains 3D drawing classes and drawables
 * @author F. Esquembre (http://fem.um.es).
 * Last version : July 2003
 */

package org.opensourcephysics.displayejs;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.AffineTransform;

import org.opensourcephysics.display.DrawingPanel;
import org.opensourcephysics.displayejs.utils.Utils;

/**
 * An InteractiveImage is an InteractiveElement that displays an Image
 */
public class InteractiveImage extends AbstractInteractiveElement {

  /* Implementation variables */
  private double a1=0.0, b1=0.0;
  private double[] coordinates  = new double[6]; // the input for all projections
  private double[] pixelOrigin  = new double[5]; // The projection of the origin (and size)
  private Object3D[] objects    = new Object3D[] { new Object3D(this,0) };
  private AffineTransform transform = new AffineTransform();

  /**
   * Default constructor
   */
  public InteractiveImage () { this(null,null); }

  /**
   * Constructor that specifies an image file and its codebase
   */
  public InteractiveImage (String _codebase, String _imageFile) {
    if (_imageFile!=null) {
      javax.swing.ImageIcon icon = Utils.icon(_codebase,_imageFile);
      if (icon!=null) style.displayObject = icon.getImage();
    }
  }

// ----------------------------------------------
// Implementation of Interactive and Drawable3D
// ----------------------------------------------

   public org.opensourcephysics.display.Interactive findInteractive (DrawingPanel _panel, int _xpix, int _ypix) {
     if (!visible) return null;
     if (hasChanged || _panel!=panelWithValidProjection) projectPoints (_panel);
     // In 3D it is not possible to change size interactively, due to the effect of style.position and style.angle
//    if (sizeEnabled      && Math.abs(a2-_xpix)<SENSIBILITY               && Math.abs(b2-_ypix)<SENSIBILITY)               return sizeTarget;
     if (positionEnabled && Math.abs(pixelOrigin[0]-_xpix)<SENSIBILITY && Math.abs(pixelOrigin[1]-_ypix)<SENSIBILITY) return new InteractionTargetElementPosition(this);
     return null;
    }

  public Object3D[] getObjects3D(DrawingPanel3D _panel) {
    if (!visible) return null;
    if (hasChanged || _panel!=panelWithValidProjection) projectPoints (_panel);
    return objects;
  }

  // No need to project, projection has already been computed in getObjects3D
  public void draw (DrawingPanel3D _panel, Graphics2D _g2, int _index) { drawIt (_g2); }

  public void draw (DrawingPanel _panel, Graphics _g) {
    if (!visible) return;
//    if (hasChanged || _panel!=panelWithValidProjection)
    projectPoints (_panel);
    drawIt ((Graphics2D) _g);
  }

// -------------------------------------
//  Private methods
// -------------------------------------

  protected void projectPoints (DrawingPanel _panel) {
//    System.out.println("Projecting image");
    if (group!=null) {
      coordinates[0] = group.x + x*group.sizex;  coordinates[1] = group.y + y*group.sizey; coordinates[2] = group.z + z*group.sizez;
      coordinates[3] = sizex*group.sizex;        coordinates[4] = sizey*group.sizey;       coordinates[5] = sizez*group.sizez;
    }
    else {
      coordinates[0] = x;     coordinates[1] = y;     coordinates[2] = z;
      coordinates[3] = sizex; coordinates[4] = sizey; coordinates[5] = sizez;
    }
    _panel.project(coordinates,pixelOrigin);
    objects[0].distance = pixelOrigin[4];
    a1 = pixelOrigin[0]; b1 = pixelOrigin[1];
    switch (style.position) {
      default :
      case Style.CENTERED:   a1-=pixelOrigin[2]/2.0; b1-=pixelOrigin[3]/2.0; break;
      case Style.NORTH:      a1-=pixelOrigin[2]/2.0;                                     break;
      case Style.SOUTH:      a1-=pixelOrigin[2]/2.0; b1-=pixelOrigin[3];     break;
      case Style.EAST:       a1-=pixelOrigin[2];     b1-=pixelOrigin[3]/2.0; break;
      case Style.SOUTH_EAST: a1-=pixelOrigin[2];     b1-=pixelOrigin[3];     break;
      case Style.NORTH_EAST: a1-=pixelOrigin[2];                                         break;
      case Style.WEST:                               b1-=pixelOrigin[3]/2.0;             break;
      case Style.SOUTH_WEST:                         b1-=pixelOrigin[3];                 break;
      case Style.NORTH_WEST:                                                                         break;
    }
    hasChanged = false;
    panelWithValidProjection = _panel;
  }

  private void drawIt (Graphics2D _g2) {
    if (! (style.displayObject instanceof Image)) return;
    AffineTransform originalTransform = _g2.getTransform();
    transform.setTransform(originalTransform);
    transform.rotate(-style.angle,pixelOrigin[0],pixelOrigin[1]);
    _g2.setTransform(transform);
    _g2.drawImage((Image) style.displayObject,(int)a1,(int)b1,(int)pixelOrigin[2],(int)pixelOrigin[3],null);
    _g2.setTransform(originalTransform);
  }
}
