/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;

public class VisualizationHints implements org.opensourcephysics.display3d.core.VisualizationHints {
  static final int HINT_DISPLAY_MODE    = 0;
  static final int HINT_DECORATION_TYPE = 1;
  static final int HINT_REMOVE_HIDDEN_LINES = 2;
  static final int HINT_PAN   = 3;
  static final int HINT_ALPHA = 4;
  static final int HINT_BETA  = 5;
  static final int HINT_ALPHA_AND_BETA = 6;
  static final int HINT_ZOOM = 7;
  static final int HINT_ALLOW_QUICK_REDRAW = 8;
  static final int HINT_USE_COLOR_DEPTH = 9;

  // Configuration variables
  private boolean removeHiddenLines=true, allowQuickRedraw=true, useColorDepth=true;
  private int displayMode = DISPLAY_PERSPECTIVE;
  private int decorationType=DECORATION_CUBE;
  private int panA=0, panB=0;
  private double alpha=0.0, beta=0.0, zoom=1.0;

  // Implementation variables
  /**
   * The DrawingPanel3D to which it belongs.
   * This is needed to report to it any change that implies a call to update()
   */
  private DrawingPanel3D panel;

  /**
   * Package-private constructor
   * VisualizationHints objects are obtained from DrawingPanel3Ds using the
   * getVisualizationHints() method.
   * @param _panel DrawingPanel3D
   * @see DrawingPanel3D
   */
  VisualizationHints (DrawingPanel3D _panel) { this.panel = _panel; }

  /**
   * Only for the use of the XMLLoader for DrawingPanel3D!
   * Sets the panel of these hints
   * @param aPanel DrawingPanel3D
   */
  void setPanel (DrawingPanel3D aPanel) { this.panel = aPanel; }

  public void setDisplayMode (int _mode) {
    this.displayMode = _mode;
    panel.hintChanged(HINT_DISPLAY_MODE);
  }
  final public int getDisplayMode () { return this.displayMode; }

  public void setDecorationType (int _value) {
    this.decorationType = _value;
    panel.hintChanged(HINT_DECORATION_TYPE);
  }
  final public int getDecorationType () { return this.decorationType; }

  public void setRemoveHiddenLines (boolean _value) {
    this.removeHiddenLines = _value;
    panel.hintChanged(HINT_REMOVE_HIDDEN_LINES);
  }
  final public boolean isRemoveHiddenLines () { return this.removeHiddenLines; }

  public void setAllowQuickRedraw (boolean _value) {
    this.allowQuickRedraw = _value;
    panel.hintChanged(HINT_ALLOW_QUICK_REDRAW);
  }
  final public boolean isAllowQuickRedraw () { return this.allowQuickRedraw; }

  public void setUseColorDepth (boolean _value) {
    this.useColorDepth = _value;
    panel.hintChanged(HINT_USE_COLOR_DEPTH);
  }
  final public boolean isUseColorDepth () { return this.useColorDepth; }

  public void setPan (int _panA, int _panB) {
    this.panA = _panA;
    this.panB = _panB;
    panel.hintChanged(HINT_PAN);
  }
  final public java.awt.Point getPan () { return new java.awt.Point(this.panA,this.panB); }
  final public int getPanA () { return this.panA; }
  final public int getPanB () { return this.panB; }

  public void setAlpha (double _alpha) {
    this.alpha = _alpha;
    panel.hintChanged(HINT_ALPHA);
  }
  final public double getAlpha () { return this.alpha; }

  public void setBeta (double _beta) {
    this.beta = _beta;
    panel.hintChanged(HINT_BETA);
  }
  final public double getBeta () { return this.beta; }

  public void setAlphaAndBeta (double _alpha, double _beta) {
    this.alpha = _alpha;
    this.beta  = _beta;
    panel.hintChanged(HINT_ALPHA_AND_BETA);
  }

  public void setZoom (double _zoom) {
    this.zoom = _zoom;
    panel.hintChanged(HINT_ZOOM);
  }
  final public double getZoom () { return this.zoom; }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  public static XML.ObjectLoader getLoader() {
    return new VisualizationHintsLoader();
  }

  protected static class VisualizationHintsLoader extends XMLLoader {
    public void saveObject(XMLControl control, Object obj) {
      VisualizationHints hints = (VisualizationHints) obj;
      control.setValue("display mode", hints.displayMode);
      control.setValue("decoration type", hints.decorationType);
      control.setValue("remove hidden lines", hints.removeHiddenLines);
      control.setValue("allow quick redraw", hints.allowQuickRedraw);
      control.setValue("horizontal rotation", hints.alpha);
      control.setValue("vertical rotation", hints.beta);
      control.setValue("zoom", hints.zoom);
      control.setValue("horizontal pan", hints.panA);
      control.setValue("vertical pan", hints.panB);
    }

    public Object createObject(XMLControl control) {
      return new VisualizationHints((DrawingPanel3D)null);
    }

    public Object loadObject(XMLControl control, Object obj) {
      VisualizationHints hints = (VisualizationHints) obj;
      hints.displayMode = control.getInt("display mode");
      hints.decorationType = control.getInt("decoration type");
      hints.removeHiddenLines = control.getBoolean("remove hidden lines");
      hints.allowQuickRedraw = control.getBoolean("allow quick redraw");
      hints.alpha = control.getDouble("horizontal rotation");
      hints.beta = control.getDouble("vertical rotation");
      hints.zoom = control.getDouble("zoom");
      hints.panA = control.getInt("horizontal pan");
      hints.panB = control.getInt("vertical pan");
      return obj;
    }
  }

}
