/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import org.opensourcephysics.controls.*;

public class Style implements org.opensourcephysics.display3d.core.Style {
  static final int STYLE_LINE_COLOR = 0;
  static final int STYLE_LINE_WIDTH = 1;
  static final int STYLE_FILL_COLOR = 2;
  static final int STYLE_RESOLUTION = 3;

  // Configuration variables
  private Color lineColor=Color.black;
  private float lineWidth=1.0f;
  private Color fillColor=Color.blue;
  private org.opensourcephysics.display3d.core.Resolution resolution=null;

  // Implementation variables
  /**
   * The owner element. This is needed to report to the element any change.
   */
  private Element element=null;
  private Stroke lineStroke=new BasicStroke(lineWidth);

  /**
   * Package-private constructor
   * User must obtain Style objects from elements, by using the getStyle() method
   * @param _element Element
   */
  Style (Element _element) { this.element = _element; }

  /**
   * Sets the element. For the use of ElementLoader only.
   * @param _element Element
   */
  void setElement(Element _element) { this.element = _element; }

  public void setLineColor (Color _color) {
    this.lineColor = _color;
    element.styleChanged(STYLE_LINE_COLOR);
  }
  final public Color getLineColor () { return this.lineColor; }

  public void setLineWidth (float _width) {
    if (this.lineWidth==_width) return;
    this.lineStroke = new BasicStroke(this.lineWidth=_width);
    element.styleChanged(STYLE_LINE_WIDTH);
  }
  final public float getLineWidth () { return this.lineWidth; }

  /**
   * Gets the Stroke derived from the line width
   * @return Stroke
   * @see java.awt.Stroke
   */
  final Stroke getLineStroke () { return this.lineStroke; }

  public void setFillColor (Color _color) {
    this.fillColor = _color;
    element.styleChanged(STYLE_FILL_COLOR);
  }
  final public Color getFillColor () { return this.fillColor; }

  public void setResolution (org.opensourcephysics.display3d.core.Resolution _res) {
    this.resolution = _res; // No need to clone. Resolution is unmutable
    element.styleChanged(STYLE_RESOLUTION);
  }
  // No danger. Resolution is unmutable
  final public org.opensourcephysics.display3d.core.Resolution getResolution () { return this.resolution; }


// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  public static XML.ObjectLoader getLoader() {
    return new StyleLoader();
  }

  protected static class StyleLoader extends XMLLoader {
    public void saveObject(XMLControl control, Object obj) {
      Style style = (Style) obj;
      control.setValue("line color", style.lineColor);
      control.setValue("line width", style.lineWidth);
      control.setValue("fill color", style.fillColor);
      control.setValue("resolution", style.resolution);
    }

    public Object createObject(XMLControl control) {
      return new Style((Element)null);
    }

    public Object loadObject(XMLControl control, Object obj) {
      Style style = (Style) obj;
      style.lineColor = (Color) control.getObject("line color");
      float width = (float) control.getDouble("line width");
      style.lineWidth = width;
      style.lineStroke = new BasicStroke(width);
      style.fillColor = (Color) control.getObject("fill color");
      style.resolution = (org.opensourcephysics.display3d.core.Resolution) control.getObject("resolution");
      return obj;
    }

  }

}
