/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;
import java.util.*;

/**
 * <p>Title: Group</p>
 * <p>Description: A Group is an element that is made of other elements.</p>
 * The group's position, size, visibility and transformation do affect the
 * elements in the group. The group's style doesn't, though.
 * @author Francisco Esquembre
 * @version March 2005
 * @see Style
 */
public class Group extends Element implements org.opensourcephysics.display3d.core.Group {
  // Implementation variables
  private ArrayList elementList = new ArrayList();
  private ArrayList list3D = new ArrayList();  // The list of Objects3D
  private Object3D[] minimalObjects = new Object3D[1];  // The array of Objects3D

// ----------------------------------------------------
// Implementation of core.Group
// ----------------------------------------------------

  public void addElement(org.opensourcephysics.display3d.core.Element element) {
    if (! (element instanceof Element) )
      throw new UnsupportedOperationException("Can't add element to group (incorrect implementation)");
    if (!elementList.contains(element)) elementList.add(element);
    ((Element) element).setGroup(this);
  }

  public void removeElement(org.opensourcephysics.display3d.core.Element element) { elementList.remove(element); }

  public void removeAllElements() { elementList.clear(); }

  public synchronized ArrayList getElements() { return (ArrayList) elementList.clone(); }

// ----------------------------------------------------
// Abstract part of Element
// ----------------------------------------------------

  Object3D[] getObjects3D() {
    if (!isVisible()) return null;
    list3D.clear();
    for (Iterator  it = elementList.iterator(); it.hasNext(); ) {
      Object3D[] objects = ((Element) it.next()).getObjects3D();
      if (objects!=null) for (int i=0, n=objects.length; i<n;  i++) list3D.add(objects[i]);
    }
    setElementChanged(false);
    if (list3D.size()==0) return null;
    return (Object3D[]) list3D.toArray(minimalObjects);
  }

  void draw (java.awt.Graphics2D _g2, int _index) {
    System.out.println ("Group draw (i): I should not be called!");
  }

  void drawQuickly (java.awt.Graphics2D _g2) {
    for (Iterator  it = elementList.iterator(); it.hasNext(); ) ((Element) it.next()).drawQuickly(_g2);
    setElementChanged(false);
  }

  // Overwrites its parent
  void setNeedToProject(boolean _need) {
    for (Iterator  it = elementList.iterator(); it.hasNext(); ) ((Element) it.next()).setNeedToProject(_need);
  }

  void styleChanged (int styleThatChanged) {
    // Do nothing. For the moment, style changes to a group don't affect its children
    // for (Iterator  it = elementList.iterator(); it.hasNext(); ) ((Element) it.next()).styleChanged() (styleThatChanged);
  }

  public void getExtrema (double[] min, double[] max) {
    double minX = Double.POSITIVE_INFINITY, maxX = Double.NEGATIVE_INFINITY;
    double minY = Double.POSITIVE_INFINITY, maxY = Double.NEGATIVE_INFINITY;
    double minZ = Double.POSITIVE_INFINITY, maxZ = Double.NEGATIVE_INFINITY;
    double [] firstPoint = new double[3], secondPoint = new double[3];
    Iterator it = getElements().iterator();
    while (it.hasNext()) {
      ((Element) it.next()).getExtrema (firstPoint,secondPoint);
      minX = Math.min (Math.min (minX,firstPoint[0]),secondPoint[0]);
      maxX = Math.max (Math.max (maxX,firstPoint[0]),secondPoint[0]);
      minY = Math.min (Math.min (minY,firstPoint[1]),secondPoint[1]);
      maxY = Math.max (Math.max (maxY,firstPoint[1]),secondPoint[1]);
      minZ = Math.min (Math.min (minZ,firstPoint[2]),secondPoint[2]);
      maxZ = Math.max (Math.max (maxZ,firstPoint[2]),secondPoint[2]);
    }
    min[0] = minX; max[0] = maxX;
    min[1] = minY; max[1] = maxY;
    min[2] = minZ; max[2] = maxZ;
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  public static XML.ObjectLoader getLoader() {
    return new GroupLoader();
  }

  protected static class GroupLoader extends ElementLoader {
    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      Group group = (Group) obj;
      control.setValue("elements", group.getElements());
    }

    public Object createObject(XMLControl control) {
      return new Group();
    }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      Group group = (Group) obj;
      Collection elements = (Collection) control.getObject("elements");
      if (elements!=null) {
        group.removeAllElements();
        Iterator it = elements.iterator();
        while (it.hasNext()) group.addElement((Element) it.next());
      }
      return obj;
    }
  }

}
