/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import java.util.*;
import org.opensourcephysics.controls.*;

/**
 * <p>Title: ElementSegment</p>
 * <p>Description: A Segment using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementTrail extends Element implements org.opensourcephysics.display3d.core.ElementTrail {
  // Configuration variables
  private boolean connected=true;
  private int maximum=0;

  // Implementation variables
  private int theFirstPoint = 0;
  protected ArrayList list=new ArrayList();
  private TrailPoint[] points = new TrailPoint[0];

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void addPoint(double x, double y, double z) { addPoint (x,y,z,this.connected); }

  public void moveToPoint(double x, double y, double z) { addPoint (x,y,z,false); }

  public void setMaximumPoints (int maximum) { this.maximum = Math.max(maximum,0); clear(); }

  public int getMaximumPoints () { return this.maximum; }

  public void setConnected(boolean connected) { this.connected = connected; }

  public boolean isConnected(boolean connected) { return this.connected; }

  public synchronized void clear() {
    list.clear();
    points = new TrailPoint[0];
    theFirstPoint = 0;
  }

  private void addPoint (double _x, double _y, double _z, boolean _c) {
    int position=list.size();
    if (maximum>0 && position>=maximum) {
      position = theFirstPoint;
      list.remove(theFirstPoint);
      theFirstPoint = (theFirstPoint+1) % maximum;
    }
    TrailPoint point = new TrailPoint(position, _x, _y, _z, _c);
    list.add(position,point);
    if (getPanel()!=null) point.transformAndProject();
  }

  public void getExtrema (double[] min, double[] max) {
    double minX = Double.POSITIVE_INFINITY, maxX = Double.NEGATIVE_INFINITY;
    double minY = Double.POSITIVE_INFINITY, maxY = Double.NEGATIVE_INFINITY;
    double minZ = Double.POSITIVE_INFINITY, maxZ = Double.NEGATIVE_INFINITY;
    double [] aPoint = new double[3];
    TrailPoint[] tmpPoints = new TrailPoint[0];
    tmpPoints = (TrailPoint[]) list.toArray(tmpPoints);
    for (int i=0,n=tmpPoints.length; i<n; i++) {
      aPoint[0] = tmpPoints[i].xp; aPoint[1] = tmpPoints[i].yp; aPoint[2] = tmpPoints[i].zp;
      toSpace(aPoint);
      minX = Math.min (minX,aPoint[0]);
      maxX = Math.max (maxX,aPoint[0]);
      minY = Math.min (minY,aPoint[1]);
      maxY = Math.max (maxY,aPoint[1]);
      minZ = Math.min (minZ,aPoint[2]);
      maxZ = Math.max (maxZ,aPoint[2]);
    }
    min[0] = minX; max[0] = maxX;
    min[1] = minY; max[1] = maxY;
    min[2] = minZ; max[2] = maxZ;
  }

// -------------------------------------
// Abstract part of Element or Parent methods overwritten
// -------------------------------------

  synchronized Object3D[] getObjects3D() {
    if (!isVisible() || list.size()<=0) return null;
    points = (TrailPoint[]) list.toArray(points);
    if (hasChanged()) transformAndProjectPoints();
    else if (needsToProject()) projectPoints();
    return points;
  }

  void draw (Graphics2D _g2, int _index) {
    TrailPoint point = points[_index];
    Color theColor = getPanel().projectColor(getRealStyle().getLineColor(), point.getDistance());
    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.setColor(theColor);
    if (_index==theFirstPoint || !point.connected)
      _g2.drawLine((int)point.pixel[0],(int)point.pixel[1],(int)point.pixel[0],(int)point.pixel[1]);
    else {
      if (_index==0) _index = maximum;
      TrailPoint pointPrev = points[_index-1];
      _g2.drawLine((int)point.pixel[0],(int)point.pixel[1],(int)pointPrev.pixel[0],(int)pointPrev.pixel[1]);
    }
  }

  void drawQuickly (Graphics2D _g2) {
    if (!isVisible() || list.size()<=0) return;
    points = (TrailPoint[]) list.toArray(points);
    if (hasChanged()) transformAndProjectPoints();
    else if (needsToProject()) projectPoints();

    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.setColor(getRealStyle().getLineColor());
    TrailPoint point = points[theFirstPoint];
    int aPrev = (int)point.pixel[0], bPrev = (int)point.pixel[1];
    _g2.drawLine(aPrev,bPrev,aPrev,bPrev);
    for (int i=1,n=points.length; i<n; i++) {   // The order is relevant
      int index;
      if (maximum>0) index = (i+theFirstPoint) % maximum;
      else index = i;
      point = points[index];
      if (point.connected) _g2.drawLine((int)point.pixel[0],(int)point.pixel[1],aPrev,bPrev);
      else                 _g2.drawLine((int)point.pixel[0],(int)point.pixel[1],(int)point.pixel[0],(int)point.pixel[1]);
      aPrev = (int)point.pixel[0]; bPrev = (int)point.pixel[1];
    }
  }

  void styleChanged (int styleThatChanged) {
    switch (styleThatChanged) {
//      case Style.STYLE_RESOLUTION : setElementChanged(true); break;
    }
  }

// -------------------------------------
// Private methods
// -------------------------------------

  synchronized void transformAndProjectPoints () {
    for (int i=0,n=points.length; i<n; i++)  points[i].transformAndProject();
    setNeedToProject(false);
    setElementChanged(false);
  }

  synchronized void projectPoints () {
    for (int i=0,n=points.length; i<n; i++)  points[i].transformAndProject();
    setNeedToProject(false);
  }

// ----------------------------------------------------
// A class for the individual points of the trail
// ----------------------------------------------------

  private class TrailPoint extends Object3D {
    boolean connected;
    private double xp,yp,zp;
    private double[] coordinates = new double[3];
    double[] pixel = new double[3];

    TrailPoint (int _index, double _x, double _y, double _z, boolean _c) {
      super (ElementTrail.this,_index);
      xp = _x; yp = _y;  zp = _z;
      connected = _c;
      }

    void transformAndProject () {
      coordinates[0] = xp; coordinates[1] = yp; coordinates[2] = zp;
      toSpace(coordinates);
      getPanel().project(coordinates,pixel);
      super.setDistance(pixel[2]);
    }

    void project () {
      getPanel().project(coordinates,pixel);
      super.setDistance(pixel[2]);
    }

  }  // End of class TrailPoint

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementTrailLoader(); }

  static private class ElementTrailLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementTrail(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementTrail element = (ElementTrail) obj;
      control.setValue("maximum", element.maximum);
      control.setValue("connected", element.connected);
      // Don't save the points since loadObject will clear the trail
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementTrail element = (ElementTrail) obj;
      element.setConnected(control.getBoolean("connected"));
      element.setMaximumPoints(control.getInt("maximum"));
      // This implies element.clear()
      return obj;
    }


  }

}
