/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import java.awt.geom.AffineTransform;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.display.TextLine;

/**
 * <p>Title: ElementText</p>
 * <p>Description: A Text using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementText extends Element implements org.opensourcephysics.display3d.core.ElementText {
  // Configuration variables
  private int justify=JUSTIFICATION_CENTER;
  private double angle = 0.0;

  // Implementation variables
  private double[] coordinates = new double[3]; // The input for all projections
  private double[] pixel  = new double[3]; // The ouput of all projections
  private Object3D[] objects = new Object3D[] { new Object3D(this,0) };
  private AffineTransform transform = new AffineTransform();
  private TextLine textLine = new TextLine();

// -------------------------------------
// New configuration methods
// -------------------------------------

  public ElementText () {
    super();
  }

  public ElementText (String text) {
    this();
    setText(text);
  }

  public void setText (String text) { textLine.setText(text); }
  public String getText () { return textLine.getText(); }

  public void setFont (Font font) { textLine.setFont(font); }
  public Font getFont () { return textLine.getFont(); }

  public void setJustification (int justification) {
    this.justify = justification;
    switch (justification) {
      default :
      case JUSTIFICATION_CENTER: textLine.setJustification(TextLine.CENTER); break;
      case JUSTIFICATION_LEFT: textLine.setJustification(TextLine.LEFT); break;
      case JUSTIFICATION_RIGHT: textLine.setJustification(TextLine.RIGHT); break;
    }
  }
  public int getJustification () { return this.justify; }

  public void setRotationAngle (double angle) { this.angle = angle; }
  public double getRotationAngle () { return this.angle; }

// -------------------------------------
// Abstract part of Element or Parent methods overwritten
// -------------------------------------

  Object3D[] getObjects3D() {
    if (!isVisible()) return null;
    if (hasChanged() || needsToProject()) projectPoints ();
    return objects;
  }

  void draw (Graphics2D _g2, int _index) {
    // Allow the panel to adjust color according to depth
    Color theColor = getPanel().projectColor(getRealStyle().getLineColor(),objects[0].getDistance());
    drawIt (_g2,theColor);
  }

  void drawQuickly (Graphics2D _g2) {
    if (!isVisible()) return;
    if (hasChanged() || needsToProject()) projectPoints ();
    drawIt (_g2, getRealStyle().getLineColor());
  }

  void styleChanged (int styleThatChanged) { } // Not affected

  void getExtrema (double[] min, double[] max) {
    min[0] = 0; max[0] = 0;
    min[1] = 0; max[1] = 0;
    min[2] = 0; max[2] = 0;
    toSpace(min);
    toSpace(max);
  }

// -------------------------------------
// Private methods
// -------------------------------------

  private void projectPoints () {
    coordinates[0] = coordinates[1] = coordinates[2] = 0.0;
    toSpace (coordinates);
    getPanel().project(coordinates,pixel);
    objects[0].setDistance(pixel[2]);
    setElementChanged(false);
    setNeedToProject(false);
  }

  private void drawIt (Graphics2D _g2, Color _color) {
    textLine.setColor(_color);
    if (angle!=0.0) {
      AffineTransform originalTransform = _g2.getTransform();
      transform.setTransform(originalTransform);
      transform.rotate(angle, pixel[0], pixel[1]);
      _g2.setTransform(transform);
      textLine.drawText(_g2, (int) pixel[0], (int) pixel[1]);
      _g2.setTransform(originalTransform);
    }
    else textLine.drawText(_g2, (int) pixel[0], (int) pixel[1]);
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementTextLoader(); }

  static private class ElementTextLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementText(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementText element = (ElementText) obj;
      control.setValue("text", element.getText());
      control.setValue("font", element.getFont());
      control.setValue("justification", element.getJustification());
      control.setValue("rotation angle", element.getRotationAngle());
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementText element = (ElementText) obj;
      element.setText(control.getString("text"));
      element.setFont((Font)control.getObject("font"));
      element.setJustification(control.getInt("justification"));
      element.setRotationAngle(control.getDouble("rotation angle"));
      return obj;
    }

  }

}
