/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;

/**
 * <p>Title: ElementBox</p>
 * <p>Description: Painter's algorithm implementation of a Surface</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementSurface extends AbstractTile implements org.opensourcephysics.display3d.core.ElementSurface {
  // Configuration variables
  protected double[][][] data;

  // Implementation variables
  protected int nu = -1, nv = -1; // Make sure arrays are allocated

  { // Initialization block
    setXYZ(0.0,0.0,0.0);
    setSizeXYZ(1.0,1.0,1.0);
  }

// -------------------------------------
// Configuration
// -------------------------------------

  public void setData (double[][][] data) {
    this.data = data;
    setElementChanged(true);
  }
  public double[][][] getData () { return this.data; }

  void styleChanged (int styleThatChanged) { } // Not affected

// -------------------------------------
//  Private or protected methods
// -------------------------------------

  public void getExtrema (double[] min, double[] max) {
    double minX = Double.POSITIVE_INFINITY, maxX = Double.NEGATIVE_INFINITY;
    double minY = Double.POSITIVE_INFINITY, maxY = Double.NEGATIVE_INFINITY;
    double minZ = Double.POSITIVE_INFINITY, maxZ = Double.NEGATIVE_INFINITY;
    double [] aPoint = new double[3];
    int theNu = data.length-1, theNv = data[0].length-1;
    for (int i=0,n1=data.length; i<n1; i++)  {
      for (int j=0,n2=data[0].length; j<n2; j++)  {
        System.arraycopy(data[i][j], 0, aPoint, 0, 3);
        toSpace(aPoint);
        minX = Math.min(minX, aPoint[0]);
        maxX = Math.max(maxX, aPoint[0]);
        minY = Math.min(minY, aPoint[1]);
        maxY = Math.max(maxY, aPoint[1]);
        minZ = Math.min(minZ, aPoint[2]);
        maxZ = Math.max(maxZ, aPoint[2]);
      }
    }
    min[0] = minX; max[0] = maxX;
    min[1] = minY; max[1] = maxY;
    min[2] = minZ; max[2] = maxZ;
  }

  protected synchronized void computeCorners () {
    if (data==null) return;
    int theNu = data.length-1, theNv = data[0].length-1;
    if (nu==theNu && nv==theNv); // No need to reallocate arrays
    else {
      nu = theNu; nv = theNv;
      setCorners(new double [nu*nv][4][3]); // Reallocate arrays
    }
    int tile = 0;
    for (int v=0; v<nv; v++) {
      for (int u=0; u<nu; u++, tile++) {
        for (int k=0; k<3; k++) {
          corners[tile][0][k] = data[u  ][v  ][k];
          corners[tile][1][k] = data[u+1][v  ][k];
          corners[tile][2][k] = data[u+1][v+1][k];
          corners[tile][3][k] = data[u  ][v+1][k];
        }
      }
    }
    for (int i=0; i<numberOfTiles; i++)
      for (int j=0, sides=corners[i].length; j<sides; j++) {
        toSpace(corners[i][j]);
      }
    setElementChanged(false);
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementSurfaceLoader(); }

  static private class ElementSurfaceLoader extends ElementLoader {
    public Object createObject(XMLControl control) { return new ElementSurface(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementSurface element = (ElementSurface) obj;
      control.setValue("data", element.data);
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementSurface element = (ElementSurface) obj;
      element.setData((double[][][]) control.getObject("data"));
      return obj;
    }

  }

}
