/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.display3d.simple3d.utils.*;

/**
 * <p>Title: ElementSegment</p>
 * <p>Description: A Segment using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementSpring extends Element implements org.opensourcephysics.display3d.core.ElementSpring {
  // Configuration variables
  private double radius=0.1;

  // Implementation variables
  private int loops=-1, pointsPerLoop=-1;  // Make sure arrays are allocated
  private int segments = 0;
  private int aPoints[]=null, bPoints[]=null;
  private double points[][] = null;
  private double pixel[]  = new double[3]; // The output for all projections
  private Object3D[] objects=null; // The Objects3D of this Element

  { // Initialization block
    getStyle().setResolution(new Resolution (8,15,1)); // the 1 is meaningless
  }

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void setRadius (double radius) {
    this.radius = radius;
    setElementChanged(true);
  }
  public double getRadius () { return this.radius; }

// -------------------------------------
// Abstract part of Element or Parent methods overwritten
// -------------------------------------

  Object3D[] getObjects3D() {
    if (!isVisible()) return null;
    if (hasChanged()) { computePoints(); projectPoints(); }
    else if (needsToProject()) projectPoints();
    return objects;
  }

  void draw (Graphics2D _g2, int _index) {
    // Allow the panel to adjust color according to depth
    Color theColor = getPanel().projectColor(getRealStyle().getLineColor(),objects[_index].getDistance());
    _g2.setColor (theColor);
    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.drawLine(aPoints[_index],bPoints[_index],aPoints[_index+1],bPoints[_index+1]);
  }

  void drawQuickly (Graphics2D _g2) {
    if (!isVisible()) return;
    if (hasChanged()) { computePoints(); projectPoints(); }
    else if (needsToProject()) projectPoints();
    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.setColor (getRealStyle().getLineColor());
    _g2.drawPolyline(aPoints,bPoints,segments+1);
  }

  void styleChanged (int styleThatChanged) {
    switch (styleThatChanged) {
      case Style.STYLE_RESOLUTION : setElementChanged(true); break;
    }
  }

  void getExtrema (double[] min, double[] max) {
    min[0] = 0; max[0] = 1;
    min[1] = 0; max[1] = 1;
    min[2] = 0; max[2] = 1;
    toSpace(min);
    toSpace(max);
  }

// -------------------------------------
// Private methods
// -------------------------------------

  void projectPoints () {
    for (int i = 0; i < segments; i++) {
      getPanel().project(points[i], pixel);
      aPoints[i] = (int) pixel[0];
      bPoints[i] = (int) pixel[1];
      objects[i].setDistance(pixel[2]); //distance is given by the first point
    }
    getPanel().project(points[segments], pixel);
    aPoints[segments] = (int) pixel[0];
    bPoints[segments] = (int) pixel[1];
    setNeedToProject(false);
  }

  private void computePoints () {
    int theLoops = loops, thePPL = pointsPerLoop;
    org.opensourcephysics.display3d.core.Resolution res = getRealStyle().getResolution();
    if (res!=null) {
      switch (res.getType()) {
        case Resolution.DIVISIONS :
          theLoops = Math.max(res.getN1(),0);
          thePPL   = Math.max(res.getN2(),1);
          break;
      }
    }
    if (theLoops==loops && thePPL==pointsPerLoop); // No need to reallocate arrays
    else { // Reallocate arrays
      loops = theLoops;
      pointsPerLoop = thePPL;
      segments = loops*pointsPerLoop + 3;
      points  = new double [segments+1][3];
      aPoints = new int [segments+1];
      bPoints = new int [segments+1];
      objects = new Object3D[segments];
      for (int i=0; i<segments; i++) objects[i] = new Object3D(this,i);
    }
    Point3D size = new Point3D (getSizeX(),getSizeY(),getSizeZ());
    Point3D u1 = VectorAlgebra.normalTo (size);
    Point3D u2 = VectorAlgebra.normalize (VectorAlgebra.crossProduct(size,u1));
    double delta=2.0*Math.PI/pointsPerLoop;
    int pre = pointsPerLoop/2;
    for (int i=0; i<=segments; i++) {
      int k;
      if (i<pre) k = 0;
      else if (i<pointsPerLoop) k = i-pre;
      else if (i>(segments-pre)) k = 0;
      else if (i>(segments-pointsPerLoop)) k = segments-i-pre;
      else k = pre;
      double angle = i*delta;
      double cos = Math.cos(angle), sin = Math.sin(angle);
      points[i][0] = i*getSizeX()/segments + k*radius*(cos*u1.getX() + sin*u2.getX())/pre;
      points[i][1] = i*getSizeY()/segments + k*radius*(cos*u1.getY() + sin*u2.getY())/pre;
      points[i][2] = i*getSizeZ()/segments + k*radius*(cos*u1.getZ() + sin*u2.getZ())/pre;
      toSpaceSized (points[i]); // apply the transformation(s)
    }
    setElementChanged(false);
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementSpringLoader(); }

  static private class ElementSpringLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementSpring(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementSpring element = (ElementSpring) obj;
      control.setValue("radius", element.radius);
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementSpring element = (ElementSpring) obj;
      element.radius = control.getDouble("radius");
      return obj;
    }

  }

}
