/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.display3d.simple3d.utils.*;

/**
 * <p>Title: ElementPolygon</p>
 * <p>Description: A Polygon using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementPolygon extends Element implements org.opensourcephysics.display3d.core.ElementPolygon {
  // Configuration variables
  private boolean closed=true;
  private double coordinates[][] = new double[0][0];

  // Implementation variables
  private int aPoints[]=null, bPoints[]=null;
  private double [][] transformedCoordinates=new double[0][0];
  private double center[] = new double[3]; // The center of the poligon
  private double pixel[]  = new double[3]; // Output of panel's projections

  protected Object3D[] lineObjects=null;  // Objects3D for each of the lines
  protected Object3D[] closedObject= new Object3D[] { new Object3D(this,-1) }; // A special object for a closed poligon

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void setClosed (boolean closed) { this.closed = closed; }
  public boolean isClosed () { return this.closed; }

  /**
   * Sets the data for the points of the polygon.
   * Each entry in the data array corresponds to one vertex.
   * If the polygon is closed, the last point will be connected
   * to the first one and the interior will be filled
   * (unless the fill color of the style is set to null).
   * The data array is copied, so subsequence changes to the original
   * array do not affect the polygon, until this setData() methos is invoked.
   * @param data double[][] the double[nPoints][3] array with the data
   */
  public void setData (double[][] data) {
    if (coordinates.length!=data.length) {
      int n = data.length;
      coordinates  = new double [n][3];
      transformedCoordinates = new double [n][3];
      aPoints = new int [n];
      bPoints = new int [n];
      lineObjects = new Object3D[n];
      for (int i=0; i<n; i++) lineObjects[i] = new Object3D(this,i);
    }
    for (int i=0, n=data.length; i<n; i++)  System.arraycopy(data[i],0,coordinates[i],0,3);
    setElementChanged(true);

  }
  /**
   * Gets (a copy of) the data of the points for the polygon
   * @return double[][] the double[nPoints][3] array with the data
   */
    public double[][] getData () {
      double[][] data = new double[coordinates.length][3];
      for (int i=0,n=coordinates.length; i<n; i++)  System.arraycopy(coordinates[i],0,data[i],0,3);
      return data;
    }

// -------------------------------------
// Abstract part of Element or Parent methods overwritten
// -------------------------------------

    public void getExtrema (double[] min, double[] max) {
      double minX = Double.POSITIVE_INFINITY, maxX = Double.NEGATIVE_INFINITY;
      double minY = Double.POSITIVE_INFINITY, maxY = Double.NEGATIVE_INFINITY;
      double minZ = Double.POSITIVE_INFINITY, maxZ = Double.NEGATIVE_INFINITY;
      double [] aPoint = new double[3];
      for (int i=0,n=coordinates.length; i<n; i++)  {
        System.arraycopy(coordinates[i],0,aPoint,0,3);
        toSpace(aPoint);
        minX = Math.min (minX,aPoint[0]);
        maxX = Math.max (maxX,aPoint[0]);
        minY = Math.min (minY,aPoint[1]);
        maxY = Math.max (maxY,aPoint[1]);
        minZ = Math.min (minZ,aPoint[2]);
        maxZ = Math.max (maxZ,aPoint[2]);
      }
      min[0] = minX; max[0] = maxX;
      min[1] = minY; max[1] = maxY;
      min[2] = minZ; max[2] = maxZ;
    }

  Object3D[] getObjects3D() {
    if (!isVisible() || coordinates.length==0) return null;
    if (hasChanged()) transformAndProject();
    else if (needsToProject()) project();
    if (closed && getRealStyle().getFillColor()!=null) return closedObject;
    else return lineObjects;
  }

  void draw (Graphics2D _g2, int _index) {
    if (_index<0) { // Interior ==> closed = true and fillPattern!=null
      Color theFillColor = getPanel().projectColor(getRealStyle().getFillColor(),closedObject[0].getDistance());
      _g2.setPaint(theFillColor);
      _g2.fillPolygon(aPoints,bPoints,aPoints.length);
      if (getRealStyle().getLineColor()!=null) {
        Color theColor = getPanel().projectColor(getRealStyle().getLineColor(),closedObject[0].getDistance());
        _g2.setStroke(getRealStyle().getLineStroke());
        _g2.setColor (theColor);
        int n = aPoints.length-1;
        for (int i=0; i<n; i++) _g2.drawLine(aPoints[i],bPoints[i],aPoints[i+1],bPoints[i+1]);
        _g2.drawLine(aPoints[n],bPoints[n],aPoints[0],bPoints[0]);
      }
      return;
    }
    if (getRealStyle().getLineColor()==null) return;
    Color theColor = getPanel().projectColor(getRealStyle().getLineColor(),lineObjects[_index].getDistance());
    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.setColor (theColor);
    int sides = aPoints.length-1;
    if (_index<sides) _g2.drawLine(aPoints[_index],bPoints[_index],aPoints[_index+1],bPoints[_index+1]); // regular segment
    else _g2.drawLine(aPoints[sides],bPoints[sides],aPoints[0],bPoints[0]);// if (_index==sides) { // Last closing segment
  }

  void drawQuickly (Graphics2D _g2) {
    if (!isVisible()) return;
    if (hasChanged()) transformAndProject();
    else if (needsToProject()) project();
    _g2.setStroke(getRealStyle().getLineStroke());
    _g2.setColor (getRealStyle().getLineColor());
    _g2.drawPolyline(aPoints,bPoints,aPoints.length);
    int sides = aPoints.length-1;
    if (closed) _g2.drawLine(aPoints[sides],bPoints[sides],aPoints[0],bPoints[0]);
  }

  void styleChanged (int styleThatChanged) { }  // Not affected

// -------------------------------------
// Private methods
// -------------------------------------

  void transformAndProject () {
    center[0] = center[1] = center[2] = 0.0;
    for (int i=0,n=coordinates.length; i<n; i++) {
      for (int k=0; k<3; k++) {
        center[k] += coordinates[i][k];
        transformedCoordinates[i][k] = coordinates[i][k];
      }
      toSpace (transformedCoordinates[i]);
      getPanel().project (transformedCoordinates[i],pixel);
      aPoints[i] = (int) pixel[0];
      bPoints[i] = (int) pixel[1];
      lineObjects[i].setDistance(pixel[2]);
    }
    // last Segment
    if (!closed) lineObjects[coordinates.length-1].setDistance(Double.NaN);
    // The interior
    for (int k=0; k<3; k++) center[k] /= coordinates.length;
    if (closed && getRealStyle().getFillColor()!=null) {
      getPanel().project(center,pixel);
      closedObject[0].setDistance(pixel[2]);
    }
    else closedObject[0].setDistance(Double.NaN); // Will not be drawn
    setElementChanged(false);
    setNeedToProject(false);
  }

  void project () {
    for (int i=0,n=coordinates.length; i<n; i++) {
      getPanel().project (transformedCoordinates[i],pixel);
      aPoints[i] = (int) pixel[0];
      bPoints[i] = (int) pixel[1];
      lineObjects[i].setDistance(pixel[2]);
    }
    if (!closed) lineObjects[coordinates.length-1].setDistance(Double.NaN);
    // The interior
    if (closed && getRealStyle().getFillColor()!=null) {
      getPanel().project(center,pixel);
      closedObject[0].setDistance(pixel[2]);
    }
    else closedObject[0].setDistance(Double.NaN); // Will not be drawn
    setNeedToProject(false);
  }


// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementPolygonLoader(); }

  static private class ElementPolygonLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementPolygon(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementPolygon element = (ElementPolygon) obj;
      control.setValue("closed", element.closed);
      control.setValue("data", element.coordinates);
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementPolygon element = (ElementPolygon) obj;
      element.closed = control.getBoolean("closed");
      element.setData((double[][]) control.getObject("data"));
      return obj;
    }

  }

}
