/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import org.opensourcephysics.controls.*;

/**
 * <p>Title: ElementBox</p>
 * <p>Description: Painter's algorithm implementation of a Plane</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementPlane extends AbstractTile implements org.opensourcephysics.display3d.core.ElementPlane {
  // Configuration variables
  private double vectorU[] = { 1.0, 0.0, 0.0};
  private double vectorV[] = { 0.0, 1.0, 0.0};

  // Implementation variables
  private int nu = -1, nv = -1; // Make sure arrays are allocated
  private double vectorUSize = 1.0, vectorVSize = 1.0;

  { // Initialization block
    setXYZ(0.0,0.0,0.0);
    setSizeXYZ(1.0,1.0,1.0);
  }

// -------------------------------------
// Configuration
// -------------------------------------

  public void setFirstDirection (double[] vector) {
    vectorU[0] = vector[0];
    vectorU[1] = vector[1];
    vectorU[2] = vector[2];
    vectorUSize = Math.sqrt(vectorU[0]*vectorU[0]+vectorU[1]*vectorU[1]+vectorU[2]*vectorU[2]);
    setElementChanged (true);
  }
  public double[] getFirstDirection () {
    return new double[] {vectorU[0], vectorU[1], vectorU[2]};
  }

  public void setSecondDirection (double[] vector) {
    vectorV[0] = vector[0];
    vectorV[1] = vector[1];
    vectorV[2] = vector[2];
    vectorVSize = Math.sqrt(vectorV[0]*vectorV[0]+vectorV[1]*vectorV[1]+vectorV[2]*vectorV[2]);
    setElementChanged (true);
  }
  public double[] getSecondDirection () {
    return new double[] {vectorU[0], vectorU[1], vectorU[2]};
  }

// -------------------------------------
//  Private or protected methods
// -------------------------------------

  protected synchronized void computeCorners () {
    int theNu = 1, theNv = 1;
    org.opensourcephysics.display3d.core.Resolution res = getRealStyle().getResolution();
    if (res!=null) {
      switch (res.getType()) {
        case Resolution.DIVISIONS :
          theNu = Math.max(res.getN1(),1);
          theNv = Math.max(res.getN2(),1);
          break;
        case Resolution.MAX_LENGTH :
          theNu = Math.max((int) Math.round(0.49 + Math.abs(getSizeX())*vectorUSize/res.getMaxLength()), 1);
          theNv = Math.max((int) Math.round(0.49 + Math.abs(getSizeY())*vectorVSize/res.getMaxLength()), 1);
        break;
      }
    }
    if (nu!=theNu || nv!=theNv) { // Reallocate arrays
      nu = theNu; nv = theNv;
      setCorners(new double [nu*nv][4][3]);
    }
    int tile = 0;
    double du = getSizeX()/nu,  dv = getSizeY()/nv;
    for (int i=0; i<nu; i++) { // x-y sides
      double u = i*du-getSizeX()/2;
      for (int j=0; j<nv; j++) {
        double v = j*dv-getSizeY()/2;
        for (int k=0; k<3; k++) corners[tile][0][k] = u     *vectorU[k] + v     *vectorV[k];
        for (int k=0; k<3; k++) corners[tile][1][k] = (u+du)*vectorU[k] + v     *vectorV[k];
        for (int k=0; k<3; k++) corners[tile][2][k] = (u+du)*vectorU[k] + (v+dv)*vectorV[k];
        for (int k=0; k<3; k++) corners[tile][3][k] = u     *vectorU[k] + (v+dv)*vectorV[k];
        tile++; // The upper side
      }
    }
    for (int i=0; i<numberOfTiles; i++)
      for (int j=0, sides=corners[i].length; j<sides; j++) {
        toSpaceSized(corners[i][j]);
      }
    setElementChanged(false);
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementPlaneLoader(); }

  static private class ElementPlaneLoader extends ElementLoader {
    public Object createObject(XMLControl control) { return new ElementPlane(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementPlane element = (ElementPlane) obj;
      control.setValue("first direction",  element.vectorU);
      control.setValue("second direction", element.vectorV);
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementPlane element = (ElementPlane) obj;
      element.setFirstDirection ((double[]) control.getObject("first direction"));
      element.setSecondDirection((double[]) control.getObject("second direction"));
      return obj;
    }

  }

}
