/*
 * The org.opensourcephysics.display3d.simple3d package implements the
 * classes of the org.opensourcephysics.display3d package using
 * the so called painter's algorithm, in which the diferent parts of the
 * 3D scene are painted from back to front, thus achieving a simple, but
 * for many cases effective, hidden-line removal.
 *
 * Copyright (c) 2005  The Open Source Physics project
 *                     http://www.opensourcephysics.org
 */

package org.opensourcephysics.display3d.simple3d;

import java.awt.*;
import java.awt.geom.AffineTransform;
import org.opensourcephysics.controls.*;
import org.opensourcephysics.tools.*;

/**
 * <p>Title: ElementText</p>
 * <p>Description: A Text using the painter's algorithm</p>
 * @author Francisco Esquembre
 * @version March 2005
 */
public class ElementImage extends Element implements org.opensourcephysics.display3d.core.ElementImage {
  // Configuration variables
  private String imageFile = null;
  private double angle = 0.0;

  // Implementation variables
  private Resource resource=null;
  private Image image = null;
  private double[] coordinates = new double[3];
  private double[] fullCoordinates = new double[6]; // The input for all projections
  private double[] pixel  = new double[5]; // The ouput of all projections
  private Object3D[] objects = new Object3D[] { new Object3D(this,0) };
  private AffineTransform transform = new AffineTransform();

// -------------------------------------
// New configuration methods
// -------------------------------------

  public void setImageFile (String file) {
    this.imageFile = file;
    if (file!=null) {
      resource = ResourceLoader.getResource(file);
      if (resource!=null) image = resource.getImage();
    }
  }
  public String getImageFile () { return this.imageFile; }

  public void setRotationAngle (double angle) { this.angle = angle; }
  public double getRotationAngle () { return this.angle; }

// -------------------------------------
// Abstract part of Element or Parent methods overwritten
// -------------------------------------

  Object3D[] getObjects3D() {
    if (!isVisible()) return null;
    if (hasChanged() || needsToProject()) projectPoints ();
    return objects;
  }

  void draw (Graphics2D _g2, int _index) {
    drawIt (_g2);
  }

  void drawQuickly (Graphics2D _g2) {
    if (!isVisible()) return;
    if (hasChanged() || needsToProject()) projectPoints ();
    drawIt (_g2);
  }

  void styleChanged (int styleThatChanged) { } // Not affected

// -------------------------------------
// Private methods
// -------------------------------------

  private void projectPoints () {
    coordinates[0] = coordinates[1] = coordinates[2] = 0.0;
    fullCoordinates[3] = getSizeX(); fullCoordinates[4] = getSizeY(); fullCoordinates[5] = getSizeZ();
    toSpace (coordinates);
    fullCoordinates[0] = coordinates[0]; fullCoordinates[1] = coordinates[1]; fullCoordinates[2] = coordinates[2];
    getPanel().project(fullCoordinates,pixel);
    objects[0].setDistance(pixel[4]);
    setElementChanged(false);
    setNeedToProject(false);
  }

  private void drawIt (Graphics2D _g2) {
    if (image==null) return;
    int diameter = (int) Math.max(pixel[2],pixel[3]), radius = diameter/2;
    int xc = (int)pixel[0]-radius, yc = (int)pixel[1]-radius;
    if (angle!=0.0) {
      AffineTransform originalTransform = _g2.getTransform();
      transform.setTransform(originalTransform);
      transform.rotate(-angle, pixel[0], pixel[1]);
      _g2.setTransform(transform);
      _g2.drawImage(image,(int)pixel[0]-radius,(int)pixel[1]-radius,diameter,diameter,null);
      _g2.setTransform(originalTransform);
    }
    else _g2.drawImage(image,(int)pixel[0]-radius,(int)pixel[1]-radius,diameter,diameter,null);
  }

// ----------------------------------------------------
// XML loader
// ----------------------------------------------------

  /**
   * Returns an XML.ObjectLoader to save and load object data.
   * @return the XML.ObjectLoader
   */
  public static XML.ObjectLoader getLoader() { return new ElementImageLoader(); }

  static private class ElementImageLoader extends ElementLoader {

    public Object createObject(XMLControl control) { return new ElementImage(); }

    public void saveObject(XMLControl control, Object obj) {
      super.saveObject(control,obj);
      ElementImage element = (ElementImage) obj;
      control.setValue("image file", element.getImageFile());
      control.setValue("rotation angle", element.getRotationAngle());
   }

    public Object loadObject(XMLControl control, Object obj) {
      super.loadObject(control,obj);
      ElementImage element = (ElementImage) obj;
      element.setImageFile(control.getString("image file"));
      element.setRotationAngle(control.getDouble("rotation angle"));
      return obj;
    }

  }

}
